import React, { useState, useEffect } from "react";
import { Form, OverlayTrigger, Tooltip, Button } from "react-bootstrap";
import { toast } from "react-toastify";
import CustomDataTable from "../../components/CustomDataTable";
import DeleteModel from "../../components/DeleteModel";
import ParishCategoryApi from "../../api/ParishCategoryApi";
import AdminApi from "../../api/admin";
import Loader from "../../components/loader";
import { Cookies } from "react-cookie";
import Helpers from "../../components/Helpers";
import { useNavigate } from "react-router-dom";

const ParishCategories = () => {
  let navigate = useNavigate();
  const [data, setData] = useState({ name: "" });
  const [oldValues, setOldValues] = useState({});
  const [list, setList] = useState([]);
  const [icon_id, setIcon_id] = useState("");
  const [validated, setValidated] = useState(false);
  const [show, setShow] = useState(false);
  const [LoadData, setLoadData] = useState(false);
  const [cp, setCp] = useState(false);
  const [isError, setIsError] = useState(false);

  const cookies = new Cookies();
  const userData = cookies.get("LoveStrongUser") || null;

  const parishApi = new ParishCategoryApi();
  const adminApi = new AdminApi();

  const handleChanges = (e) => {
    setData({ ...data, [e.target.name]: e.target.value });
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    e.stopPropagation();

    if (e.currentTarget.checkValidity()) {
      if (icon_id === "") {
        parishApi.addCategory(data).then((res) => {
          console.log("Add Category Response:", res);
          if (res && res.success) {
            getList();
            toast.success(res.msg);
            clearfield();
            log("Add", res.data?.id);
          } else {
            toast.error(res?.msg || "Something went wrong");
          }
        });
      } else {
        parishApi.updateCategory(data, icon_id).then((res) => {
          if (res && res.success) {
            getList();
            toast.success(res.msg);
            clearfield();
            log("Edit", icon_id);
          } else {
            toast.error(res?.msg || "Something went wrong");
          }
        });
      }
    } else {
      setValidated(true);
    }
  };

  const clearfield = () => {
    setData({ name: "" });
    setIcon_id("");
  };

  const handleShow = (item) => {
    setShow(true);
    setIcon_id(item);
    setData({ name: item.name });
  };

  const handleClose = () => setShow(false);

  const DeleteData = () => {
    parishApi.deleteCategory(data, icon_id.id).then((res) => {
      if (res.success) {
        getList();
        toast.success(res.msg);
        log("Delete", icon_id);
        clearfield();
      } else {
        toast.error(res.msg);
      }
      setShow(false);
    });
  };

  const getList = () => {
    setLoadData(true);
    parishApi
      .getAllCategories()
      .then((res) => {
        console.log("API Response ===>", res); // ✅ Check output here
        const sorted = res.data.sort((a, b) => a.name.localeCompare(b.name));
        setList(sorted);
        setLoadData(false);
      })
      .catch((err) => {
        console.error("Error fetching categories:", err); // ⚠️ Show error in console
        setLoadData(false);
      });
  };

  const SearchData = (event) => {
    setLoadData(true);
    const key = event.target.value;
    if (key) {
      console.warn(key);
      parishApi
        .searchCategory(key) // ✅ Fix here
        .then((response) => {
          setList(response.data);
          setLoadData(false);
        })
        .catch((error) => {
          setLoadData(false);
          toast.error(error.toString());
        });
    } else {
      setLoadData(false);
      getList();
    }
  };

  const log = async (action, id) => {
    await adminApi.addLogReport({
      byWhom: `${userData?.userName || ""} (${userData?.userRole || ""})`,
      pagelink: window.location.href,
      description: `${action} Parish Category (Id: ${id}, Name: ${data.name})`,
      action,
    });
  };

  useEffect(() => {
    getList();
    log("View", "");
  }, []);

  const columns = [
    ...(userData?.userRole === "ADMIN"
      ? [
          {
            id: 2,
            name: "Action",
            cell: (row) => (
              <>
                <button
                  className="font-size"
                  onClick={() => {
                    setIcon_id(row.id);
                    setData({ name: row.name });
                  }}
                >
                  <i className="mdi mdi-pencil-box text-md text-primary" />
                </button>
                {row.name === "OTHER" ? null : (
                  <button className="font-size" onClick={() => handleShow(row)}>
                    <i className="mdi mdi-delete-forever text-md text-danger mx-2" />
                  </button>
                )}
              </>
            ),
          },
        ]
      : []),

    {
      id: 1,
      name: "Parish Category",
      selector: (row) => row.name,
      sortable: true,
      cell: (row) => (
        <OverlayTrigger
          placement="right"
          overlay={<Tooltip>{row.name}</Tooltip>}
        >
          <span>{row.name}</span>
        </OverlayTrigger>
      ),
    },

    {
      id: 3,
      name: "Created At",
      selector: (row) => row.createdAt,
      sortable: true,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          overlay={
            <Tooltip>
              {Helpers.convertUtcToTimezone(
                row.createdAt,
                "MM-DD-YYYY hh:mm A"
              )}
            </Tooltip>
          }
        >
          <span>
            {Helpers.convertUtcToTimezone(row.createdAt, "MM-DD-YYYY")}
          </span>
        </OverlayTrigger>
      ),
    },
  ];

  return (
    <div className="row">
      <div className="col-sm-12">
        <div className="col-md-6 grid-margin stretch-card">
          <div className="card custom-card">
            <div className="card-body">
              <h4 className="card-title Couple_heading">Add Parish Category</h4>
              <Form onSubmit={handleSubmit} noValidate validated={validated}>
                <div className="form-group row flex-grow Couple_heading">
                  <Form.Group className="col-sm-11 d-flex">
                    <Form.Control
                      required
                      type="text"
                      placeholder="Enter new category"
                      name="name"
                      value={data.name}
                      onChange={handleChanges}
                    />
                    <div className="text-big mx-2">
                      <button
                        className="font-size"
                        type="submit"
                        disabled={cp || isError}
                      >
                        {icon_id == "" ? (
                          <i
                            className="mdi mdi-plus-circle font-size"
                            title="Add"
                          ></i>
                        ) : (
                          <i
                            className="mdi mdi-checkbox-marked-circle mx-2 font-size"
                            title="Edit"
                          ></i>
                        )}
                      </button>
                    </div>
                  </Form.Group>
                </div>
              </Form>
            </div>
          </div>
        </div>

        <div className="col-lg-12 grid-margin stretch-card">
          <div className="card custom-card">
            <div className="card-body">
              <div className="d-flex justify-content-between align-items-start Couple_heading flex-wrap gap-2">
                <h4 className="card-title">Parish Categories</h4>
                <div className="d-flex align-items-end justify-content-end ms-auto gap-2">
                  <Button
                    className="btn-sm"
                    variant="outline-primary"
                    onClick={() => navigate("/parish")}
                  >
                    Manage Parish
                  </Button>
                  <div className="form-group mb-0">
                    <div className="input-group setSearch">
                      <input
                        onChange={SearchData}
                        type="text"
                        className="form-control searchData"
                        placeholder="Search..."
                        aria-label="Recipient's username"
                        aria-describedby="basic-addon2"
                        style={{ minWidth: "200px" }}
                      />
                      <div className="input-group-append">
                        <button
                          className="btn btn-sm btn-primary"
                          type="button"
                        >
                          <i className="icon-search"></i>
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              <CustomDataTable
                loading={LoadData}
                data={list}
                columns={columns}
                pointerOnHover
                highlightOnHover
              />
            </div>
          </div>
        </div>
      </div>

      <DeleteModel
        show={show}
        content={
          <>
            <h5 className="mt-4 fw-bold">Parish Category Detail</h5>
            <div className="p-3 border rounded-3 bg-light shadow-sm">
              {icon_id.name && (
                <div className="mb-2">
                  <span className="fw-bold">Parish Category:</span>{" "}
                  {icon_id.name}{" "}
                </div>
              )}
            </div>
          </>
        }
        handleClose={handleClose}
        DeleteData={DeleteData}
      />
    </div>
  );
};

export default ParishCategories;
