import React, { useEffect, useState, useRef } from "react";
import { useParams, Link } from "react-router-dom";
import { Card, Form, Button, Spinner } from "react-bootstrap";
import { toast } from "react-toastify";
import AuthApi from "../../api/authApi";
import Helpers from "../../components/Helpers";
import companyLogo from "../../assets/images/logo.png";

const TicketReply = () => {
  const { id } = useParams();
  const authApi = new AuthApi();

  const [ticket, setTicket] = useState(null);
  const [messages, setMessages] = useState([]);
  const [loading, setLoading] = useState(true);
  const [reply, setReply] = useState("");
  const [attachments, setAttachments] = useState([]);
  const [previewUrls, setPreviewUrls] = useState([]);
  const [sending, setSending] = useState(false);
  const [previewImage, setPreviewImage] = useState(null); // ✅ full-page image preview

  const messagesEndRef = useRef(null);

  useEffect(() => {
    fetchTicketData();
  }, [id]);

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  }, [messages]);

  const fetchTicketData = async () => {
    try {
      setLoading(true);
      const ticketRes = await authApi.getTicketById({ ticket_id: id });
      const messagesRes = await authApi.getTicketMessages({ ticket_id: id });

      if (ticketRes.success) setTicket(ticketRes.data);
      else toast.error(ticketRes.msg);

      if (messagesRes.success) setMessages(messagesRes.data);
      else toast.error(messagesRes.msg);
    } catch {
      toast.error("Error fetching ticket data");
    } finally {
      setLoading(false);
    }
  };

  const handleFileChange = (e) => {
    const files = Array.from(e.target.files || []);
    if (!files.length) return;

    setAttachments((prev) => {
      const combined = [...prev];
      files.forEach((f) => {
        const exists = combined.some(
          (p) => p.name === f.name && p.size === f.size && p.type === f.type
        );
        if (!exists) combined.push(f);
      });
      return combined;
    });

    e.target.value = null;
  };

  useEffect(() => {
    if (!attachments.length) {
      setPreviewUrls([]);
      return;
    }

    const urls = attachments.map((file) => ({
      url: URL.createObjectURL(file),
      type: file.type,
    }));

    setPreviewUrls(urls);

    return () => urls.forEach((u) => URL.revokeObjectURL(u.url));
  }, [attachments]);

  const removeAttachment = (index) => {
    setAttachments((prev) => prev.filter((_, i) => i !== index));
  };

  const handleSendReply = async () => {
    if (!reply.trim() && attachments.length === 0) {
      toast.error("Please type a reply or add attachments before sending.");
      return;
    }

    try {
      setSending(true);

      const payload = new FormData();
      payload.append("ticket_id", ticket.id);
      payload.append("message", reply);
      payload.append("userId", ticket.assignedTo || "ADMIN");

      attachments.forEach((file) => payload.append("attachments", file));

      const res = await authApi.addTicketMessage(payload, {
        headers: { "Content-Type": "multipart/form-data" },
      });

      if (res.success) {
        const newMessage = {
          ...res.data,
          attachments: Array.isArray(res.data.attachments)
            ? res.data.attachments
            : res.data.attachments
            ? JSON.parse(res.data.attachments)
            : [],
        };
        setMessages([newMessage, ...messages]);
        setReply("");
        setAttachments([]);
        setPreviewUrls([]);
        toast.success("Reply sent!");
      } else toast.error(res.msg);
    } catch {
      toast.error("Failed to send reply");
    } finally {
      setSending(false);
    }
  };

  const parseAttachments = (attachments) => {
    let files = [];
    if (attachments) {
      try {
        let parsed =
          typeof attachments === "string"
            ? JSON.parse(attachments)
            : attachments;
        if (typeof parsed === "string") parsed = JSON.parse(parsed);
        if (Array.isArray(parsed)) files = parsed;
      } catch {
        files = [];
      }
    }
    return files;
  };

  if (loading)
    return (
      <div className="d-flex justify-content-center align-items-center vh-100">
        <Spinner animation="border" />
      </div>
    );

  return (
    <div className="container-scroller mainpannel">
      <div className="container-fluid page-body-wrapper full-page-wrapper">
        <div className="content-wrapper d-flex align-items-center auth px-0">
          <div className="container py-4">
            <div className="row w-100 mx-0">
              <div className="col-lg-10 mx-auto">
                <Link to="/dashboard">
                  <div className="brand-logo">
                    <img src={companyLogo} alt="logo" className="logo center" />
                  </div>
                </Link>
              </div>

              {ticket && (
                <Card className="mb-4 shadow-sm border-0">
                  <Card.Body>
                    <h3 className="mb-3 bold">Post a Ticket</h3>
                    <h4 className="mb-3">{ticket.subject}</h4>
                    <div className="mb-2 text-muted small">
                      <strong>Created:</strong>{" "}
                      {Helpers.convertUtcToTimezone(
                        ticket.createdAt,
                        "M/D/YYYY"
                      )}{" "}
                      | <strong>Updated:</strong>{" "}
                      {Helpers.convertUtcToTimezone(
                        ticket.updatedAt,
                        "M/D/YYYY"
                      )}
                    </div>
                    <hr />
                    <div className="row g-3">
                      <div className="col-md-4">
                        <strong>From:</strong> {ticket.firstName}{" "}
                        {ticket.lastName} ({ticket.email})
                      </div>
                      <div className="col-md-4">
                        <strong>Status:</strong>{" "}
                        <span
                          className="badge"
                          style={{
                            backgroundColor:
                              ticket.status === "NEW"
                                ? "#52cdff" // blue
                                : ticket.status === "INPROGRESS"
                                ? "#ffa725" // orange
                                : ticket.status === "CLOSED"
                                ? "#9E9E9E" // gey
                                : "",
                            color:
                              ticket.status === "INPROGRESS"
                                ? "black"
                                : "white", // contrast fix
                          }}
                        >
                          {ticket.status}
                        </span>
                      </div>

                      <div className="col-md-4">
                        <strong>Priority:</strong>{" "}
                        <span
                          className="badge"
                          style={{
                            backgroundColor:
                              ticket.priority === "HIGH"
                                ? "#F44336" // Red
                                : ticket.priority === "MEDIUM"
                                ? "#FFC107" // Yellow
                                : ticket.priority === "LOW"
                                ? "#4CAF50" // Green
                                : "",
                            color:
                              ticket.priority === "MEDIUM" ? "black" : "white", // contrast for orange
                          }}
                        >
                          {ticket.priority}
                        </span>
                      </div>
                    </div>
                  </Card.Body>
                </Card>
              )}

              {/* Reply Box */}
              <Card className="shadow-sm border-0 mb-3">
                <Card.Body>
                  <Form.Control
                    as="textarea"
                    rows={4}
                    value={reply}
                    onChange={(e) => setReply(e.target.value)}
                    placeholder="Type your reply..."
                    className="mb-2 replyField"
                  />
                  <Form.Control
                    type="file"
                    multiple
                    onChange={handleFileChange}
                    className="mb-2 fileUploadInput"
                  />

                  {attachments.length > 0 && (
                    <ul className="list-group mt-2">
                      {attachments.map((file, idx) => {
                        const preview = previewUrls[idx]?.url;
                        const isImage = file.type.startsWith("image/");
                        const isVideo = file.type.startsWith("video/");

                        return (
                          <li
                            key={idx}
                            className="list-group-item d-flex justify-content-between align-items-center"
                          >
                            <div className="d-flex align-items-center">
                              {isImage && preview && (
                                <img
                                  src={preview}
                                  alt={file.name}
                                  style={{
                                    width: 48,
                                    height: 48,
                                    objectFit: "cover",
                                    marginRight: 10,
                                    cursor: "pointer",
                                  }}
                                  onClick={() => setPreviewImage(preview)}
                                />
                              )}
                              {isVideo && preview && (
                                <video
                                  src={preview}
                                  style={{
                                    width: 80,
                                    height: 48,
                                    objectFit: "cover",
                                    marginRight: 10,
                                  }}
                                  muted
                                  playsInline
                                />
                              )}
                              <div>
                                <div>{file.name}</div>
                                <small className="text-muted">
                                  {(file.size / 1024 / 1024).toFixed(2)} MB
                                </small>
                              </div>
                            </div>
                            <Button
                              variant="outline-danger"
                              size="sm"
                              onClick={() => removeAttachment(idx)}
                            >
                              ✕
                            </Button>
                          </li>
                        );
                      })}
                    </ul>
                  )}

                  <div className="text-end mt-2">
                    <Button
                      variant="primary"
                      disabled={sending}
                      onClick={handleSendReply}
                    >
                      {sending ? "Sending..." : "Send Reply"}
                    </Button>
                  </div>
                </Card.Body>
              </Card>

              {/* Messages */}
              <div
                className="p-3 border rounded bg-white"
                style={{ maxHeight: "400px", overflowY: "auto" }}
              >
                {messages.length === 0 && (
                  <div className="text-center text-muted">No messages yet</div>
                )}

                {messages.map((msg) => {
                  const isCustomer = msg.senderEmail === ticket?.email;
                  const files = parseAttachments(msg.attachments);

                  return (
                    <div key={msg.id} className="mb-3">
                      <Card className="shadow-sm">
                        <div className="d-flex">
                          <div
                            className="p-3 text-white d-flex align-items-center justify-content-center"
                            style={{
                              background: isCustomer ? "#0d6efd" : "#6c757d",
                              borderTopLeftRadius: "0.5rem",
                              borderBottomLeftRadius: "0.5rem",
                              minWidth: 100,
                            }}
                          >
                            <div className="fw-bold text-center senderName">
                              {msg.senderName || msg.senderEmail}
                            </div>
                          </div>

                          <div className="flex-grow-1 p-3">
                            <div className="small text-muted mb-2">
                              {new Date(msg.createdAt).toLocaleString()}
                            </div>
                            <div>{msg.message}</div>

                            {files.length > 0 && (
                              <div className="mt-2 d-flex flex-wrap gap-2">
                                {files.map((url, idx) => {
                                  const ext = url
                                    .split(".")
                                    .pop()
                                    .toLowerCase();
                                  if (
                                    ["jpg", "jpeg", "png", "gif"].includes(ext)
                                  ) {
                                    return (
                                      <img
                                        key={idx}
                                        src={url}
                                        alt="attachment"
                                        style={{
                                          maxWidth: 120,
                                          borderRadius: 4,
                                          cursor: "pointer",
                                        }}
                                        onClick={() => setPreviewImage(url)}
                                      />
                                    );
                                  } else if (
                                    ["mp4", "avi", "mov"].includes(ext)
                                  ) {
                                    return (
                                      <video
                                        key={idx}
                                        src={url}
                                        controls
                                        style={{
                                          maxWidth: 200,
                                          borderRadius: 4,
                                        }}
                                      />
                                    );
                                  } else {
                                    return (
                                      <a
                                        key={idx}
                                        href={url}
                                        target="_blank"
                                        rel="noreferrer"
                                      >
                                        {url.split("/").pop()}
                                      </a>
                                    );
                                  }
                                })}
                              </div>
                            )}
                          </div>
                        </div>
                      </Card>
                    </div>
                  );
                })}
                <div ref={messagesEndRef}></div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Full-page Image Preview */}
      {previewImage && (
        <div
          onClick={() => setPreviewImage(null)}
          style={{
            position: "fixed",
            top: 0,
            left: 0,
            width: "100vw",
            height: "100vh",
            backgroundColor: "rgba(0,0,0,0.8)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            zIndex: 1050,
            cursor: "zoom-out",
          }}
        >
          <img
            src={previewImage}
            alt="preview"
            style={{
              maxWidth: "90%",
              maxHeight: "90%",
              borderRadius: 8,
              boxShadow: "0 0 20px rgba(0,0,0,0.5)",
            }}
          />
        </div>
      )}
    </div>
  );
};

export default TicketReply;
