import { useState, useEffect, useRef } from "react";
import { Modal, Button, Form, Spinner, Card, Row, Col } from "react-bootstrap";
import { toast } from "react-toastify";
import AuthApi from "../../api/authApi";
import HelpApi from "../../api/helpApi";
import Helpers from "../../components/Helpers";

function MessagesModal({
  show,
  handleClose,
  selectedTicket,
  setSelectedTicket,
}) {
  const [messages, setMessages] = useState([]);
  const [newMsg, setNewMsg] = useState("");
  const [attachments, setAttachments] = useState([]);
  const [previewUrls, setPreviewUrls] = useState([]);
  const [data, setData] = useState("");
  const [loading, setLoading] = useState(false);
  const [sending, setSending] = useState(false);
  const [saveChanges, setSaveChanges] = useState(false);
  const [addReply, setAddreply] = useState(false);
  const [previewImage, setPreviewImage] = useState(null); // ✅ Added for image preview

  const chatEndRef = useRef(null);
  const authApi = new AuthApi();
  const helpApi = new HelpApi();

  useEffect(() => {
    if (selectedTicket) fetchMessages();
  }, [selectedTicket]);

  useEffect(() => {
    chatEndRef.current?.scrollIntoView({ behavior: "smooth" });
  }, [messages]);

  const fetchMessages = async () => {
    try {
      setLoading(true);
      const res = await authApi.getTicketMessages({
        ticket_id: selectedTicket.id,
      });
      if (res.success) setMessages(res.data);
      else toast.error(res.msg);
    } catch {
      toast.error("Failed to fetch messages");
    } finally {
      setLoading(false);
    }
  };

  const handleFileChange = (e) => {
    const files = Array.from(e.target.files || []);
    if (!files.length) return;

    setAttachments((prev) => {
      const combined = [...prev];
      files.forEach((f) => {
        const exists = combined.some(
          (p) => p.name === f.name && p.size === f.size && p.type === f.type
        );
        if (!exists) combined.push(f);
      });
      return combined;
    });

    e.target.value = null;
  };

  useEffect(() => {
    if (!attachments || attachments.length === 0) {
      setPreviewUrls([]);
      return;
    }

    const urls = attachments.map((file) => ({
      url: URL.createObjectURL(file),
      type: file.type,
    }));

    setPreviewUrls(urls);

    return () => {
      urls.forEach((u) => {
        try {
          URL.revokeObjectURL(u.url);
        } catch {}
      });
    };
  }, [attachments]);

  const sendMessage = async () => {
    if (!newMsg.trim() && attachments.length === 0) {
      toast.error("Message or attachment required");
      return;
    }
    try {
      setSending(true);

      const payload = new FormData();
      payload.append("ticket_id", selectedTicket.id);
      payload.append("message", newMsg);
      payload.append("userId", "ADMIN");

      attachments.forEach((file) => {
        payload.append("attachments", file);
      });

      const res = await authApi.addTicketMessage(payload, {
        headers: { "Content-Type": "multipart/form-data" },
      });

      if (res.success) {
        setNewMsg("");
        setAttachments([]);
        setPreviewUrls([]);

        const newMessage = {
          ...res.data,
          attachments: Array.isArray(res.data.attachments)
            ? res.data.attachments
            : res.data.attachments
            ? JSON.parse(res.data.attachments)
            : [],
        };

        setMessages([newMessage, ...messages]);
        toast.success("Reply sent!");
        setAddreply(false);
      } else {
        toast.error(res.msg);
      }
    } catch (err) {
      toast.error("Failed to send message");
    } finally {
      setSending(false);
    }
  };

  const handleChangeOptions = (ticket_id, e) => {
    const { name, value } = e;

    setSaveChanges(true);

    setData((prev) => ({
      ...prev,
      id: ticket_id,
      [name]: value,
    }));

    setSelectedTicket((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const saveTicketChanges = async () => {
    try {
      const res = await helpApi.updateStatus(data);

      setSelectedTicket((prev) => ({
        ...prev,
        status: res.data?.status,
        priority: res.data?.priority,
      }));

      toast.success("Ticket updated successfully");
      setSaveChanges(false);
    } catch (error) {
      console.error("Update status error:", error);
      toast.error(error.response?.data?.message || "Failed to update ticket");
    }
  };

  return (
    <Modal show={show} onHide={handleClose} size="lg" centered>
      <Modal.Header closeButton className="bg-light">
        <Modal.Title>
          📝 View Ticket - {selectedTicket.ticketNumber}{" "}
        </Modal.Title>
      </Modal.Header>

      <Modal.Body style={{ background: "#f8f9fa" }}>
        {loading ? (
          <div className="d-flex justify-content-center align-items-center p-5">
            <Spinner animation="border" />
          </div>
        ) : (
          <>
            <h3 className="mb-3">{selectedTicket?.subject}</h3>

            {/* Ticket Meta */}
            <Card className="mb-3 shadow-sm border-0">
              <Card.Body>
                <Row className="mb-3 text-muted small">
                  <Col md="6">
                    <strong>Created:</strong>{" "}
                    {Helpers.convertUtcToTimezone(
                      selectedTicket.createdAt,
                      "M/D/YYYY"
                    )}
                  </Col>
                  <Col md="6" className="text-md-end">
                    <strong>Updated:</strong>{" "}
                    {Helpers.convertUtcToTimezone(
                      selectedTicket.updatedAt,
                      "M/D/YYYY"
                    )}
                  </Col>
                </Row>

                <hr className="my-2" />

                <Row className="align-items-center g-3">
                  <Col md="4">
                    <strong className="me-2">Category:</strong>
                    {selectedTicket?.category}
                  </Col>

                  <Col md="4">
                    <strong className="me-2">Status:</strong>
                    <Form.Select
                      size="sm"
                      name="status"
                      value={selectedTicket.status}
                      onChange={(e) =>
                        handleChangeOptions(selectedTicket.id, e.target)
                      }
                      style={{
                        width: "150px",
                        backgroundColor:
                          selectedTicket.status === "NEW"
                            ? "#52cdff"
                            : selectedTicket.status === "INPROGRESS"
                            ? "#ffa725"
                            : selectedTicket.status === "CLOSED"
                            ? "#9E9E9E"
                            : "",
                        color:
                          selectedTicket.status === "INPROGRESS"
                            ? "black"
                            : "white", // better contrast
                      }}
                    >
                      <option value="NEW">New</option>
                      <option value="INPROGRESS">In Progress</option>
                      <option value="CLOSED">Closed</option>
                    </Form.Select>
                  </Col>

                  <Col md="4">
                    <strong className="me-2">Priority:</strong>
                    <Form.Select
                      name="priority"
                      size="sm"
                      value={selectedTicket.priority || ""}
                      onChange={(e) =>
                        handleChangeOptions(selectedTicket.id, e.target)
                      }
                      style={{
                        width: "150px",
                        backgroundColor:
                          selectedTicket.priority === "HIGH"
                            ? "#F44336"
                            : selectedTicket.priority === "MEDIUM"
                            ? "#FFC107"
                            : selectedTicket.priority === "LOW"
                            ? "#4CAF50"
                            : "#f0f0f0",
                        color:
                          selectedTicket.priority === "MEDIUM"
                            ? "black"
                            : selectedTicket.priority === "HIGH" ||
                              selectedTicket.priority === "LOW"
                            ? "white"
                            : "black", // default text color
                      }}
                    >
                      <option value="">Select Priority</option>
                      <option value="LOW">Low</option>
                      <option value="MEDIUM">Medium</option>
                      <option value="HIGH">High</option>
                    </Form.Select>
                  </Col>
                </Row>
              </Card.Body>
            </Card>

            {/* Actions */}
            <div className="d-flex gap-2 mb-2">
              {!addReply && (
                <Button
                  variant="primary"
                  onClick={() => setAddreply(true)}
                  disabled={sending}
                >
                  Add Reply
                </Button>
              )}
              {saveChanges && (
                <Button variant="success" onClick={saveTicketChanges}>
                  Save Changes
                </Button>
              )}
            </div>

            {/* Reply Section */}
            {addReply && (
              <>
                <Form.Control
                  as="textarea"
                  rows={5}
                  value={newMsg}
                  onChange={(e) => setNewMsg(e.target.value)}
                  placeholder="Type your reply..."
                  className="mb-2 replyField"
                />

                {/* Attachments */}
                <Form.Control
                  className="mt-2 fileUploadInput"
                  type="file"
                  multiple
                  accept=".jpg,.jpeg,.png,.mp4,.avi,.mov"
                  onChange={handleFileChange}
                />

                {attachments.length > 0 && (
                  <ul className="list-group mt-2">
                    {attachments.map((file, idx) => {
                      const preview = previewUrls[idx]?.url;
                      const isImage = file.type.startsWith("image/");
                      const isVideo = file.type.startsWith("video/");

                      return (
                        <li
                          key={idx}
                          className="list-group-item d-flex justify-content-between align-items-center"
                        >
                          <div className="d-flex align-items-center">
                            {isImage && preview && (
                              <img
                                src={preview}
                                alt={file.name}
                                style={{
                                  width: 48,
                                  height: 48,
                                  objectFit: "cover",
                                  marginRight: 10,
                                  cursor: "pointer",
                                }}
                                onClick={() => setPreviewImage(preview)}
                              />
                            )}
                            {isVideo && preview && (
                              <video
                                src={preview}
                                style={{
                                  width: 80,
                                  height: 48,
                                  objectFit: "cover",
                                  marginRight: 10,
                                }}
                                muted
                                playsInline
                              />
                            )}
                            <div>
                              <div>{file.name}</div>
                              <small className="text-muted">
                                {(file.size / 1024 / 1024).toFixed(2)} MB
                              </small>
                            </div>
                          </div>
                          <Button
                            variant="outline-danger"
                            size="sm"
                            onClick={() =>
                              setAttachments((prev) =>
                                prev.filter((_, i) => i !== idx)
                              )
                            }
                          >
                            ✕
                          </Button>
                        </li>
                      );
                    })}
                  </ul>
                )}

                <Button
                  variant="primary"
                  onClick={sendMessage}
                  disabled={sending}
                  className="mt-3"
                >
                  {sending ? "Sending..." : "Send"}
                </Button>
              </>
            )}

            {/* Messages */}
            <div
              className="p-3 border rounded bg-white mt-3"
              style={{ maxHeight: "400px", overflowY: "auto" }}
            >
              <div ref={chatEndRef}></div>
              {messages.length === 0 && (
                <div className="text-center text-muted">No messages yet</div>
              )}

              {messages.map((msg) => {
                let files = [];
                if (msg.attachments) {
                  if (Array.isArray(msg.attachments)) {
                    files = msg.attachments;
                  } else {
                    try {
                      files = JSON.parse(msg.attachments);
                      if (!Array.isArray(files)) files = [];
                    } catch (e) {
                      files = [];
                    }
                  }
                }

                return (
                  <div key={msg.id} className="mb-3">
                    <Card className="shadow-sm">
                      <div className="d-flex">
                        <div
                          className="p-3 text-white d-flex align-items-center justify-content-center"
                          style={{
                            background: "#6c757d",
                            borderTopLeftRadius: "0.5rem",
                            borderBottomLeftRadius: "0.5rem",
                            minWidth: 100,
                          }}
                        >
                          <div className="fw-bold text-center senderName">
                            {msg.senderName || msg.senderEmail}
                          </div>
                        </div>

                        <div className="flex-grow-1 p-3">
                          <div className="small text-muted mb-2">
                            {new Date(msg.createdAt).toLocaleString()}
                          </div>
                          <div>{msg.message}</div>

                          {/* Attachments */}
                          {files.length > 0 && (
                            <div className="mt-2 d-flex flex-wrap gap-2">
                              {files.map((url, idx) => {
                                const ext = url.split(".").pop().toLowerCase();
                                if (
                                  ["jpg", "jpeg", "png", "gif"].includes(ext)
                                ) {
                                  return (
                                    <img
                                      key={idx}
                                      src={url}
                                      alt="attachment"
                                      style={{
                                        maxWidth: 120,
                                        borderRadius: 4,
                                        cursor: "pointer",
                                      }}
                                      onClick={() => setPreviewImage(url)}
                                    />
                                  );
                                } else if (
                                  ["mp4", "avi", "mov"].includes(ext)
                                ) {
                                  return (
                                    <video
                                      key={idx}
                                      src={url}
                                      controls
                                      style={{
                                        maxWidth: 200,
                                        borderRadius: 4,
                                      }}
                                    />
                                  );
                                } else {
                                  return (
                                    <a
                                      key={idx}
                                      href={url}
                                      target="_blank"
                                      rel="noreferrer"
                                    >
                                      {url.split("/").pop()}
                                    </a>
                                  );
                                }
                              })}
                            </div>
                          )}
                        </div>
                      </div>
                    </Card>
                  </div>
                );
              })}
            </div>

            {/* Full-page Image Preview */}
            {previewImage && (
              <div
                onClick={() => setPreviewImage(null)}
                style={{
                  position: "fixed",
                  top: 0,
                  left: 0,
                  width: "100vw",
                  height: "100vh",
                  backgroundColor: "rgba(0,0,0,0.8)",
                  display: "flex",
                  alignItems: "center",
                  justifyContent: "center",
                  zIndex: 1050,
                  cursor: "zoom-out",
                }}
              >
                <img
                  src={previewImage}
                  alt="preview"
                  style={{
                    maxWidth: "90%",
                    maxHeight: "90%",
                    borderRadius: 8,
                    boxShadow: "0 0 20px rgba(0,0,0,0.5)",
                  }}
                />
              </div>
            )}
          </>
        )}
      </Modal.Body>
    </Modal>
  );
}

export default MessagesModal;
