import { useState, useEffect } from "react";
import { toast } from "react-toastify";
import { Modal, Button, Form, Spinner } from "react-bootstrap";
import AuthApi from "../../api/authApi";
import { RefreshCw } from "lucide-react";
import { useTranslation } from "react-i18next";
import { useNavigate, useLocation, Link } from "react-router-dom";
import companyLogo from "../../assets/images/logo.png";

const CreateTicket = () => {
  const { t } = useTranslation();
  const location = useLocation();

  const authApi = new AuthApi();
  const navigate = useNavigate();
  const [validated, setValidated] = useState(false);
  const [captchaText, setCaptchaText] = useState("");
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "",
    email: "",
    mobile: "",
    subject: "",
    message: "",
    captchaAnswer: "",
    category: "",
  });
  const [attachments, setAttachments] = useState([]);
  const [previewUrls, setPreviewUrls] = useState([]);
  const [loadData, setLoadData] = useState(false);
  const [emailValid, setEmailValid] = useState(true);

  const generateCaptcha = () => {
    setCaptchaText(Math.random().toString(36).substring(2, 8).toUpperCase());
  };

  useEffect(() => {
    generateCaptcha();
  }, []);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData({ ...formData, [name]: value });

    // Validate email inline
    if (name === "email") {
      const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      setEmailValid(re.test(value));
    }
  };

  // ✅ Handle file change (append + dedupe)
  const handleFileChange = (e) => {
    const files = Array.from(e.target.files || []);
    if (!files.length) return;

    const validFiles = [];
    const invalidFiles = [];

    files.forEach((file) => {
      const isImage = file.type.startsWith("image/");
      const isVideo = file.type.startsWith("video/");

      // ✅ Check type
      if (!isImage && !isVideo) {
        invalidFiles.push(`${file.name} ${t("invalid_type")}`);
        return;
      }

      // ✅ Check size
      if (isImage && file.size > 3 * 1024 * 1024) {
        invalidFiles.push(`${file.name}, ${t("three_mb_size")}`);
        return;
      }

      if (isVideo && file.size > 10 * 1024 * 1024) {
        invalidFiles.push(`${file.name}, ${t("ten_mb_size")}`);
        return;
      }

      validFiles.push(file);
    });

    // Show invalid files in toast
    if (invalidFiles.length > 0) {
      toast.error(`${t("files_are_invalid")}: ${invalidFiles.join(", ")}`);
    }

    if (validFiles.length === 0) return;

    // Add valid files, avoid duplicates
    setAttachments((prev) => {
      const combined = [...prev];
      validFiles.forEach((f) => {
        const exists = combined.some(
          (p) => p.name === f.name && p.size === f.size && p.type === f.type
        );
        if (!exists) combined.push(f);
      });
      return combined;
    });

    e.target.value = null;
  };

  // ✅ Generate / revoke preview URLs
  useEffect(() => {
    if (!attachments || attachments.length === 0) {
      setPreviewUrls([]);
      return;
    }

    const urls = attachments.map((file) => ({
      url: URL.createObjectURL(file),
      type: file.type,
    }));

    setPreviewUrls(urls);

    return () => {
      urls.forEach((u) => {
        try {
          URL.revokeObjectURL(u.url);
        } catch (err) {}
      });
    };
  }, [attachments]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setValidated(true);
    setLoadData(true);
    if (!formData.firstName || !formData.email || !formData.captchaAnswer) {
      toast.error(`${t("all_required_fields")}`);
      setLoadData(false);
      return;
    }

    try {
      const payload = new FormData();
      Object.keys(formData).forEach((key) => {
        payload.append(key, formData[key]);
      });
      payload.append("captcha", captchaText);
      payload.append("pageName", location?.state?.pageName || "");

      attachments.forEach((file) => {
        payload.append("attachments", file);
      });

      const res = await authApi.createTicket(payload, {
        headers: { "Content-Type": "multipart/form-data" },
      });

      if (res?.success) {
        setLoadData(false);
        if (res.msg === "Invalid captcha") {
          toast.success(`${t("invalid_captcha")}`);
        } else {
          toast.success(`${t("ticket_successfully")}`);
        }
        navigate("/dashboard");
      } else {
        setLoadData(false);
        toast.error(res.msg);
      }
    } catch (error) {
      setLoadData(false);
      console.log("error--->", error);
      toast.error(`${t("failed_to_create_ticket")}`);
    }
  };

  return (
    <div className="container-scroller mainpannel">
      <div className="container-fluid page-body-wrapper full-page-wrapper">
        <div className="content-wrapper d-flex align-items-center auth px-0">
          <div className="row w-100 mx-0">
            <div className="col-lg-10 mx-auto">
              <div className="auth-form-light text-left py-5 px-4 px-sm-5">
                <div className="text-center mb-4">
                  <Link to="/dashboard">
                    <div className="brand-logo">
                      <img
                        src={companyLogo}
                        alt="logo"
                        className="logo center"
                      />
                    </div>
                  </Link>
                  <h3 className="fw-bold text-primary">
                    {`${t("post_a_ticket")}`}
                  </h3>
                </div>

                <div className="ticket-form-container">
                  {loadData && (
                    <div
                      className="d-flex justify-content-center align-items-center"
                      style={{ minHeight: "300px" }}
                    >
                      <Spinner animation="border" style={{ color: "red" }} />
                    </div>
                  )}
                  {!loadData && (
                    <Form
                      noValidate
                      validated={validated}
                      onSubmit={handleSubmit}
                    >
                      <div className="row g-3">
                        {/* Ticket Category */}
                        <div className="col-12">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {t("ticket_category")}{" "}
                              <span className="text-danger">*</span>
                            </Form.Label>
                            <Form.Select
                              required
                              name="category"
                              value={formData.category}
                              onChange={handleChange}
                              className="locationDropdown"
                            >
                              <option value="">{t("select_category")}</option>
                              <option value="Retreat Registration">
                                {t("retreat_registration")}
                              </option>
                              <option value="Retreat Financials">
                                {t("retreat_financials")}
                              </option>
                              <option value="Retreat Roster">
                                {t("retreat_roster")}
                              </option>
                              <option value="Retreat Closure">
                                {t("retreat_closure")}
                              </option>
                              <option value="Database Issues">
                                {t("database_issues")}
                              </option>
                              <option value="Retreat Reports">
                                {t("retreat_reports")}
                              </option>
                              <option value="Request to Feature">
                                {t("request_to_feature")}
                              </option>
                              <option value="How do I...">
                                {t("how_do_i")}
                              </option>
                              <option value="Other">{t("other")}</option>
                            </Form.Select>
                            <Form.Control.Feedback type="invalid">
                              {t("category_reqired")}
                            </Form.Control.Feedback>
                          </Form.Group>
                        </div>

                        {/* Subject */}
                        <div className="col-12">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("subject")}`}{" "}
                              <span className="text-danger">*</span>
                            </Form.Label>
                            <Form.Control
                              required
                              type="text"
                              name="subject"
                              placeholder={`${t("enter_ticket_subject")}`}
                              value={formData.subject}
                              onChange={handleChange}
                            />
                          </Form.Group>
                        </div>

                        {/* Message */}
                        <div className="col-12">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("message")}`}{" "}
                              <span className="text-danger">*</span>
                            </Form.Label>
                            <Form.Control
                              as="textarea"
                              rows={5}
                              required
                              name="message"
                              placeholder={`${t("describe_detail")}`}
                              value={formData.message}
                              onChange={handleChange}
                            />
                          </Form.Group>
                        </div>

                        <div className="col-12">
                          <h4 className="bold">Enter Your Info:</h4>
                        </div>

                        {/* First Name */}
                        <div className="col-md-6">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("first_name")}`}{" "}
                              <span className="text-danger">*</span>
                            </Form.Label>
                            <Form.Control
                              required
                              type="text"
                              name="firstName"
                              placeholder={`${t("enter_your_name")}`}
                              value={formData.firstName}
                              onChange={handleChange}
                            />
                          </Form.Group>
                        </div>

                        {/* Last Name */}
                        <div className="col-md-6">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("last_name")}`}
                            </Form.Label>
                            <Form.Control
                              required
                              type="text"
                              name="lastName"
                              placeholder={`${t("enter_your_last_name")}`}
                              value={formData.lastName}
                              onChange={handleChange}
                            />
                          </Form.Group>
                        </div>

                        {/* Email */}
                        <div className="col-md-6">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("email")}`}{" "}
                              <span className="text-danger">*</span>
                            </Form.Label>
                            <Form.Control
                              required
                              type="email"
                              name="email"
                              placeholder={`${t("enter_your_email")}`}
                              value={formData.email}
                              onChange={handleChange}
                              isInvalid={!emailValid} // ✅ show inline feedback
                            />
                            <Form.Control.Feedback type="invalid">
                              {t("invalid_email")}
                            </Form.Control.Feedback>
                          </Form.Group>
                        </div>

                        {/* Mobile */}
                        <div className="col-md-6">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {`${t("mobile")}`}
                            </Form.Label>
                            <Form.Control
                              required
                              type="text"
                              name="mobile"
                              placeholder={`${t("enter_your_mobile")}`}
                              value={formData.mobile}
                              onChange={handleChange}
                            />
                          </Form.Group>
                        </div>

                        {/* Attachments */}
                        <div className="col-12">
                          <Form.Group>
                            <Form.Label className="fw-semibold">
                              {t("attachments")}{" "}
                              <small className="text-muted">
                                {t("allowed")}{" "}
                              </small>
                            </Form.Label>
                            <Form.Control
                              type="file"
                              accept="image/*,video/*"
                              multiple
                              onChange={handleFileChange}
                            />
                          </Form.Group>

                          {attachments.length > 0 && (
                            <div className="mt-3">
                              <h6 className="fw-semibold mb-2">
                                {t("selected_files") || "Selected files"}
                              </h6>
                              <ul className="list-group">
                                {attachments.map((file, idx) => {
                                  const preview = previewUrls[idx]?.url;
                                  const isImage =
                                    file.type.startsWith("image/");
                                  const isVideo =
                                    file.type.startsWith("video/");

                                  return (
                                    <li
                                      key={`${file.name}-${file.size}-${idx}`}
                                      className="list-group-item d-flex justify-content-between align-items-center"
                                    >
                                      <div className="d-flex align-items-center">
                                        {isImage && preview ? (
                                          <img
                                            src={preview}
                                            alt={file.name}
                                            style={{
                                              width: "48px",
                                              height: "48px",
                                              objectFit: "cover",
                                              borderRadius: "6px",
                                              marginRight: "12px",
                                            }}
                                          />
                                        ) : isVideo && preview ? (
                                          <video
                                            src={preview}
                                            style={{
                                              width: "80px",
                                              height: "48px",
                                              objectFit: "cover",
                                              borderRadius: "6px",
                                              marginRight: "12px",
                                            }}
                                            muted
                                            playsInline
                                          />
                                        ) : (
                                          <span
                                            style={{
                                              display: "inline-block",
                                              width: 48,
                                              height: 48,
                                              lineHeight: "48px",
                                              textAlign: "center",
                                              borderRadius: 6,
                                              background: "#f1f1f1",
                                              marginRight: 12,
                                              fontSize: 12,
                                            }}
                                          >
                                            {file.type.split("/")[1] || "file"}
                                          </span>
                                        )}
                                        <div>
                                          <div
                                            className="fw-medium"
                                            style={{ maxWidth: 380 }}
                                          >
                                            {file.name}
                                          </div>
                                          <small className="text-muted">
                                            {(file.size / 1024 / 1024).toFixed(
                                              2
                                            )}{" "}
                                            MB • {file.type || "unknown"}
                                          </small>
                                        </div>
                                      </div>

                                      <Button
                                        variant="outline-danger"
                                        size="sm"
                                        onClick={() =>
                                          setAttachments((prev) =>
                                            prev.filter((_, i) => i !== idx)
                                          )
                                        }
                                      >
                                        ✕
                                      </Button>
                                    </li>
                                  );
                                })}
                              </ul>
                            </div>
                          )}
                        </div>

                        {/* Captcha */}
                        <div className="row g-3 align-items-center">
                          <div className="col-md-2">
                            <div className="captcha-box d-flex align-items-center justify-content-between">
                              <span>{captchaText}</span>
                              <RefreshCw
                                className="refresh-icon ms-2"
                                onClick={generateCaptcha}
                                size={18}
                              />
                            </div>
                          </div>
                          <div className="col-md-2">
                            <Form.Group>
                              <Form.Label className="fw-semibold">
                                {`${t("enter_captcha")}`}{" "}
                                <span className="text-danger">*</span>
                              </Form.Label>
                              <Form.Control
                                required
                                type="text"
                                name="captchaAnswer"
                                placeholder={`${t("enter_captcha")}`}
                                value={formData.captchaAnswer}
                                onChange={handleChange}
                              />
                            </Form.Group>
                          </div>
                        </div>
                      </div>

                      {/* Buttons */}
                      <div className="d-flex justify-content-end mt-4">
                        <Link to="/dashboard">
                          <Button
                            variant="secondary"
                            className="px-4 py-2 me-3"
                          >
                            {t("cancel")}
                          </Button>
                        </Link>
                        <Button
                          variant="primary"
                          type="submit"
                          className="px-4 py-2"
                        >
                          {t("post_a_ticket")}
                        </Button>
                      </div>
                    </Form>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default CreateTicket;
