import React, { useState, useEffect } from "react";
import { Card, Row, Col, Container, Spinner } from "react-bootstrap";
import { toast } from "react-toastify";
import AuthApi from "../../api/authApi";
import moment from "moment-timezone";
import companyLogo from "../../assets/images/logo.png";
import dummyImage from "../../assets/images/uploadRetreat.png";

import { Link } from "react-router-dom";

const SITE_URL = process.env.REACT_APP_FRONTEND_URL;

const UpcomingRetreats = () => {
  const [upcomingRetreats, setUpcomingRetreats] = useState([]);
  const [loading, setLoading] = useState(false);
  let authApi = new AuthApi();

  useEffect(() => {
    const fetchRetreats = async () => {
      setLoading(true);
      try {
        const response = await authApi.getAllIncommingRetreats();
        if (response.success) {
          const currentDate = new Date();
          const upcoming = response.data.filter((retreat) => {
            const retreatDate = new Date(retreat.dateFrom);
            return retreatDate >= currentDate;
          });
          setUpcomingRetreats(upcoming);
        } else {
          toast.error("Failed to fetch upcoming retreats");
        }
      } catch (error) {
        toast.error("Error fetching retreats");
      }
      setLoading(false);
    };

    fetchRetreats();
  }, []);

  const handleCardClick = (id) => {
    window.open(`${SITE_URL}/registration-retreat/${id}`, "_blank");
  };

  return (
    <div className="container-scroller mainpannel">
      <div className="container-fluid page-body-wrapper full-page-wrapper">
        <div className="content-wrapper d-flex align-items-center auth px-0">
          <Container className="mt-5">
            {/* Logo */}
            <Link
              to="/dashboard"
              className="d-flex justify-content-center mb-4"
            >
              <img
                src={companyLogo}
                alt="logo"
                style={{ width: "140px", height: "auto" }}
              />
            </Link>

            {/* Title */}
            <h2 className="fw-bold text-center mb-5 text-primary">
              Upcoming Retreats
            </h2>

            {loading ? (
              <div className="text-center my-5">
                <Spinner animation="border" variant="primary" />
                <p className="mt-2 text-muted">Loading retreats...</p>
              </div>
            ) : upcomingRetreats.length === 0 ? (
              <p className="text-center text-muted fst-italic">
                No upcoming retreats available.
              </p>
            ) : (
              <Row className="g-4">
                {upcomingRetreats.map((retreat) => {
                  const retreatImage =
                    retreat.image1 || retreat.image2 || dummyImage;

                  return (
                    <Col md={6} xs={12} key={retreat.id}>
                      <Card
                        className="border-0 shadow-lg h-100 retreat-card"
                        style={{
                          borderRadius: "18px",
                          overflow: "hidden",
                          cursor: "pointer",
                          transition:
                            "transform 0.3s ease, box-shadow 0.3s ease",
                        }}
                        onClick={() => handleCardClick(retreat.id)}
                        onMouseEnter={(e) =>
                          (e.currentTarget.style.transform = "translateY(-6px)")
                        }
                        onMouseLeave={(e) =>
                          (e.currentTarget.style.transform = "translateY(0)")
                        }
                      >
                        <div className="d-flex flex-column flex-md-row h-100">
                          {/* Left Side (Image) */}
                          <div
                            style={{
                              height: "200px",
                              width: "200px",
                              flexShrink: 0,
                            }}
                          >
                            <img
                              src={retreatImage}
                              alt="Retreat"
                              style={{
                                height: "100%",
                                width: "100%",
                                objectFit: "contain", // keeps the full image visible
                              }}
                            />
                          </div>

                          {/* Right Side (Text Info) */}
                          <div
                            className="retreat-list-title"
                            style={{ flex: 1 }}
                          >
                            <h3 className="text-danger fw-bold">
                              {retreat?.parishName}
                              {retreat.language
                                ? ` (${retreat.language[0].toUpperCase()}${retreat.language.slice(
                                    1
                                  )}) `
                                : null}
                            </h3>
                            <h3>
                              {retreat.location
                                ? `${retreat.location[0].toUpperCase()}${retreat.location.slice(
                                    1
                                  )}`
                                : ""}
                            </h3>
                            <p>
                              {retreat.dateFrom === retreat.dateTo
                                ? `${moment(retreat && retreat.dateFrom)
                                    .tz("America/New_York")
                                    .format("MMM-D-YYYY")}`
                                : `${
                                    moment(retreat && retreat.dateFrom)
                                      .tz("America/New_York")
                                      .format("MMM D") +
                                    " - " +
                                    moment(retreat.dateTo)
                                      .tz("America/New_York")
                                      .format("MMM D") +
                                    ", " +
                                    moment(retreat && retreat.dateFrom)
                                      .tz("America/New_York")
                                      .format("YYYY")
                                  }`}
                            </p>
                          </div>
                        </div>
                      </Card>
                    </Col>
                  );
                })}
              </Row>
            )}
          </Container>
        </div>
      </div>
    </div>
  );
};

export default UpcomingRetreats;
