import React, { useState } from "react";
import { Form, Button, Row, Col, Modal, Card, Badge } from "react-bootstrap";
import { toast } from "react-toastify";
import AuthApi from "../../api/authApi";
import { useNavigate } from "react-router-dom";
import moment from "moment-timezone";

const NewsletterLink = () => {
  const [showModal, setShowModal] = useState(false);
  const [selectedRetreats, setSelectedRetreats] = useState([]);
  const [upcomingRetreats, setUpcomingRetreats] = useState([]);
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    mobile: "",
  });
  const navigate = useNavigate();
  let authApi = new AuthApi();

  const handleOpenModal = async () => {
    try {
      const response = await authApi.getAllIncommingRetreats();
      if (response.success) {
        const currentDate = new Date();
        const upcoming = response.data.filter((retreat) => {
          const retreatDate = new Date(retreat.dateFrom);
          return retreatDate >= currentDate;
        });

        setUpcomingRetreats(upcoming);
        setShowModal(true);
      } else {
        toast.error("Failed to fetch upcoming retreats");
      }
    } catch (error) {
      toast.error("Error fetching retreats");
    }
  };

  const handleRetreatSelection = (retreatId) => {
    setSelectedRetreats((prev) =>
      prev.includes(retreatId)
        ? prev.filter((id) => id !== retreatId)
        : [...prev, retreatId]
    );
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();

    if (selectedRetreats.length === 0) {
      toast.error("Please select at least one retreat");
      return;
    }

    try {
      const submissionData = {
        ...formData,
        selectedRetreats,
        interestType: "team_participation",
      };

      const response = await authApi.submitTeamInterest(submissionData);

      if (response.success) {
        toast.success("🎉 Your interest has been submitted successfully!");
        setShowModal(false);
        setFormData({ name: "", email: "", mobile: "" });
        setSelectedRetreats([]);
      } else {
        toast.error(response.msg || "Failed to submit interest");
      }
    } catch (error) {
      toast.error("Error submitting interest");
    }
  };

  return (
    <>
      {/* CTA Section */}
      <Card className="shadow-sm border-0 mt-4">
        <Card.Body className="text-center p-4">
          <h5 className="fw-bold">
            🌟 Interested in joining our retreat team?
          </h5>
          <p className="text-muted mb-3">
            Be part of something meaningful. Express your interest to serve as a
            Retreat Team Member!
          </p>
          <Button variant="primary" size="lg" onClick={handleOpenModal}>
            Express Interest
          </Button>
        </Card.Body>
      </Card>

      {/* Modal */}
      <Modal
        show={showModal}
        onHide={() => setShowModal(false)}
        size="lg"
        centered
      >
        <Modal.Header closeButton>
          <Modal.Title className="fw-bold text-primary">
            Express Interest ✨
          </Modal.Title>
        </Modal.Header>
        <Form onSubmit={handleSubmit}>
          <Modal.Body>
            {/* Retreat Selection */}
            <Row className="mb-4">
              <Col md={12}>
                <Form.Label className="fw-semibold">
                  Select Retreat(s) *
                </Form.Label>
                <div className="retreat-selection d-flex flex-wrap gap-3 mt-2">
                  {upcomingRetreats.length === 0 && (
                    <p className="text-muted fst-italic">
                      No upcoming retreats available.
                    </p>
                  )}
                  {upcomingRetreats.map((retreat) => (
                    <Card
                      key={retreat.id}
                      className={`p-3 retreat-card shadow-sm ${
                        selectedRetreats.includes(retreat.id)
                          ? "border border-primary"
                          : "border-light"
                      }`}
                      style={{ cursor: "pointer", minWidth: "250px" }}
                      onClick={() => handleRetreatSelection(retreat.id)}
                    >
                      {/* Parish Name */}
                      <h6 className="fw-bold mb-1">{retreat.parishName}</h6>

                      {/* Language + Location */}
                      <small className="text-muted d-block">
                        {retreat.language &&
                          `${retreat.language[0].toUpperCase()}${retreat.language.slice(
                            1
                          )}`}
                        {retreat.location && ` • ${retreat.location}`}
                      </small>

                      {/* Dates */}
                      <small className="text-primary fw-semibold">
                        {retreat.dateFrom === retreat.dateTo ? (
                          <span>
                            {moment(retreat.dateFrom)
                              .tz("America/New_York")
                              .format("MMM DD, YYYY")}
                          </span>
                        ) : (
                          <span>
                            {moment(retreat.dateFrom)
                              .tz("America/New_York")
                              .format("MMM DD")}{" "}
                            -{" "}
                            {moment(retreat.dateTo)
                              .tz("America/New_York")
                              .format("MMM DD, YYYY")}
                          </span>
                        )}
                      </small>

                      {/* Selected Badge */}
                      {selectedRetreats.includes(retreat.id) && (
                        <Badge bg="primary" className="mt-2">
                          Selected
                        </Badge>
                      )}
                    </Card>
                  ))}
                </div>
              </Col>
            </Row>

            {/* Personal Info */}
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Full Name *</Form.Label>
                  <Form.Control
                    type="text"
                    name="name"
                    value={formData.name}
                    onChange={handleInputChange}
                    required
                    placeholder="Enter your full name"
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Email *</Form.Label>
                  <Form.Control
                    type="email"
                    name="email"
                    value={formData.email}
                    onChange={handleInputChange}
                    required
                    placeholder="Enter your email"
                  />
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>Mobile Number *</Form.Label>
                  <Form.Control
                    type="tel"
                    name="mobile"
                    value={formData.mobile}
                    onChange={handleInputChange}
                    required
                    placeholder="Enter your mobile number"
                    pattern="^[0-9()\\- ]*$"
                  />
                </Form.Group>
              </Col>
            </Row>
          </Modal.Body>
          <Modal.Footer className="d-flex justify-content-between">
            <Button
              variant="outline-secondary"
              onClick={() => setShowModal(false)}
            >
              Cancel
            </Button>
            <Button variant="primary" type="submit">
              🚀 Submit Interest
            </Button>
          </Modal.Footer>
        </Form>
      </Modal>
    </>
  );
};

export default NewsletterLink;
