import React, { useState, useEffect } from "react";
import {
  Form,
  Row,
  Col,
  Button,
  Modal,
  OverlayTrigger,
  Tooltip,
} from "react-bootstrap";
import { toast } from "react-toastify";
import { Cookies, useCookies } from "react-cookie";
import RetreatApi from "../../api/retreatApi";
import AdminApi from "../../api/admin";
import DeleteModel from "../../components/DeleteModel";
import CommitteeApi from "../../api/committeeApi";
import moment from "moment";
import Select from "react-select";
import CustomDataTable from "../../components/CustomDataTable";
import Helpers from "../../components/Helpers";
import { useNavigate } from "react-router-dom";

const Committee = () => {
  let navigate = useNavigate();
  let retreatApi = new RetreatApi();
  let committeeApi = new CommitteeApi();
  let adminApi = new AdminApi();
  const [showRegModal, setShowREgModal] = useState(false);
  const [validated, setValidated] = useState(false);
  const [retreatOptions, setRetreatOptions] = useState([]);
  const [selectedRetreats, setSelectedRetreats] = useState([]);
  const [allCommittee, setAllCommittee] = useState([]);
  const [loadData, setLoadData] = useState(false);
  const [searchKey, setSearchKey] = useState("");
  const [total, setTotal] = useState(20);
  const [show, setShow] = useState(false);
  const [memberId, setMemberId] = useState("");

  const [data, setData] = useState({
    id: "",
    firstName: "",
    lastName: "",
    eMail: "",
    mobile: "",
    retreatIds: [],
  });

  let cookies = new Cookies();
  let userData = cookies.get("LoveStrongUser") || null;
  const [cookie, setCookie] = useCookies(["LoveStrongUser"]);

  useEffect(() => {
    allCommitteMembers();
    getAllRetreats();
  }, []);

  const allCommitteMembers = async (
    pageNo = 1,
    size = 20,
    searchKeyword = null,
    sortBy = {}
  ) => {
    setLoadData(true);
    let d = {};
    if (pageNo && size) {
      d.pageNo = pageNo;
      d.size = size;
    }
    if (searchKeyword !== null) {
      d.searchKeyword = searchKeyword;
    } else {
      d.searchKeyword = searchKey;
    }
    if (sortBy.sortColumn) {
      d.sortBy = sortBy;
    }
    await committeeApi
      .getAllCommitteMembers(d)
      .then((res) => {
        if (res.success) {
          setLoadData(false);
          let totalRecords = res.totalRecords ? res.totalRecords : 0;
          setTotal(totalRecords);
          setAllCommittee(res.data);
        } else {
          setLoadData(false);
          toast.error(res.msg);
        }
      })
      .catch((error) => {
        setLoadData(false);
        toast.error(error);
      });
  };

  const getAllRetreats = async () => {
    try {
      const response = await retreatApi.getAllRetreatSearch();
      const sortedList = response.data.sort((a, b) =>
        a.title.localeCompare(b.title)
      );
      let retreatList = sortedList.map((item) => ({
        label: item.title
          ? `${item.title.charAt(0).toUpperCase() + item.title.slice(1)} ${
              moment(item.dateFrom).format("MMM DD") +
              " - " +
              moment(item.dateTo).format("MMM DD") +
              ", " +
              moment(item.dateFrom).format("YYYY")
            }`
          : "",
        value: item.id,
        dateFrom: moment(item.dateFrom),
      }));
      setRetreatOptions(retreatList);
    } catch (error) {
      toast.error(error.toString());
    }
  };

  const handleSelectRetreat = (selectedOptions) => {
    setSelectedRetreats(selectedOptions);
    const ids = selectedOptions.map((option) => option.value);
    setData((prev) => ({
      ...prev,
      retreatIds: ids,
    }));
  };

  const handleChanges = (e) => {
    let { name, value } = e.target;
    setData((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    e.stopPropagation();
    const form = e.currentTarget;
    if (form.checkValidity()) {
      try {
        let data1 = new FormData();
        let stateRef = { ...data };
        data1.append("fromData", JSON.stringify(stateRef));
        committeeApi
          .createCommitteeMember(data1)
          .then(async (res) => {
            if (res.success) {
              let report = {
                byWhom:
                  `${
                    userData && userData.userName ? userData.userName : null
                  } ` +
                  ` (${
                    userData && userData.userRole ? userData.userRole : null
                  })`,
                pagelink: window.location.href,
                description:
                  `Added new committee member` +
                  ` (${data && data.firstName ? `${data.firstName}` : ""}` +
                  `${data && data.lastName ? ` ${data.lastName}` : ""})`,
                action: "Add",
              };
              await adminApi
                .addLogReport(report)
                .then((response) => {
                  if (response.success) {
                    console.log("Created new log report.");
                  }
                })
                .catch((error) => {
                  console.log(error);
                });
              toast.success(res.msg);
              clearfield();
              setShowREgModal(false);
              allCommitteMembers();
            } else {
              toast.error(res.msg);
            }
          })
          .catch((error) => {
            console.log(error);
          });
        setShowREgModal(false);
        clearfield();
      } catch (err) {
        toast.error("Error submitting data.");
      }
    }
    setValidated(true);
  };

  const SearchData = (event) => {
    const key = event.target.value;
    if (key !== "") {
      setSearchKey(event.target.value);
      allCommitteMembers(1, 20, event.target.value);
    } else {
      setSearchKey("");
      allCommitteMembers(1, 20, "");
    }
  };

  const clearfield = () => {
    setData({
      id: "",
      firstName: "",
      lastName: "",
      eMail: "",
      mobile: "",
      retreatIds: [],
    });
    setSelectedRetreats([]);
    setValidated(false);
  };

  const renderTooltip = (props) => (
    <Tooltip id="button-tooltip">{props}</Tooltip>
  );

  const handleShow = (item) => {
    setShow(true);
    setMemberId(item);
  };

  const handleClose = () => setShow(false);

  const deleteMember = async () => {
    await committeeApi
      .deleteMember({ id: memberId.id })
      .then(async (res) => {
        if (res.success) {
          allCommitteMembers();
          setShow(false);
          setMemberId("");
          toast.success(res.msg);
          let report = {
            byWhom:
              `${userData && userData.userName ? userData.userName : null} ` +
              ` (${userData && userData.userRole ? userData.userRole : null})`,
            pagelink: window.location.href,
            description:
              `Deleted Committee member` +
              ` (${
                memberId && memberId.firstName ? `${memberId.firstName}` : ""
              }` +
              `${
                memberId && memberId.lastName ? ` ${memberId.lastName}` : ""
              })`,
            action: "Delete",
          };
          await adminApi
            .addLogReport(report)
            .then((response) => {
              if (response.success) {
                console.log("Created new log report.");
              }
            })
            .catch((error) => {
              console.log(error);
            });
        } else {
          toast.error(res.msg);
        }
      })
      .catch((error) => {
        toast.error(error);
      });
  };


const editMember = async (item) => {
  setData({
    ...data,
    id: item.id,
    firstName: item.firstName,
    lastName: item.lastName,
    eMail: item.eMail,
    mobile: item.mobile,
    retreatIds: item.retreatIds,
  });

  // Convert item.retreatIds (IDs) to selected options
  const selected = retreatOptions.filter((r) =>
    item.retreatIds.includes(r.value)
  );
  setSelectedRetreats(selected);

  setShowREgModal(true);
};


  const columns = [
    ...(userData && userData.userRole === "ADMIN"
      ? [
          {
            id: "action",
            name: "Edit",
            button: true,
            cell: (row) => (
              <>
                <button
                  className="font-size"
                  onClick={() => {
                    editMember(row);
                  }}
                >
                  <i
                    className="mdi mdi-pencil-box text-lg text-primary font-size"
                    title="Edit"
                  ></i>
                </button>
                <button className="font-size" onClick={() => handleShow(row)}>
                  <i
                    className="mdi mdi-delete-forever text-md text-danger mx-2 font-size"
                    title="Delete"
                  ></i>
                </button>
              </>
            ),
            reorder: true,
          },
        ]
      : []),
    {
      id: "firstName",
      name: "First Name",
      selector: (row) => row.firstName,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(row.firstName)}
        >
          <span>
            {row.firstName &&
              row.firstName.slice(0, 10) +
                (row.firstName.length > 10 ? "..." : "")}
          </span>
        </OverlayTrigger>
      ),
      sortable: true,
      reorder: true,
    },
    {
      id: "lastName",
      name: "Last Name",
      selector: (row) => row.lastName,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(row.lastName)}
        >
          <span>
            {row.lastName &&
              row.lastName.slice(0, 10) +
                (row.lastName.length > 10 ? "..." : "")}
          </span>
        </OverlayTrigger>
      ),
      sortable: true,
      reorder: true,
    },

    {
      id: "eMail",
      name: "Email",
      selector: (row) => row.eMail,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(row.eMail)}
        >
          <a href={`mailto:${row.eMail}`}>
            <span>
              {row.eMail &&
                row.eMail.slice(0, 12) + (row.eMail.length > 12 ? "..." : "")}
            </span>
          </a>
        </OverlayTrigger>
      ),
      sortable: true,
      reorder: true,
    },
    {
      id: "mobile",
      name: "Mobile",
      selector: (row) => row.mobile,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(row.mobile)}
        >
          <span>
            {row.mobile &&
              row.mobile.slice(0, 10) + (row.mobile.length > 10 ? "..." : "")}
          </span>
        </OverlayTrigger>
      ),
      sortable: true,
      reorder: true,
    },
    {
      id: "title",
      name: "Retreat Title",
      selector: (row) => (
        <>
          <a
            href=""
            onClick={(e) => {
              e.preventDefault();
              navigate(`/retreatRoster?retreatId=${row.id}`);
            }}
          >
            <div>{row.title}</div>
          </a>
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(
              row.dateFrom === row.dateTo
                ? moment(row && row.dateFrom)
                    .tz("America/New_York")
                    .format("MMM-DD-YYYY")
                : moment(row && row.dateFrom)
                    .tz("America/New_York")
                    .format("MMM DD") +
                    " - " +
                    moment(row && row.dateTo)
                      .tz("America/New_York")
                      .format("MMM DD") +
                    ", " +
                    moment(row && row.dateFrom)
                      .tz("America/New_York")
                      .format("YYYY")
            )}
          >
            {row.dateFrom === row.dateTo ? (
              <span>
                {moment(row && row.dateFrom)
                  .tz("America/New_York")
                  .format("MMM-DD-YYYY")}
              </span>
            ) : (
              <span>
                {moment(row && row.dateFrom)
                  .tz("America/New_York")
                  .format("MMM-DD") +
                  " - " +
                  moment(row.dateTo).tz("America/New_York").format("MMM DD") +
                  ", " +
                  moment(row && row.dateFrom)
                    .tz("America/New_York")
                    .format("YYYY")}
              </span>
            )}
          </OverlayTrigger>
        </>
      ),
      sortable: true,
      reorder: true,
      width: "300px",
    },
    {
      id: "created",
      name: "Created",
      selector: (row) =>
        Helpers.convertUtcToTimezone(row.createdAt, "MM-DD-YYYY"),
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(
            Helpers.convertUtcToTimezone(row.createdAt, "DD-MM-YYYY hh:mm:a")
          )}
        >
          <span>
            {Helpers.convertUtcToTimezone(row.createdAt, "MM-DD-YYYY")}
          </span>
        </OverlayTrigger>
      ),
      reorder: true,
      style: {
        cursor: "text",
      },
    },
  ];

  return (
    <>
      <div className="row ext-space">
        <div className="col-lg-12 d-flex flex-column">
          <div className="row flex-grow">
            <div className="col-12 grid-margin stretch-card">
              <div className="card custom-card">
                <div className="card-body">
                  <Row className="m-0">
                    <Col md={4}>
                      <div className="form-group content1">
                        <div className="input-group setSearch">
                          <input
                            onChange={SearchData}
                            type="text"
                            className="form-control searchData"
                            placeholder="Search..."
                          />
                          <div className="input-group-append">
                            <button
                              className="btn btn-sm btn-primary"
                              type="button"
                            >
                              <i className="icon-search"></i>
                            </button>
                          </div>
                        </div>
                      </div>
                    </Col>
                    <Col md={5}>
                      <div className="import_buttons content1">
                        <button
                          className="text-big add-user"
                          type="button"
                          onClick={() => {
                            setShowREgModal(true);
                            getAllRetreats();
                            clearfield();
                          }}
                        >
                          <i className="mdi mdi-account-plus text-white"></i>
                        </button>
                      </div>
                    </Col>
                  </Row>
                  <CustomDataTable
                    loading={loadData}
                    data={allCommittee}
                    columns={columns}
                    total={total}
                    countPerPage={20}
                    isServerSide={true}
                    pagination={true}
                    sortServer={true}
                    dataFetchApi={allCommitteMembers}
                    pointerOnHover={true}
                    highlightOnHover={true}
                  />
                </div>
              </div>
            </div>
          </div>
        </div>

        <Modal
          show={showRegModal}
          onHide={() => {
            setShowREgModal(false);
            clearfield();
          }}
          size="lg"
        >
          <Modal.Header closeButton>
            <Modal.Title>Add Committee Member</Modal.Title>
          </Modal.Header>
          <Form
            onSubmit={handleSubmit}
            noValidate
            validated={validated}
            className="forms-sample"
          >
            <Modal.Body>
              <p className="card-description">
                Enter Committee Member details here..
              </p>
              <hr />
              <Row>
                <Col lg={6} md={6} sm={6} xs={12}>
                  <Form.Group>
                    <Form.Label>First Name</Form.Label>
                    <Form.Control
                      required
                      size="sm"
                      type="text"
                      name="firstName"
                      placeholder="First Name"
                      value={data.firstName}
                      onChange={handleChanges}
                    />
                    <Form.Control.Feedback type="invalid">
                      First Name is required!
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
                <Col lg={6} md={6} sm={6} xs={12}>
                  <Form.Group>
                    <Form.Label>Last Name</Form.Label>
                    <Form.Control
                      required
                      size="sm"
                      type="text"
                      name="lastName"
                      placeholder="Last Name"
                      value={data.lastName}
                      onChange={handleChanges}
                    />
                    <Form.Control.Feedback type="invalid">
                      Last Name is required!
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
                <Col lg={6} md={6} sm={6} xs={12} className="mt-4">
                  <Form.Group>
                    <Form.Label>Email</Form.Label>
                    <Form.Control
                      required
                      size="sm"
                      type="email"
                      name="eMail"
                      placeholder="Email"
                      value={data.eMail}
                      onChange={handleChanges}
                    />
                    <Form.Control.Feedback type="invalid">
                      Email is required!
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
                <Col lg={6} md={6} sm={6} xs={12} className="mt-4">
                  <Form.Group>
                    <Form.Label>Mobile</Form.Label>
                    <Form.Control
                      required
                      size="sm"
                      type="text"
                      name="mobile"
                      placeholder="Mobile"
                      value={data.mobile}
                      onChange={handleChanges}
                    />
                    <Form.Control.Feedback type="invalid">
                      Mobile is required!
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
                <Col lg={12} className="form-group bold mt-4">
                  <Form.Label>Retreats</Form.Label>
                  <Select
                    id="retreat"
                    name="retreat"
                    options={retreatOptions}
                    value={selectedRetreats}
                    onChange={handleSelectRetreat}
                    isSearchable
                    isMulti
                  />
                </Col>
              </Row>
              <hr />
            </Modal.Body>
            <Modal.Footer>
              {userData?.userRole !== "SUBADMIN" && (
                <Button variant="primary" type="submit">
                  Submit
                </Button>
              )}
              <Button
                variant="secondary"
                onClick={() => {
                  setShowREgModal(false);
                  clearfield();
                }}
              >
                Cancel
              </Button>
            </Modal.Footer>
          </Form>
        </Modal>

        {show ? (
          <DeleteModel
            show={show}
            handleClose={handleClose}
            DeleteData={deleteMember}
          ></DeleteModel>
        ) : null}
      </div>
    </>
  );
};

export default Committee;
