const db = require("../database/dbConnection");
const coupleSchema = db.couple;
const parishSchema = db.parish;
const serviceSchema = db.service;
const roleSchema = db.roles;
const talkSchema = db.talk;
const retreatRoastSchema = db.retreatRoaster;
const { Op, Sequelize, NUMBER } = require("sequelize");
const { PROJECT_DIR } = require("../../config");
const { SENDLOGINDETAILBYADMIN } = require("../helper/emailTemplate");
const userSchema = db.user;
const Email = require("../helper/sendEmail");
const fs = require("fs");
const path = require("path");

class RetreatRoasterController {
  async getRetreatRoastData(req, res) {
    try {
      const {
        searchKeyword,
        sortBy,
        pageNo,
        size,
        retreatId,
        export: exportData,
      } = req.body;

      // Default pagination setup
      let query = { skip: 0, limit: 10 }; // Default limit is 10 if size is not provided
      if (pageNo && size) {
        const page = parseInt(pageNo);
        const limit = parseInt(size);
        if (page < 1) {
          return res.status(400).send({
            error: true,
            msg: "Invalid page number, should start with 1",
          });
        }
        query.skip = limit * (page - 1);
        query.limit = limit;
      }

      // Sort setup
      let sortDirection = "DESC";
      let sortColumn = "couples.primaryKey"; // Default sort column
      if (sortBy) {
        // Set sort direction based on the provided value (1 for ASC, otherwise DESC)
        sortDirection = sortBy.sortDirection == 1 ? "ASC" : "DESC";

        // Normalize sortColumn by removing spaces and converting to lowercase
        const normalizedSortColumn = sortBy.sortColumn
          .toLowerCase()
          .replace(/\s+/g, "");

        // Switch case to handle sorting by the normalized column
        switch (normalizedSortColumn) {
          case "hisfirstname":
            sortColumn = "couples.hisFirstName";
            break;

          case "hislastname":
            sortColumn = "couples.hisLastName";
            break;

          case "herfirstname":
            sortColumn = "couples.herFirstName";
            break;

          case "herlastname":
            sortColumn = "couples.herLastName";
            break;

          case "parish":
            sortColumn = "parishes.parish";
            break;

          case "role":
            sortColumn = "roles.role";
            break;

          case "service":
            sortColumn = "services.service";
            break;

          case "talk":
            sortColumn = "talks.talk";
            break;

          case "hisemail":
            sortColumn = "couples.hisEmail";
            break;

          case "heremail":
            sortColumn = "couples.herEmail";
            break;

          case "hismobile":
            sortColumn = "couples.hisMobile";
            break;

          case "hermobile":
            sortColumn = "couples.herMobile";
            break;

          case "primarykey":
            sortColumn = "couples.primaryKey";
            break;

          case "retreatant":
            sortColumn = "retreatroasters.attendeeType";
            break;

          case "address":
            sortColumn = "couples.address";
            break;

          case "allergies":
            sortColumn = "couples.allergies";
            break;

          case "anniversarydate":
            sortColumn = "couples.anniversary_date";
            break;

          case "created":
            sortColumn = "retreatroasters.createdAt";
            break;
          case "type":
            sortColumn = "retreatroasters.attendeeType";
            break;

          /*case "underage35":
            sortColumn = "couples.under_age_35";
            break;

          case "hearaboutus":
            sortColumn = "couples.hear_us";
            break;

          case "referralfromafriendorfamilymember":
            sortColumn = "couples.referral_from_a_friend_or_family_member";
            break;

          case "emergencyname1":
            sortColumn = "couples.emergency_name1";
            break;

          case "emergencyname2":
            sortColumn = "couples.emergency_name2";
            break;

          case "emergencyrelationship1":
            sortColumn = "couples.emergency_relationship1";
            break;

          case "emergencyrelationship2":
            sortColumn = "couples.emergency_relationship2";
            break;

          case "emergencyphone1":
            sortColumn = "couples.emergency_phone1";
            break;

          case "emergencyphone2":
            sortColumn = "couples.emergency_phone2";
            break;*/

          default:
            // Default sort column if none of the above matches
            sortColumn = "couples.primaryKey";
            break;
        }
      }

      // Search setup
      let searchQuery = "";
      if (searchKeyword && searchKeyword !== "") {
        searchQuery = `(
          couples.hisFirstName LIKE :searchKeyword OR
          couples.hisLastName LIKE :searchKeyword OR
          couples.herFirstName LIKE :searchKeyword OR
          couples.herLastName LIKE :searchKeyword OR
          couples.hisEmail LIKE :searchKeyword OR
          couples.herEmail LIKE :searchKeyword OR
          parishes.parish LIKE :searchKeyword OR
          roles.role LIKE :searchKeyword OR
          services.service LIKE :searchKeyword OR
          couples.primaryKey LIKE :searchKeyword OR
          retreatroasters.attendeeType LIKE :searchKeyword OR
          couples.address LIKE :searchKeyword OR
          couples.allergies LIKE :searchKeyword OR
          couples.anniversary_date LIKE :searchKeyword OR
          CONCAT(couples.hisFirstName, ' ', couples.hisLastName) LIKE :searchKeyword OR
          CONCAT(couples.herFirstName, ' ', couples.herLastName) LIKE :searchKeyword
        )`;
        /*couples.under_age_35 LIKE :searchKeyword OR
          couples.hear_us LIKE :searchKeyword OR
          couples.referral_from_a_friend_or_family_member LIKE :searchKeyword OR
          couples.emergency_name1 LIKE :searchKeyword OR
          couples.emergency_name2 LIKE :searchKeyword OR
          couples.emergency_relationship1 LIKE :searchKeyword OR
          couples.emergency_relationship2 LIKE :searchKeyword OR
          couples.emergency_phone1 LIKE :searchKeyword OR
          couples.emergency_phone2 LIKE :searchKeyword OR*/ // Removed these from search criteria
      }

      // Base query
      let baseQuery = `
        SELECT 
          retreats.id AS retreatId,
          retreats.freeze,
          retreatroasters.id,
          retreatroasters.coupleId,
          retreatroasters.attendeeType,
          retreatroasters.createdAt,
          couples.hisFirstName,
          couples.primaryKey,
          couples.hisLastName,
          couples.herFirstName,
          couples.herLastName,
          couples.hisEmail,
          couples.primaryKey,
          couples.herEmail,
          couples.hisMobile,
          couples.herMobile,
          couples.city,
          couples.state,
          couples.zip,
          couples.notes,
          couples.imageUrl,
          couples.address,
          couples.allergies,
          couples.anniversary_date,
          couples.under_age_35,
          couples.hear_us,
          couples.referral_from_a_friend_or_family_member,
          couples.emergency_name1,
          couples.emergency_name2,
          couples.emergency_relationship1,
          couples.emergency_relationship2,
          couples.emergency_phone1,
          couples.emergency_phone2,
          parishes.parish,
          roles.role,
          services.service,
talks.talk,couples.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId
        FROM retreatroasters
        LEFT JOIN couples ON retreatroasters.coupleId = couples.id
        LEFT JOIN parishes ON couples.parishId = parishes.id
        LEFT JOIN roles ON retreatroasters.roleId = roles.id
        LEFT JOIN services ON retreatroasters.serviceId = services.id
        LEFT JOIN talks ON retreatroasters.talkId = talks.id
        INNER JOIN retreats ON retreatroasters.retreatId = retreats.id
        WHERE retreatroasters.retreatId = :retreatId AND retreatroasters.stage = 'VERIFIED'
      `;

      // Add search condition if present
      if (searchQuery) {
        baseQuery += ` AND ${searchQuery}`;
      }

      // Add sort and group by
      baseQuery += ` GROUP BY retreatroasters.coupleId ORDER BY ${sortColumn} ${sortDirection}`;

      // Add pagination if not exporting
      if (!exportData) {
        baseQuery += ` LIMIT :limit OFFSET :offset`;
      }

      // Total count query
      const totalCountQuery = `
        SELECT COUNT(DISTINCT retreatroasters.coupleId) AS total
        FROM retreatroasters
        WHERE retreatroasters.retreatId = :retreatId AND retreatroasters.stage = 'VERIFIED'
      `;

      // Query to get service and talk counts
      const servTalkTeamTotalQuery = `
        SELECT
          SUM(CASE WHEN retreatroasters.attendeeType = 'Team' THEN 1 ELSE 0 END) AS countTeam,
          SUM(CASE WHEN retreatroasters.attendeeType = 'Retreatant' THEN 1 ELSE 0 END) AS countret,
          COUNT(retreatroasters.serviceId) AS serviceTotal,
          COUNT(retreatroasters.talkId) AS talksTotal
        FROM retreatroasters
        WHERE retreatroasters.retreatId = :retreatId AND retreatroasters.stage = 'VERIFIED'`;
      // Execute the queries in parallel
      const [retreatData, totalRecords, servTalkTeamTotal] = await Promise.all([
        db.sequelize.query(baseQuery, {
          replacements: {
            retreatId,
            searchKeyword: `%${searchKeyword}%`,
            limit: query.limit,
            offset: query.skip,
          },
          type: db.Sequelize.QueryTypes.SELECT,
        }),
        db.sequelize.query(totalCountQuery, {
          replacements: { retreatId },
          type: db.Sequelize.QueryTypes.SELECT,
        }),
        db.sequelize.query(servTalkTeamTotalQuery, {
          replacements: { retreatId },
          type: db.Sequelize.QueryTypes.SELECT,
        }),
      ]);
      let totalTeamCount = `SELECT *  FROM retreatroasters WHERE retreatroasters.retreatId = ${retreatId} AND retreatroasters.stage = 'VERIFIED' AND retreatroasters.attendeeType = 'Team' GROUP BY retreatroasters.coupleId`;
      let teamCount = await db.sequelize
        .query(totalTeamCount)
        .then(([results, metadata]) => results);
      let totaRetCount = `SELECT *  FROM retreatroasters WHERE retreatroasters.retreatId = ${retreatId} AND retreatroasters.stage = 'VERIFIED' AND retreatroasters.attendeeType = 'Retreatant' GROUP BY retreatroasters.coupleId`;
      let retCount = await db.sequelize
        .query(totaRetCount)
        .then(([results, metadata]) => results);
      // Send response with data
      return res.status(200).send({
        success: true,
        data: retreatData,
        totalRecords: totalRecords[0].total,
        servTalkTeamTotal: {
          serviceTotal: servTalkTeamTotal[0].serviceTotal,
          talksTotal: servTalkTeamTotal[0].talksTotal,
          countTeam: teamCount.length,
          countret: retCount.length,
        },
      });
    } catch (error) {
      console.error("Error in getRetreatRoastData:", error);
      return res.status(500).send({
        error: true,
        msg: "Something went wrong. Please try again.",
      });
    }
  }
  async getSpritualRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.imageUrl,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.city,couples.state,couples.notes,couples.zip,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member, retreatroasters.coupleId,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON couples.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND roles.role = "Spiritual Mentor" AND retreatroasters.stage = "VERIFIED"  GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again",
      });
    }
  }

  async getTeamRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.coupleId,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.imageUrl,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.city,couples.state,couples.notes,couples.zip,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON couples.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND retreatroasters.attendeeType = "Team" AND retreatroasters.stage = "VERIFIED" GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getLeadCoupleRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.coupleId,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.imageUrl,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.city,couples.state,couples.notes,couples.zip,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON couples.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND roles.role = "Lead" AND retreatroasters.stage = "VERIFIED"  GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getServiceRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.coupleId,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.imageUrl,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON retreatroasters.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND services.service != "NULL" AND retreatroasters.stage = "VERIFIED"  GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async gettalkRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.coupleId,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.imageUrl,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON retreatroasters.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND talks.talk != "NULL" AND retreatroasters.stage = "VERIFIED"  GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getRetreatantRoastData(req, res) {
    try {
      if (req.body.retreatId) {
        let select = `SELECT retreats.freeze,retreatroasters.id,retreatroasters.coupleId,retreatroasters.attendeeType,retreatroasters.createdAt,couples.primaryKey,couples.hisLastName,couples.hisFirstName,couples.herFirstName,couples.herLastName,couples.city,couples.imageUrl,couples.state,couples.notes,couples.zip,couples.address,couples.allergies,couples.anniversary_date,couples.hear_us,couples.emergency_name1,couples.	emergency_relationship1,couples.emergency_phone1,couples.emergency_name2,couples.emergency_relationship2,couples.emergency_phone2,couples.under_age_35,couples.referral_from_a_friend_or_family_member,parishes.parish,roles.role,services.service,talks.talk,couples.hisEmail,couples.herEmail,couples.hisMobile,couples.herMobile,retreatroasters.parishId,retreatroasters.roleId,retreatroasters.serviceId,retreatroasters.talkId`;
        let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN parishes ON couples.parishId = parishes.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE retreatroasters.retreatId = ${req.body.retreatId} AND retreatroasters.attendeeType = "Retreatant" AND retreatroasters.stage = "VERIFIED" GROUP BY retreatroasters.coupleId`;
        let queryRecords = select + from;
        let responseData1 = await db.sequelize
          .query(queryRecords)
          .then(([results, metadata]) => results);
        return res.status(200).send({
          success: true,
          data: responseData1,
          totalRecords: responseData1.length,
        });
      } else {
        return res.status(200).send({
          success: false,
          msg: "Retreat Id is undefined",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async deleteRetreatRoast(req, res) {
    try {
      let { id } = req.body;
      if (id) {
        let deleteretreatRoast = await retreatRoastSchema.destroy({
          where: { id: id },
        });
        return res.status(200).send({
          success: true,
          msg: "Retreat deleted successfully.",
        });
      } else {
        return res.status(200).json({
          success: false,
          msg: "Error in ID.",
        });
      }
    } catch (error) {
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async saveRetreatRoasterExcel(req, res) {
    try {
      let data = req.body;
      let createdCouples = [],
        updatedCouples = [],
        createdRoasters = [],
        updatedRoasters = [];

      const normalize = (num) => (num ? String(num).replace(/\D/g, "") : "");

      const isValidDate = (date) => {
        if (!date) return false;
        const parsed = new Date(date);
        return !isNaN(parsed);
      };

      const parseAnniversaryDate = (dateStr) => {
        if (!dateStr) return null;
        const parts = dateStr.split("-");
        if (parts.length === 2) {
          const year = new Date().getFullYear();
          return `${year}-${parts[0].padStart(2, "0")}-${parts[1].padStart(
            2,
            "0"
          )}`;
        }
        return isValidDate(dateStr) ? dateStr : null;
      };

      const fixUnderAge35 = (val) => {
        if (!val || val === "") return "NO";
        return val;
      };

      for (const row of data) {
        // ✅ Skip completely empty rows
        if (
          !row.hisFirstName &&
          !row.herFirstName &&
          !row.hisEmail &&
          !row.herEmail &&
          !row.hisMobile &&
          !row.herMobile
        ) {
          console.log("Skipped empty row:", row);
          continue;
        }

        // ✅ Normalize mobiles
        row.hisMobile = normalize(row.hisMobile);
        row.herMobile = normalize(row.herMobile);

        // ✅ Set empty emails/mobiles to NULL
        row.hisEmail =
          row.hisEmail && row.hisEmail.trim() !== "" ? row.hisEmail : null;
        row.herEmail =
          row.herEmail && row.herEmail.trim() !== "" ? row.herEmail : null;
        row.hisMobile = row.hisMobile !== "" ? row.hisMobile : null;
        row.herMobile = row.herMobile !== "" ? row.herMobile : null;

        // ✅ Auto-assign primaryKey if missing
        if (!row.primaryKey || row.primaryKey === "") {
          const maxCouple = await coupleSchema.findOne({
            order: [["primaryKey", "DESC"]],
            attributes: ["primaryKey"],
            raw: true,
          });
          let nextPrimaryKey =
            maxCouple && maxCouple.primaryKey
              ? Number(maxCouple.primaryKey) + 1
              : 1;
          row.primaryKey = String(nextPrimaryKey);
        }

        // ✅ Dynamic Duplicate Check
        const coupleWhere = [];
        if (row.hisEmail) {
          coupleWhere.push(
            { hisEmail: row.hisEmail },
            { herEmail: row.hisEmail }
          );
        }
        if (row.herEmail) {
          coupleWhere.push(
            { hisEmail: row.herEmail },
            { herEmail: row.herEmail }
          );
        }
        if (row.hisMobile) {
          coupleWhere.push(
            { hisMobile: row.hisMobile },
            { herMobile: row.hisMobile }
          );
        }
        if (row.herMobile) {
          coupleWhere.push(
            { hisMobile: row.herMobile },
            { herMobile: row.herMobile }
          );
        }

        let existingCouple = null;
        if (coupleWhere.length > 0) {
          existingCouple = await coupleSchema.findOne({
            where: { [Op.or]: coupleWhere },
            raw: true,
          });
        }

        let coupleId;
        const coupleDataToUpdate = {
          primaryKey: row.primaryKey,
          hisFirstName: row.hisFirstName,
          hisLastName: row.hisLastName,
          hisEmail: row.hisEmail,
          hisMobile: row.hisMobile,
          herFirstName: row.herFirstName,
          herLastName: row.herLastName,
          herEmail: row.herEmail,
          herMobile: row.herMobile,
          address: row.address,
          city: row.city,
          state: row.state,
          zip: row.zip,
          anniversary_date: parseAnniversaryDate(row.anniversary_date),
          allergies: row.allergies,
          under_age_35: fixUnderAge35(row.under_age_35),
          emergency_name1: row.emergency_name1,
          emergency_relationship1: row.emergency_relationship1,
          emergency_phone1: row.emergency_phone1,
          emergency_name2: row.emergency_name2,
          emergency_relationship2: row.emergency_relationship2,
          emergency_phone2: row.emergency_phone2,
          hear_us: Array.isArray(row.hear_us)
            ? row.hear_us.join(", ")
            : row.hear_us,
          referral_from_a_friend_or_family_member:
            row.referral_from_a_friend_or_family_member,
          notes: row.notes,
          parishId: row.parishId || null,
          imageUrl: null,
        };

        if (existingCouple) {
          coupleId = existingCouple.id;
          await coupleSchema.update(coupleDataToUpdate, {
            where: { id: coupleId },
          });
          updatedCouples.push(coupleId);
        } else {
          const newCouple = await coupleSchema.create(coupleDataToUpdate);
          coupleId = newCouple.id;
          createdCouples.push(coupleId);
        }

        // ✅ User creation & update
        const users = [
          {
            userName: row.hisFirstName,
            email: row.hisEmail,
            firstName: row.hisFirstName,
            lastName: row.hisLastName,
            mobile: row.hisMobile,
            accountStatus: "ACTIVE",
            userRole: "COUPLE",
            coupleId,
          },
          {
            userName: row.herFirstName,
            email: row.herEmail,
            firstName: row.herFirstName,
            lastName: row.herLastName,
            mobile: row.herMobile,
            accountStatus: "ACTIVE",
            userRole: "COUPLE",
            coupleId,
          },
        ];

        for (const user of users) {
          if (user.email) {
            const existUser = await userSchema.findOne({
              where: { email: user.email },
            });
            if (existUser) {
              await userSchema.update(user, { where: { email: user.email } });
            } else {
              let randompass = Math.random().toString(36).slice(-8);
              user.password = "p" + randompass.slice(1);
              await userSchema.create(user);
              const emailBody = SENDLOGINDETAILBYADMIN.replace(
                /#firstName#/g,
                user.firstName
              )
                .replace(/#lastName#/g, user.lastName)
                .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
                .replace(/#email#/g, user.email)
                .replace(/#password#/g, user.password);
              await Email.sendEmail(
                user.email,
                "Couple Login Details - LOVESTRONG Marriage",
                emailBody
              );
            }
          }
        }

        // ✅ Roaster Insert/Update
        const existRoaster = await retreatRoastSchema.findOne({
          where: { retreatId: row.retreatId, coupleId },
          raw: true,
        });

        const roasterData = {
          retreatId: row.retreatId,
          coupleId,
          roleId: row.roleId || null,
          serviceId: row.serviceId || null,
          talkId: row.talkId || null,
          attendeeType: row.attendeeType,
          stage: "VERIFIED",
        };

        if (existRoaster) {
          await retreatRoastSchema.update(roasterData, {
            where: { id: existRoaster.id },
          });
          updatedRoasters.push(existRoaster.id);
        } else {
          const newRoaster = await retreatRoastSchema.create(roasterData);
          createdRoasters.push(newRoaster.id);
        }
      }

      return res.send({
        success: true,
        createdCouples,
        updatedCouples,
        createdRoasters,
        updatedRoasters,
        msg: "Retreat Roster data imported successfully.",
      });
    } catch (err) {
      console.error(err);
      return res.send({
        success: false,
        msg: "Error importing Retreat Roster data.",
      });
    }
  }

  // async saveRetreatRoasterExcel(req, res) {
  //   try {
  //     let data = req.body;
  //     let createdCouples = [], updatedCouples = [], createdRoasters = [], updatedRoasters = [];
  //     const normalize = (num) => num ? String(num).replace(/\D/g, '') : '';

  //     const isValidDate = (date) => {
  //       if (!date) return false;
  //       const parsed = new Date(date);
  //       return !isNaN(parsed);
  //     };

  //     const parseAnniversaryDate = (dateStr) => {
  //       if (!dateStr) return null;
  //       const parts = dateStr.split('-');
  //       if (parts.length === 2) {
  //         const year = new Date().getFullYear();
  //         return `${year}-${parts[0].padStart(2, '0')}-${parts[1].padStart(2, '0')}`;
  //       }
  //       return isValidDate(dateStr) ? dateStr : null;
  //     };

  //     for (const row of data) {
  //       // Auto-assign primaryKey if missing
  //       if (!row.primaryKey || row.primaryKey === '') {
  //         const maxCouple = await coupleSchema.findOne({
  //           order: [['primaryKey', 'DESC']],
  //           attributes: ['primaryKey'],
  //           raw: true,
  //         });
  //         let nextPrimaryKey = maxCouple && maxCouple.primaryKey ? Number(maxCouple.primaryKey) + 1 : 1;
  //         row.primaryKey = String(nextPrimaryKey);
  //       }

  //       let existingCouple = await coupleSchema.findOne({
  //         where: {
  //           [Op.or]: [
  //             { hisEmail: row.hisEmail },
  //             { herEmail: row.hisEmail },
  //             { hisEmail: row.herEmail },
  //             { herEmail: row.herEmail },
  //             { hisMobile: normalize(row.hisMobile) },
  //             { herMobile: normalize(row.hisMobile) },
  //             { hisMobile: normalize(row.herMobile) },
  //             { herMobile: normalize(row.herMobile) },
  //           ]
  //         },
  //         raw: true,
  //       });

  //       let coupleId;

  //       const coupleDataToUpdate = {
  //         primaryKey: row.primaryKey,
  //         hisFirstName: row.hisFirstName,
  //         hisLastName: row.hisLastName,
  //         hisEmail: row.hisEmail,
  //         hisMobile: row.hisMobile,
  //         herFirstName: row.herFirstName,
  //         herLastName: row.herLastName,
  //         herEmail: row.herEmail,
  //         herMobile: row.herMobile,
  //         address: row.address,
  //         city: row.city,
  //         state: row.state,
  //         zip: row.zip,
  //         anniversary_date: parseAnniversaryDate(row.anniversary_date),
  //         allergies: row.allergies,
  //         under_age_35: row.under_age_35,
  //         emergency_name1: row.emergency_name1,
  //         emergency_relationship1: row.emergency_relationship1,
  //         emergency_phone1: row.emergency_phone1,
  //         emergency_name2: row.emergency_name2,
  //         emergency_relationship2: row.emergency_relationship2,
  //         emergency_phone2: row.emergency_phone2,
  //         hear_us: Array.isArray(row.hear_us) ? row.hear_us.join(', ') : row.hear_us,
  //         referral_from_a_friend_or_family_member: row.referral_from_a_friend_or_family_member,
  //         notes: row.notes,
  //         parishId: row.parishId || null
  //       };

  //       if (existingCouple) {
  //         coupleId = existingCouple.id;
  //         await coupleSchema.update(coupleDataToUpdate, { where: { id: coupleId } });
  //         updatedCouples.push(coupleId);
  //       } else {
  //         const newCouple = await coupleSchema.create(coupleDataToUpdate);
  //         coupleId = newCouple.id;
  //         createdCouples.push(coupleId);
  //       }

  //       const users = [
  //         {
  //           userName: row.hisFirstName,
  //           email: row.hisEmail,
  //           firstName: row.hisFirstName,
  //           lastName: row.hisLastName,
  //           mobile: row.hisMobile,
  //           accountStatus: 'ACTIVE',
  //           userRole: 'COUPLE',
  //           coupleId,
  //         },
  //         {
  //           userName: row.herFirstName,
  //           email: row.herEmail,
  //           firstName: row.herFirstName,
  //           lastName: row.herLastName,
  //           mobile: row.herMobile,
  //           accountStatus: 'ACTIVE',
  //           userRole: 'COUPLE',
  //           coupleId,
  //         }
  //       ];

  //       for (const user of users) {
  //         if (user.email) {
  //           const existUser = await userSchema.findOne({ where: { email: user.email } });
  //           if (existUser) {
  //             await userSchema.update(user, { where: { email: user.email } });
  //           } else {
  //             let randompass = Math.random().toString(36).slice(-8);
  //             user.password = 'p' + randompass.slice(1);
  //             await userSchema.create(user);
  //             // Optional: Send email with login details
  //           }
  //         }
  //       }

  //       const existRoaster = await retreatRoastSchema.findOne({
  //         where: { retreatId: row.retreatId, coupleId },
  //         raw: true
  //       });

  //       const roasterData = {
  //         retreatId: row.retreatId,
  //         coupleId,
  //         roleId: row.roleId || null,
  //         serviceId: row.serviceId || null,
  //         talkId: row.talkId || null,
  //         attendeeType: row.attendeeType,
  //         stage: 'VERIFIED'
  //       };

  //       if (existRoaster) {
  //         await retreatRoastSchema.update(roasterData, { where: { id: existRoaster.id } });
  //         updatedRoasters.push(existRoaster.id);
  //       } else {
  //         const newRoaster = await retreatRoastSchema.create(roasterData);
  //         createdRoasters.push(newRoaster.id);
  //       }
  //     }

  //     return res.send({
  //       success: true,
  //       createdCouples,
  //       updatedCouples,
  //       createdRoasters,
  //       updatedRoasters,
  //       msg: 'Retreat Roaster data imported successfully.'
  //     });

  //   } catch (err) {
  //     console.error(err);
  //     return res.send({ success: false, msg: 'Error importing Retreat Roaster data.' });
  //   }
  // }

  // async saveRetreatRoasterExcel(req, res) {
  //   try {
  //     let retreatRoastData = req.body;
  //     console.log("------->>",retreatRoastData);

  //     var updateRoaster = [];
  //     var createdRoasters = [];
  //     var updateCouple = [];
  //     var createdCouple = [];
  //     var paswordForCouple1 = "";
  //     var paswordForCouple2 = "";
  //     var adminObj = [];
  //     for (let i = 0; i < retreatRoastData.length; i++) {
  //       retreatRoastData[i].hisMobile =
  //         retreatRoastData[i].hisMobile.toString();
  //       retreatRoastData[i].herMobile =
  //         retreatRoastData[i].herMobile.toString();
  //       retreatRoastData[i].emergency_phone1 =
  //         retreatRoastData[i].emergency_phone1.toString();
  //       retreatRoastData[i].emergency_phone2 =
  //         retreatRoastData[i].emergency_phone2.toString();
  //       let formatObj = {};
  //       let coupleNotExist = [];
  //       let coupleExist = [];
  //       let parishId = null;
  //       let newP = null;
  //       let oldP = null;
  //       var roleId = "";
  //       var serviceId = "";
  //       var talkId = "";
  //       let maxPrimaryKeyDB = await coupleSchema.max("primaryKey");
  //       if (maxPrimaryKeyDB == null) {
  //         maxPrimaryKeyDB = 1000;
  //       }
  //       // role
  //       if (retreatRoastData[i].role) {
  //         let roleId = await roleSchema.findOne({
  //           where: { role: retreatRoastData[i].role },
  //           raw: true,
  //         });
  //         if (roleId) {
  //           formatObj.roleId = roleId.id;
  //         }
  //       }
  //       // service
  //       if (retreatRoastData[i].service) {
  //         let serviceId = await serviceSchema.findOne({
  //           where: { service: retreatRoastData[i].service },
  //           raw: true,
  //         });
  //         if (serviceId) {
  //           formatObj.serviceId = serviceId.id;
  //         }
  //       }
  //       // talk
  //       if (retreatRoastData[i].talk) {
  //         let talkId = await talkSchema.findOne({
  //           where: { talk: retreatRoastData[i].talk },
  //           raw: true,
  //         });
  //         if (talkId) {
  //           formatObj.talkId = talkId.id;
  //         }
  //       }
  //       // parish
  //       if (retreatRoastData[i].parish) {
  //         parishId = await parishSchema.findOne({
  //           where: { parish: retreatRoastData[i].parish },
  //           raw: true,
  //         });
  //       }
  //       if (
  //         retreatRoastData[i].hisEmail === "" &&
  //         retreatRoastData[i].herEmail === "" &&
  //         retreatRoastData[i].herMobile === "" &&
  //         retreatRoastData[i].hisMobile === ""
  //       ) {
  //         let f = await coupleSchema.findAll({
  //           where: {
  //             primaryKey: retreatRoastData[i].primaryKey,
  //           },
  //           raw: true,
  //         });
  //         if (f.length > 0) {
  //           let foundInRoaster = await retreatRoastSchema.findOne({
  //             where: {
  //               [Op.and]: [
  //                 { coupleId: f[0].id },
  //                 { retreatId: retreatRoastData[i].retreatId },
  //               ],
  //             },
  //             raw: true,
  //           });
  //           if (foundInRoaster && foundInRoaster !== null) {
  //             let retreatToUpdate = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //             };

  //             let updatedData = await retreatRoastSchema.findAll({
  //               where: {
  //                 [Op.and]: [
  //                   { id: foundInRoaster.id },
  //                   { retreatId: retreatRoastData[i].retreatId },
  //                   { coupleId: f[0].id },
  //                 ],
  //               },
  //               raw: true,
  //             });
  //             if (updatedData[0].roleId) {
  //               roleId = await roleSchema.findOne({
  //                 where: { id: updatedData[0].roleId },
  //                 raw: true,
  //               });
  //             }

  //             // service
  //             if (updatedData[0].serviceId) {
  //               serviceId = await serviceSchema.findOne({
  //                 where: { id: updatedData[0].serviceId },
  //                 raw: true,
  //               });
  //             }

  //             // talk
  //             if (updatedData[0].talkId) {
  //               talkId = await talkSchema.findOne({
  //                 where: { id: updatedData[0].talkId },
  //                 raw: true,
  //               });
  //             }

  //             let updateOldRec = await retreatRoastSchema.update(
  //               retreatToUpdate,
  //               {
  //                 where: {
  //                   [Op.and]: [
  //                     { id: foundInRoaster.id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                     { coupleId: f[0].id },
  //                   ],
  //                 },
  //               }
  //             );

  //             // Check if any rows were affected by the update
  //             if (updateOldRec[0] > 0) {
  //               updateRoaster.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 oldRole: roleId ? roleId.role : "",
  //                 oldService: serviceId ? serviceId.service : "",
  //                 oldTalk: talkId ? talkId.talk : "",
  //                 oldAttendeeType: updatedData && updatedData[0].attendeeType,

  //                 newRole: retreatRoastData[i].role,
  //                 newService: retreatRoastData[i].service,
  //                 newTalk: retreatRoastData[i].talk,
  //                 newAttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             } else {
  //               console.log("No rows were updated.");
  //             }
  //           } else {
  //             let retreatData = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //               coupleId: f[0].id,
  //               retreatId: retreatRoastData[i].retreatId,
  //             };
  //             let newUser = new retreatRoastSchema(retreatData);
  //             let succ = await newUser.save();
  //             createdRoasters.push({
  //               couple:
  //                 `Couple : (${
  //                   retreatRoastData[i].primaryKey
  //                     ? `${retreatRoastData[i].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                     ? `${retreatRoastData[i].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                     : ""
  //                 })`,
  //               Role: retreatRoastData[i].role,
  //               Service: retreatRoastData[i].service,
  //               Talk: retreatRoastData[i].talk,
  //               Parish: retreatRoastData[i].parish,
  //               AttendeeType: retreatRoastData[i].attendeeType,
  //             });
  //           }
  //           coupleExist.push({
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,

  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });

  //           let oldData = await coupleSchema.findAll({
  //             where: { primaryKey: retreatRoastData[i].primaryKey },
  //             raw: true,
  //           });
  //           if (oldData[0].parishId) {
  //             oldP = await parishSchema.findOne({
  //               where: { id: oldData[0].parishId },
  //               raw: true,
  //             });
  //           }
  //           let update = await coupleSchema.update(coupleExist[0], {
  //             where: { primaryKey: retreatRoastData[i].primaryKey },
  //           });
  //           if (update[0] > 0) {
  //             // If rows were updated, fetch the updated data
  //             let updatedData = await coupleSchema.findAll({
  //               where: { primaryKey: retreatRoastData[i].primaryKey },
  //               raw: true,
  //             });
  //             if (updatedData[0].parishId) {
  //               newP = await parishSchema.findOne({
  //                 where: { id: updatedData[0].parishId },
  //                 raw: true,
  //               });
  //             }
  //             updateCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   updatedData[0].primaryKey
  //                     ? `${updatedData[0].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisLastName
  //                     ? `${updatedData[0].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisFirstName
  //                     ? `, ${updatedData[0].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].herFirstName
  //                     ? ` and  ${updatedData[0].herFirstName}`
  //                     : ""
  //                 })`,

  //               oldhisFirstName: oldData[0].hisFirstName,
  //               oldhisLastName: oldData[0].hisLastName,
  //               oldhisEmail: oldData[0].hisEmail,
  //               oldhisMobile: oldData[0].hisMobile,
  //               oldherFirstName: oldData[0].herFirstName,
  //               oldherLastName: oldData[0].herLastName,
  //               oldherEmail: oldData[0].herEmail,
  //               oldherMobile: oldData[0].herMobile,
  //               oldcity: oldData[0].city,
  //               oldstate: oldData[0].state,
  //               oldzip: oldData[0].zip,
  //               oldnotes: oldData[0].notes,
  //               oldparish: oldP && oldP.parish ? oldP.parish : "",
  //               oldaddress: oldData[0].address,
  //               oldallergies: oldData[0].allergies,
  //               oldanniversary_date: oldData[0].anniversary_date,
  //               oldhear_us: oldData[0].hear_us,
  //               oldemergency_name1: oldData[0].emergency_name1,
  //               oldemergency_relationship1: oldData[0].emergency_relationship1,
  //               oldemergency_phone1: oldData[0].emergency_phone1,
  //               oldemergency_name2: oldData[0].emergency_name2,
  //               oldemergency_relationship2: oldData[0].emergency_relationship2,
  //               oldemergency_phone2: oldData[0].emergency_phone2,
  //               oldunder_age_35: oldData[0].under_age_35,
  //               oldreferral_from_a_friend_or_family_member:
  //                 oldData[0].referral_from_a_friend_or_family_member,
  //               newhisFirstName: updatedData[0].hisFirstName,
  //               newhisLastName: updatedData[0].hisLastName,
  //               newhisEmail: updatedData[0].hisEmail,
  //               newhisMobile: updatedData[0].hisMobile,
  //               newherFirstName: updatedData[0].herFirstName,
  //               newherLastName: updatedData[0].herLastName,
  //               newherEmail: updatedData[0].herEmail,
  //               newherMobile: updatedData[0].herMobile,
  //               newcity: updatedData[0].city,
  //               newstate: updatedData[0].state,
  //               newzip: updatedData[0].zip,
  //               newnotes: updatedData[0].notes,
  //               newparish: newP && newP.parish ? newP.parish : "",
  //               newaddress: updatedData[0].address,
  //               newallergies: updatedData[0].allergies,
  //               newanniversary_date: updatedData[0].anniversary_date,
  //               newhear_us: updatedData[0].hear_us,
  //               newemergency_name1: updatedData[0].emergency_name1,
  //               newemergency_relationship1:
  //                 updatedData[0].emergency_relationship1,
  //               newemergency_phone1: updatedData[0].emergency_phone1,
  //               newemergency_name2: updatedData[0].emergency_name2,
  //               newemergency_relationship2:
  //                 updatedData[0].emergency_relationship2,
  //               newemergency_phone2: updatedData[0].emergency_phone2,
  //               newunder_age_35: updatedData[0].under_age_35,
  //               newreferral_from_a_friend_or_family_member:
  //                 updatedData[0].referral_from_a_friend_or_family_member,
  //             });
  //           }
  //         } else {
  //           coupleNotExist.push({
  //             primaryKey: maxPrimaryKeyDB + 1,
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,
  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });
  //           if (
  //             coupleNotExist.length &&
  //             coupleNotExist.length > 0 &&
  //             coupleNotExist !== ""
  //           ) {
  //             let newUser = new coupleSchema(coupleNotExist[0]);
  //             let succ = await newUser.save();
  //             if (succ.hisEmail) {
  //               let randompass1 = Math.random().toString(36).slice(-8);
  //               paswordForCouple1 =
  //                 randompass1.slice(0, 0) + "p" + randompass1.slice(1);
  //             }
  //             if (succ.herEmail) {
  //               let randompass2 = Math.random().toString(36).slice(-8);
  //               paswordForCouple2 =
  //                 randompass2.slice(0, 0) + "p" + randompass2.slice(1);
  //             }
  //             adminObj = [
  //               {
  //                 userName: succ.hisFirstName,
  //                 email: succ.hisEmail,
  //                 password: paswordForCouple1 ? paswordForCouple1 : null,
  //                 accountStatus: "ACTIVE",
  //                 userRole: "COUPLE",
  //                 coupleId: succ.id,
  //                 emailVerified: 1,
  //                 firstName: succ.hisFirstName ? succ.hisFirstName : "",
  //                 lastName: succ.hisLastName ? succ.hisLastName : "",
  //                 mobile: succ.hisMobile ? succ.hisMobile : "",
  //               },
  //               {
  //                 userName: succ.herFirstName,
  //                 email: succ.herEmail,
  //                 password: paswordForCouple2 ? paswordForCouple2 : null,
  //                 accountStatus: "ACTIVE",
  //                 userRole: "COUPLE",
  //                 coupleId: succ.id,
  //                 emailVerified: 1,
  //                 firstName: succ.herFirstName ? succ.herFirstName : "",
  //                 lastName: succ.herLastName ? succ.herLastName : "",
  //                 mobile: succ.herMobile ? succ.herMobile : "",
  //               },
  //             ];
  //             adminObj.map((succ) => {
  //               userSchema.create(succ);
  //             });
  //             if (succ.hisEmail) {
  //               let replData = SENDLOGINDETAILBYADMIN.replace(
  //                 /#firstName#/g,
  //                 succ.hisFirstName ? succ.hisFirstName : ""
  //               )
  //                 .replace(
  //                   /#lastName#/g,
  //                   succ.hisLastName ? succ.hisLastName : ""
  //                 )
  //                 .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
  //                 .replace(/#email#/g, succ.hisEmail ? succ.hisEmail : "Nil")
  //                 .replace(
  //                   /#password#/g,
  //                   paswordForCouple1 ? paswordForCouple1 : "Nil"
  //                 );
  //               Email.sendEmail(
  //                 succ.hisEmail,
  //                 "Couple Detail for Login- LOVESTRONG Marriage",
  //                 replData
  //               );
  //             }
  //             if (succ.herEmail) {
  //               let replData = SENDLOGINDETAILBYADMIN.replace(
  //                 /#firstName#/g,
  //                 succ.herFirstName ? succ.herFirstName : ""
  //               )
  //                 .replace(
  //                   /#lastName#/g,
  //                   succ.herLastName ? succ.herLastName : ""
  //                 )
  //                 .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

  //                 .replace(/#email#/g, succ.herEmail)
  //                 .replace(/#password#/g, paswordForCouple2);
  //               Email.sendEmail(
  //                 succ.herEmail,
  //                 "Couple Detail for Login- LOVESTRONG Marriage",
  //                 replData
  //               );
  //             }

  //             createdCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                 }` +
  //                 `${succ && succ.hisLastName ? `${succ.hisLastName}` : ""}` +
  //                 `${
  //                   succ && succ.hisFirstName ? `, ${succ.hisFirstName}` : ""
  //                 }` +
  //                 `${
  //                   succ && succ.herFirstName
  //                     ? ` and  ${succ.herFirstName}`
  //                     : ""
  //                 })`,
  //               primaryKey: succ.primaryKey,
  //               hisFirstName: succ.hisFirstName,
  //               hisLastName: succ.hisLastName,
  //               hisEmail: succ.hisEmail,
  //               hisMobile: succ.hisMobile,
  //               herFirstName: succ.herFirstName,
  //               herLastName: succ.herLastName,
  //               herEmail: succ.herEmail,
  //               herMobile: succ.herMobile,
  //               city: succ.city,
  //               state: succ.state,
  //               zip: succ.zip,
  //               notes: succ.notes,
  //               parish: retreatRoastData[i].parish,
  //               address: succ.address,
  //               allergies: succ.allergies,
  //               anniversary_date: succ.anniversary_date,
  //               hear_us: succ.hear_us,
  //               emergency_name1: succ.emergency_name1,
  //               emergency_relationship1: succ.emergency_relationship1,
  //               emergency_phone1: succ.emergency_phone1,
  //               emergency_name2: succ.emergency_name2,
  //               emergency_relationship2: succ.emergency_relationship2,
  //               emergency_phone2: succ.emergency_phone2,
  //               under_age_35: succ.under_age_35,
  //               referral_from_a_friend_or_family_member:
  //                 succ.referral_from_a_friend_or_family_member,
  //             });

  //             let foundRecordForRetreat = await coupleSchema.findAll({
  //               where: {
  //                 id: succ.id,
  //               },
  //               raw: true,
  //             });
  //             if (foundRecordForRetreat && foundRecordForRetreat.length > 0) {
  //               let userDataToUpdate = {
  //                 roleId: formatObj.roleId ? formatObj.roleId : null,
  //                 serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                 talkId: formatObj.talkId ? formatObj.talkId : null,
  //                 attendeeType: retreatRoastData[i].attendeeType,
  //                 coupleId:
  //                   foundRecordForRetreat && foundRecordForRetreat[0].id,
  //                 retreatId: retreatRoastData[i].retreatId,
  //               };
  //               let newRoaster = new retreatRoastSchema(userDataToUpdate);
  //               let succ1 = await newRoaster.save();
  //               createdRoasters.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 Role: retreatRoastData[i].role,
  //                 Service: retreatRoastData[i].service,
  //                 Talk: retreatRoastData[i].talk,
  //                 Parish: retreatRoastData[i].parish,
  //                 AttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             }
  //           }
  //         }
  //       }

  //       if (retreatRoastData[i].hisEmail !== "") {
  //         let foundRecord = await coupleSchema.findAll({
  //           where: {
  //             hisEmail: retreatRoastData[i].hisEmail,
  //           },
  //           raw: true,
  //         });
  //         if (foundRecord.length > 0) {
  //           let foundInRoaster = await retreatRoastSchema.findOne({
  //             where: {
  //               [Op.and]: [
  //                 { coupleId: foundRecord[0].id },
  //                 { retreatId: retreatRoastData[i].retreatId },
  //               ],
  //             },
  //             raw: true,
  //           });
  //           if (foundInRoaster && foundInRoaster !== null) {
  //             let retreatToUpdate = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //             };
  //             // Perform the update
  //             let updatedData = await retreatRoastSchema.findAll({
  //               where: {
  //                 [Op.and]: [
  //                   { id: foundInRoaster.id },
  //                   { retreatId: retreatRoastData[i].retreatId },
  //                   { coupleId: foundRecord[0].id },
  //                 ],
  //               },
  //               raw: true,
  //             });
  //             if (updatedData[0].roleId) {
  //               roleId = await roleSchema.findOne({
  //                 where: { id: updatedData[0].roleId },
  //                 raw: true,
  //               });
  //             }

  //             // service
  //             if (updatedData[0].serviceId) {
  //               serviceId = await serviceSchema.findOne({
  //                 where: { id: updatedData[0].serviceId },
  //                 raw: true,
  //               });
  //             }

  //             // talk
  //             if (updatedData[0].talkId) {
  //               talkId = await talkSchema.findOne({
  //                 where: { id: updatedData[0].talkId },
  //                 raw: true,
  //               });
  //             }

  //             let updateOldRec = await retreatRoastSchema.update(
  //               retreatToUpdate,
  //               {
  //                 where: {
  //                   [Op.and]: [
  //                     { id: foundInRoaster.id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                     { coupleId: foundRecord[0].id },
  //                   ],
  //                 },
  //               }
  //             );

  //             // Check if any rows were affected by the update
  //             if (updateOldRec[0] > 0) {
  //               updateRoaster.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 oldRole: roleId ? roleId.role : "",
  //                 oldService: serviceId ? serviceId.service : "",
  //                 oldTalk: talkId ? talkId.talk : "",
  //                 oldAttendeeType: updatedData && updatedData[0].attendeeType,
  //                 newRole: retreatRoastData[i].role,
  //                 newService: retreatRoastData[i].service,
  //                 newTalk: retreatRoastData[i].talk,
  //                 newAttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             } else {
  //               console.log("No rows were updated.");
  //             }
  //           } else {
  //             let retreatData = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //               coupleId: foundRecord[0].id,
  //               retreatId: retreatRoastData[i].retreatId,
  //             };
  //             let newUser = new retreatRoastSchema(retreatData);
  //             let succ = await newUser.save();
  //             createdRoasters.push({
  //               couple:
  //                 `Couple : (${
  //                   retreatRoastData[i].primaryKey
  //                     ? `${retreatRoastData[i].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                     ? `${retreatRoastData[i].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                     : ""
  //                 })`,
  //               Role: retreatRoastData[i].role,
  //               Service: retreatRoastData[i].service,
  //               Talk: retreatRoastData[i].talk,
  //               Parish: retreatRoastData[i].parish,
  //               AttendeeType: retreatRoastData[i].attendeeType,
  //             });
  //           }

  //           coupleExist.push({
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,
  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });
  //           let oldData = await coupleSchema.findAll({
  //             where: { hisEmail: retreatRoastData[i].hisEmail },
  //             raw: true,
  //           });
  //           if (oldData[0].parishId) {
  //             oldP = await parishSchema.findOne({
  //               where: { id: oldData[0].parishId },
  //               raw: true,
  //             });
  //           }
  //           let update = await coupleSchema.update(coupleExist[0], {
  //             where: {
  //               hisEmail: retreatRoastData[i].hisEmail,
  //             },
  //           });
  //           if (update[0] > 0) {
  //             // If rows were updated, fetch the updated data
  //             let updatedData = await coupleSchema.findAll({
  //               where: {
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //               },
  //               raw: true,
  //             });
  //             if (updatedData[0].parishId) {
  //               newP = await parishSchema.findOne({
  //                 where: { id: updatedData[0].parishId },
  //                 raw: true,
  //               });
  //             }
  //             updateCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   updatedData[0].primaryKey
  //                     ? `${updatedData[0].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisLastName
  //                     ? `${updatedData[0].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisFirstName
  //                     ? `, ${updatedData[0].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].herFirstName
  //                     ? ` and  ${updatedData[0].herFirstName}`
  //                     : ""
  //                 })`,

  //               oldhisFirstName: oldData[0].hisFirstName,
  //               oldhisLastName: oldData[0].hisLastName,
  //               oldhisEmail: oldData[0].hisEmail,
  //               oldhisMobile: oldData[0].hisMobile,
  //               oldherFirstName: oldData[0].herFirstName,
  //               oldherLastName: oldData[0].herLastName,
  //               oldherEmail: oldData[0].herEmail,
  //               oldherMobile: oldData[0].herMobile,
  //               oldcity: oldData[0].city,
  //               oldstate: oldData[0].state,
  //               oldzip: oldData[0].zip,
  //               oldnotes: oldData[0].notes,
  //               oldparish: oldP && oldP.parish ? oldP.parish : "",
  //               oldaddress: oldData[0].address,
  //               oldallergies: oldData[0].allergies,
  //               oldanniversary_date: oldData[0].anniversary_date,
  //               oldhear_us: oldData[0].hear_us,
  //               oldemergency_name1: oldData[0].emergency_name1,
  //               oldemergency_relationship1: oldData[0].emergency_relationship1,
  //               oldemergency_phone1: oldData[0].emergency_phone1,
  //               oldemergency_name2: oldData[0].emergency_name2,
  //               oldemergency_relationship2: oldData[0].emergency_relationship2,
  //               oldemergency_phone2: oldData[0].emergency_phone2,
  //               oldunder_age_35: oldData[0].under_age_35,
  //               oldreferral_from_a_friend_or_family_member:
  //                 oldData[0].referral_from_a_friend_or_family_member,
  //               newhisFirstName: updatedData[0].hisFirstName,
  //               newhisLastName: updatedData[0].hisLastName,
  //               newhisEmail: updatedData[0].hisEmail,
  //               newhisMobile: updatedData[0].hisMobile,
  //               newherFirstName: updatedData[0].herFirstName,
  //               newherLastName: updatedData[0].herLastName,
  //               newherEmail: updatedData[0].herEmail,
  //               newherMobile: updatedData[0].herMobile,
  //               newcity: updatedData[0].city,
  //               newstate: updatedData[0].state,
  //               newzip: updatedData[0].zip,
  //               newnotes: updatedData[0].notes,
  //               newparish: newP && newP.parish ? newP.parish : "",
  //               newaddress: updatedData[0].address,
  //               newallergies: updatedData[0].allergies,
  //               newanniversary_date: updatedData[0].anniversary_date,
  //               newhear_us: updatedData[0].hear_us,
  //               newemergency_name1: updatedData[0].emergency_name1,
  //               newemergency_relationship1:
  //                 updatedData[0].emergency_relationship1,
  //               newemergency_phone1: updatedData[0].emergency_phone1,
  //               newemergency_name2: updatedData[0].emergency_name2,
  //               newemergency_relationship2:
  //                 updatedData[0].emergency_relationship2,
  //               newemergency_phone2: updatedData[0].emergency_phone2,
  //               newunder_age_35: updatedData[0].under_age_35,
  //               newreferral_from_a_friend_or_family_member:
  //                 updatedData[0].referral_from_a_friend_or_family_member,
  //             });
  //           }
  //         } else {
  //           if (retreatRoastData[i].herEmail !== "") {
  //             let foundRecord = await coupleSchema.findAll({
  //               where: {
  //                 herEmail: retreatRoastData[i].herEmail,
  //               },
  //               raw: true,
  //             });
  //             if (foundRecord.length > 0) {
  //               let foundInRoaster = await retreatRoastSchema.findOne({
  //                 where: {
  //                   [Op.and]: [
  //                     { coupleId: foundRecord[0].id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                   ],
  //                 },
  //                 raw: true,
  //               });

  //               if (foundInRoaster && foundInRoaster !== null) {
  //                 let retreatToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                 };
  //                 // Perform the update
  //                 let updatedData = await retreatRoastSchema.findAll({
  //                   where: {
  //                     [Op.and]: [
  //                       { id: foundInRoaster.id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                       { coupleId: foundRecord[0].id },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });

  //                 if (updatedData[0].roleId) {
  //                   roleId = await roleSchema.findOne({
  //                     where: { id: updatedData[0].roleId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // service
  //                 if (updatedData[0].serviceId) {
  //                   serviceId = await serviceSchema.findOne({
  //                     where: { id: updatedData[0].serviceId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // talk
  //                 if (updatedData[0].talkId) {
  //                   talkId = await talkSchema.findOne({
  //                     where: { id: updatedData[0].talkId },
  //                     raw: true,
  //                   });
  //                 }

  //                 let updateOldRec = await retreatRoastSchema.update(
  //                   retreatToUpdate,
  //                   {
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         { retreatId: retreatRoastData[i].retreatId },
  //                         { coupleId: foundRecord[0].id },
  //                       ],
  //                     },
  //                   }
  //                 );

  //                 // Check if any rows were affected by the update
  //                 if (updateOldRec[0] > 0) {
  //                   updateRoaster.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     oldRole: roleId ? roleId.role : "",
  //                     oldService: serviceId ? serviceId.service : "",
  //                     oldTalk: talkId ? talkId.talk : "",
  //                     oldAttendeeType:
  //                       updatedData && updatedData[0].attendeeType,

  //                     newRole: retreatRoastData[i].role,
  //                     newService: retreatRoastData[i].service,
  //                     newTalk: retreatRoastData[i].talk,
  //                     newAttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 } else {
  //                   console.log("No rows were updated.");
  //                 }
  //               } else {
  //                 let retreatData = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId: foundRecord[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newUser = new retreatRoastSchema(retreatData);
  //                 let succ = await newUser.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }

  //               coupleExist.push({
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               let oldData = await coupleSchema.findAll({
  //                 where: { herEmail: retreatRoastData[i].herEmail },
  //                 raw: true,
  //               });
  //               if (oldData[0].parishId) {
  //                 oldP = await parishSchema.findOne({
  //                   where: { id: oldData[0].parishId },
  //                   raw: true,
  //                 });
  //               }
  //               let update = await coupleSchema.update(coupleExist[0], {
  //                 where: {
  //                   herEmail: retreatRoastData[i].herEmail,
  //                 },
  //               });

  //               if (update[0] > 0) {
  //                 // If rows were updated, fetch the updated data
  //                 let updatedData = await coupleSchema.findAll({
  //                   where: {
  //                     herEmail: retreatRoastData[i].herEmail,
  //                   },
  //                   raw: true,
  //                 });

  //                 if (updatedData[0].parishId) {
  //                   newP = await parishSchema.findOne({
  //                     where: { id: updatedData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 updateCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       updatedData[0].primaryKey
  //                         ? `${updatedData[0].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisLastName
  //                         ? `${updatedData[0].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisFirstName
  //                         ? `, ${updatedData[0].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].herFirstName
  //                         ? ` and  ${updatedData[0].herFirstName}`
  //                         : ""
  //                     })`,

  //                   oldhisFirstName: oldData[0].hisFirstName,
  //                   oldhisLastName: oldData[0].hisLastName,
  //                   oldhisEmail: oldData[0].hisEmail,
  //                   oldhisMobile: oldData[0].hisMobile,
  //                   oldherFirstName: oldData[0].herFirstName,
  //                   oldherLastName: oldData[0].herLastName,
  //                   oldherEmail: oldData[0].herEmail,
  //                   oldherMobile: oldData[0].herMobile,
  //                   oldcity: oldData[0].city,
  //                   oldstate: oldData[0].state,
  //                   oldzip: oldData[0].zip,
  //                   oldnotes: oldData[0].notes,
  //                   oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                   oldaddress: oldData[0].address,
  //                   oldallergies: oldData[0].allergies,
  //                   oldanniversary_date: oldData[0].anniversary_date,
  //                   oldhear_us: oldData[0].hear_us,
  //                   oldemergency_name1: oldData[0].emergency_name1,
  //                   oldemergency_relationship1:
  //                     oldData[0].emergency_relationship1,
  //                   oldemergency_phone1: oldData[0].emergency_phone1,
  //                   oldemergency_name2: oldData[0].emergency_name2,
  //                   oldemergency_relationship2:
  //                     oldData[0].emergency_relationship2,
  //                   oldemergency_phone2: oldData[0].emergency_phone2,
  //                   oldunder_age_35: oldData[0].under_age_35,
  //                   oldreferral_from_a_friend_or_family_member:
  //                     oldData[0].referral_from_a_friend_or_family_member,
  //                   newhisFirstName: updatedData[0].hisFirstName,
  //                   newhisLastName: updatedData[0].hisLastName,
  //                   newhisEmail: updatedData[0].hisEmail,
  //                   newhisMobile: updatedData[0].hisMobile,
  //                   newherFirstName: updatedData[0].herFirstName,
  //                   newherLastName: updatedData[0].herLastName,
  //                   newherEmail: updatedData[0].herEmail,
  //                   newherMobile: updatedData[0].herMobile,
  //                   newcity: updatedData[0].city,
  //                   newstate: updatedData[0].state,
  //                   newzip: updatedData[0].zip,
  //                   newnotes: updatedData[0].notes,
  //                   newparish: newP && newP.parish ? newP.parish : "",
  //                   newaddress: updatedData[0].address,
  //                   newallergies: updatedData[0].allergies,
  //                   newanniversary_date: updatedData[0].anniversary_date,
  //                   newhear_us: updatedData[0].hear_us,
  //                   newemergency_name1: updatedData[0].emergency_name1,
  //                   newemergency_relationship1:
  //                     updatedData[0].emergency_relationship1,
  //                   newemergency_phone1: updatedData[0].emergency_phone1,
  //                   newemergency_name2: updatedData[0].emergency_name2,
  //                   newemergency_relationship2:
  //                     updatedData[0].emergency_relationship2,
  //                   newemergency_phone2: updatedData[0].emergency_phone2,
  //                   newunder_age_35: updatedData[0].under_age_35,
  //                   newreferral_from_a_friend_or_family_member:
  //                     updatedData[0].referral_from_a_friend_or_family_member,
  //                 });
  //               }
  //             } else {
  //               let f = await coupleSchema.findAll({
  //                 where: {
  //                   primaryKey: retreatRoastData[i].primaryKey,
  //                 },
  //                 raw: true,
  //               });
  //               if (f.length > 0) {
  //                 let foundInRoaster = await retreatRoastSchema.findOne({
  //                   where: {
  //                     [Op.and]: [
  //                       { coupleId: f[0].id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });
  //                 if (foundInRoaster && foundInRoaster !== null) {
  //                   let retreatToUpdate = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                   };

  //                   let updatedData = await retreatRoastSchema.findAll({
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         { retreatId: retreatRoastData[i].retreatId },
  //                         { coupleId: f[0].id },
  //                       ],
  //                     },
  //                     raw: true,
  //                   });
  //                   if (updatedData[0].roleId) {
  //                     roleId = await roleSchema.findOne({
  //                       where: { id: updatedData[0].roleId },
  //                       raw: true,
  //                     });
  //                   }

  //                   // service
  //                   if (updatedData[0].serviceId) {
  //                     serviceId = await serviceSchema.findOne({
  //                       where: { id: updatedData[0].serviceId },
  //                       raw: true,
  //                     });
  //                   }

  //                   // talk
  //                   if (updatedData[0].talkId) {
  //                     talkId = await talkSchema.findOne({
  //                       where: { id: updatedData[0].talkId },
  //                       raw: true,
  //                     });
  //                   }

  //                   let updateOldRec = await retreatRoastSchema.update(
  //                     retreatToUpdate,
  //                     {
  //                       where: {
  //                         [Op.and]: [
  //                           { id: foundInRoaster.id },
  //                           { retreatId: retreatRoastData[i].retreatId },
  //                           { coupleId: f[0].id },
  //                         ],
  //                       },
  //                     }
  //                   );

  //                   // Check if any rows were affected by the update
  //                   if (updateOldRec[0] > 0) {
  //                     updateRoaster.push({
  //                       couple:
  //                         `Couple : (${
  //                           retreatRoastData[i].primaryKey
  //                             ? `${retreatRoastData[i].primaryKey} - `
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].hisLastName
  //                             ? `${retreatRoastData[i].hisLastName}`
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].hisFirstName
  //                             ? `, ${retreatRoastData[i].hisFirstName}`
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].herFirstName
  //                             ? ` and  ${retreatRoastData[i].herFirstName}`
  //                             : ""
  //                         })`,
  //                       oldRole: roleId ? roleId.role : "",
  //                       oldService: serviceId ? serviceId.service : "",
  //                       oldTalk: talkId ? talkId.talk : "",
  //                       oldAttendeeType:
  //                         updatedData && updatedData[0].attendeeType,

  //                       newRole: retreatRoastData[i].role,
  //                       newService: retreatRoastData[i].service,
  //                       newTalk: retreatRoastData[i].talk,
  //                       newAttendeeType: retreatRoastData[i].attendeeType,
  //                     });
  //                   } else {
  //                     console.log("No rows were updated.");
  //                   }
  //                 } else {
  //                   let retreatData = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                     coupleId: f[0].id,
  //                     retreatId: retreatRoastData[i].retreatId,
  //                   };
  //                   let newUser = new retreatRoastSchema(retreatData);
  //                   let succ = await newUser.save();
  //                   createdRoasters.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     Role: retreatRoastData[i].role,
  //                     Service: retreatRoastData[i].service,
  //                     Talk: retreatRoastData[i].talk,
  //                     Parish: retreatRoastData[i].parish,
  //                     AttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 }

  //                 coupleExist.push({
  //                   hisFirstName: retreatRoastData[i].hisFirstName,
  //                   hisLastName: retreatRoastData[i].hisLastName,
  //                   hisEmail: retreatRoastData[i].hisEmail,
  //                   hisMobile: retreatRoastData[i].hisMobile,
  //                   herFirstName: retreatRoastData[i].herFirstName,
  //                   herLastName: retreatRoastData[i].herLastName,
  //                   herEmail: retreatRoastData[i].herEmail,
  //                   herMobile: retreatRoastData[i].herMobile,
  //                   city: retreatRoastData[i].city,
  //                   state: retreatRoastData[i].state,
  //                   zip: retreatRoastData[i].zip,
  //                   notes: retreatRoastData[i].notes,
  //                   parishId: parishId && parishId.id,
  //                   address: retreatRoastData[i].address,
  //                   allergies: retreatRoastData[i].allergies,
  //                   anniversary_date:
  //                     retreatRoastData[i].anniversary_date === ""
  //                       ? null
  //                       : retreatRoastData[i].anniversary_date,
  //                   hear_us: retreatRoastData[i].hear_us
  //                     ? JSON.stringify(retreatRoastData[i].hear_us)
  //                     : [],
  //                   emergency_name1: retreatRoastData[i].emergency_name1,
  //                   emergency_relationship1:
  //                     retreatRoastData[i].emergency_relationship1,
  //                   emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                   emergency_name2: retreatRoastData[i].emergency_name2,

  //                   emergency_relationship2:
  //                     retreatRoastData[i].emergency_relationship2,
  //                   emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                   under_age_35:
  //                     retreatRoastData[i] &&
  //                     retreatRoastData[i].under_age_35 &&
  //                     retreatRoastData[i].under_age_35 !== "" &&
  //                     retreatRoastData[i].under_age_35 !== null
  //                       ? retreatRoastData[i].under_age_35
  //                       : "NO",
  //                   referral_from_a_friend_or_family_member:
  //                     retreatRoastData[i]
  //                       .referral_from_a_friend_or_family_member,
  //                 });

  //                 let oldData = await coupleSchema.findAll({
  //                   where: { primaryKey: retreatRoastData[i].primaryKey },
  //                   raw: true,
  //                 });
  //                 if (oldData[0].parishId) {
  //                   oldP = await parishSchema.findOne({
  //                     where: { id: oldData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 let update = await coupleSchema.update(coupleExist[0], {
  //                   where: { primaryKey: retreatRoastData[i].primaryKey },
  //                 });

  //                 if (update[0] > 0) {
  //                   // If rows were updated, fetch the updated data
  //                   let updatedData = await coupleSchema.findAll({
  //                     where: { primaryKey: retreatRoastData[i].primaryKey },
  //                     raw: true,
  //                   });

  //                   if (updatedData[0].parishId) {
  //                     newP = await parishSchema.findOne({
  //                       where: { id: updatedData[0].parishId },
  //                       raw: true,
  //                     });
  //                   }
  //                   updateCouple.push({
  //                     couple:
  //                       `Couple : (${
  //                         updatedData[0].primaryKey
  //                           ? `${updatedData[0].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].hisLastName
  //                           ? `${updatedData[0].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].hisFirstName
  //                           ? `, ${updatedData[0].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].herFirstName
  //                           ? ` and  ${updatedData[0].herFirstName}`
  //                           : ""
  //                       })`,

  //                     oldhisFirstName: oldData[0].hisFirstName,
  //                     oldhisLastName: oldData[0].hisLastName,
  //                     oldhisEmail: oldData[0].hisEmail,
  //                     oldhisMobile: oldData[0].hisMobile,
  //                     oldherFirstName: oldData[0].herFirstName,
  //                     oldherLastName: oldData[0].herLastName,
  //                     oldherEmail: oldData[0].herEmail,
  //                     oldherMobile: oldData[0].herMobile,
  //                     oldcity: oldData[0].city,
  //                     oldstate: oldData[0].state,
  //                     oldzip: oldData[0].zip,
  //                     oldnotes: oldData[0].notes,
  //                     oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                     oldaddress: oldData[0].address,
  //                     oldallergies: oldData[0].allergies,
  //                     oldanniversary_date: oldData[0].anniversary_date,
  //                     oldhear_us: oldData[0].hear_us,
  //                     oldemergency_name1: oldData[0].emergency_name1,
  //                     oldemergency_relationship1:
  //                       oldData[0].emergency_relationship1,
  //                     oldemergency_phone1: oldData[0].emergency_phone1,
  //                     oldemergency_name2: oldData[0].emergency_name2,
  //                     oldemergency_relationship2:
  //                       oldData[0].emergency_relationship2,
  //                     oldemergency_phone2: oldData[0].emergency_phone2,
  //                     oldunder_age_35: oldData[0].under_age_35,
  //                     oldreferral_from_a_friend_or_family_member:
  //                       oldData[0].referral_from_a_friend_or_family_member,
  //                     newhisFirstName: updatedData[0].hisFirstName,
  //                     newhisLastName: updatedData[0].hisLastName,
  //                     newhisEmail: updatedData[0].hisEmail,
  //                     newhisMobile: updatedData[0].hisMobile,
  //                     newherFirstName: updatedData[0].herFirstName,
  //                     newherLastName: updatedData[0].herLastName,
  //                     newherEmail: updatedData[0].herEmail,
  //                     newherMobile: updatedData[0].herMobile,
  //                     newcity: updatedData[0].city,
  //                     newstate: updatedData[0].state,
  //                     newzip: updatedData[0].zip,
  //                     newnotes: updatedData[0].notes,
  //                     newparish: newP && newP.parish ? newP.parish : "",
  //                     newaddress: updatedData[0].address,
  //                     newallergies: updatedData[0].allergies,
  //                     newanniversary_date: updatedData[0].anniversary_date,
  //                     newhear_us: updatedData[0].hear_us,
  //                     newemergency_name1: updatedData[0].emergency_name1,
  //                     newemergency_relationship1:
  //                       updatedData[0].emergency_relationship1,
  //                     newemergency_phone1: updatedData[0].emergency_phone1,
  //                     newemergency_name2: updatedData[0].emergency_name2,
  //                     newemergency_relationship2:
  //                       updatedData[0].emergency_relationship2,
  //                     newemergency_phone2: updatedData[0].emergency_phone2,
  //                     newunder_age_35: updatedData[0].under_age_35,
  //                     newreferral_from_a_friend_or_family_member:
  //                       updatedData[0].referral_from_a_friend_or_family_member,
  //                   });
  //                 }
  //               } else {
  //                 if (retreatRoastData[i].hisMobile !== "") {
  //                   let normalizedHisMobile = retreatRoastData[
  //                     i
  //                   ].hisMobile.replace(/\D/g, "");
  //                   let foundRecord = await coupleSchema.findAll({
  //                     where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHisMobile}'
  // `),
  //                     raw: true,
  //                   });

  //                   if (foundRecord.length > 0) {
  //                     let foundInRoaster = await retreatRoastSchema.findOne({
  //                       where: {
  //                         [Op.and]: [
  //                           { coupleId: foundRecord[0].id },
  //                           { retreatId: retreatRoastData[i].retreatId },
  //                         ],
  //                       },
  //                       raw: true,
  //                     });

  //                     if (foundInRoaster && foundInRoaster !== null) {
  //                       let retreatToUpdate = {
  //                         roleId: formatObj.roleId ? formatObj.roleId : null,
  //                         serviceId: formatObj.serviceId
  //                           ? formatObj.serviceId
  //                           : null,
  //                         talkId: formatObj.talkId ? formatObj.talkId : null,
  //                         attendeeType: retreatRoastData[i].attendeeType,
  //                       };
  //                       // Perform the update
  //                       let updatedData = await retreatRoastSchema.findAll({
  //                         where: {
  //                           [Op.and]: [
  //                             { id: foundInRoaster.id },
  //                             { retreatId: retreatRoastData[i].retreatId },
  //                             { coupleId: foundRecord[0].id },
  //                           ],
  //                         },
  //                         raw: true,
  //                       });

  //                       if (updatedData[0].roleId) {
  //                         roleId = await roleSchema.findOne({
  //                           where: { id: updatedData[0].roleId },
  //                           raw: true,
  //                         });
  //                       }

  //                       // service
  //                       if (updatedData[0].serviceId) {
  //                         serviceId = await serviceSchema.findOne({
  //                           where: { id: updatedData[0].serviceId },
  //                           raw: true,
  //                         });
  //                       }

  //                       // talk
  //                       if (updatedData[0].talkId) {
  //                         talkId = await talkSchema.findOne({
  //                           where: { id: updatedData[0].talkId },
  //                           raw: true,
  //                         });
  //                       }

  //                       let updateOldRec = await retreatRoastSchema.update(
  //                         retreatToUpdate,
  //                         {
  //                           where: {
  //                             [Op.and]: [
  //                               { id: foundInRoaster.id },
  //                               { retreatId: retreatRoastData[i].retreatId },
  //                               { coupleId: foundRecord[0].id },
  //                             ],
  //                           },
  //                         }
  //                       );

  //                       // Check if any rows were affected by the update
  //                       if (updateOldRec[0] > 0) {
  //                         updateRoaster.push({
  //                           couple:
  //                             `Couple : (${
  //                               retreatRoastData[i].primaryKey
  //                                 ? `${retreatRoastData[i].primaryKey} - `
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].hisLastName
  //                                 ? `${retreatRoastData[i].hisLastName}`
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].hisFirstName
  //                                 ? `, ${retreatRoastData[i].hisFirstName}`
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].herFirstName
  //                                 ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                 : ""
  //                             })`,
  //                           oldRole: roleId ? roleId.role : "",
  //                           oldService: serviceId ? serviceId.service : "",
  //                           oldTalk: talkId ? talkId.talk : "",
  //                           oldAttendeeType:
  //                             updatedData && updatedData[0].attendeeType,

  //                           newRole: retreatRoastData[i].role,
  //                           newService: retreatRoastData[i].service,
  //                           newTalk: retreatRoastData[i].talk,
  //                           newAttendeeType: retreatRoastData[i].attendeeType,
  //                         });
  //                       } else {
  //                         console.log("No rows were updated.");
  //                       }
  //                     } else {
  //                       let retreatData = {
  //                         roleId: formatObj.roleId ? formatObj.roleId : null,
  //                         serviceId: formatObj.serviceId
  //                           ? formatObj.serviceId
  //                           : null,
  //                         talkId: formatObj.talkId ? formatObj.talkId : null,
  //                         attendeeType: retreatRoastData[i].attendeeType,
  //                         coupleId: foundRecord[0].id,
  //                         retreatId: retreatRoastData[i].retreatId,
  //                       };
  //                       let newUser = new retreatRoastSchema(retreatData);
  //                       let succ = await newUser.save();
  //                       createdRoasters.push({
  //                         couple:
  //                           `Couple : (${
  //                             retreatRoastData[i].primaryKey
  //                               ? `${retreatRoastData[i].primaryKey} - `
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].hisLastName
  //                               ? `${retreatRoastData[i].hisLastName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].hisFirstName
  //                               ? `, ${retreatRoastData[i].hisFirstName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].herFirstName
  //                               ? ` and  ${retreatRoastData[i].herFirstName}`
  //                               : ""
  //                           })`,
  //                         Role: retreatRoastData[i].role,
  //                         Service: retreatRoastData[i].service,
  //                         Talk: retreatRoastData[i].talk,
  //                         Parish: retreatRoastData[i].parish,
  //                         AttendeeType: retreatRoastData[i].attendeeType,
  //                       });
  //                     }

  //                     coupleExist.push({
  //                       hisFirstName: retreatRoastData[i].hisFirstName,
  //                       hisLastName: retreatRoastData[i].hisLastName,
  //                       hisEmail: retreatRoastData[i].hisEmail,
  //                       hisMobile: retreatRoastData[i].hisMobile,
  //                       herFirstName: retreatRoastData[i].herFirstName,
  //                       herLastName: retreatRoastData[i].herLastName,
  //                       herEmail: retreatRoastData[i].herEmail,
  //                       herMobile: retreatRoastData[i].herMobile,
  //                       city: retreatRoastData[i].city,
  //                       state: retreatRoastData[i].state,
  //                       zip: retreatRoastData[i].zip,
  //                       notes: retreatRoastData[i].notes,
  //                       parishId: parishId && parishId.id,
  //                       address: retreatRoastData[i].address,
  //                       allergies: retreatRoastData[i].allergies,
  //                       anniversary_date:
  //                         retreatRoastData[i].anniversary_date === ""
  //                           ? null
  //                           : retreatRoastData[i].anniversary_date,
  //                       hear_us: retreatRoastData[i].hear_us
  //                         ? JSON.stringify(retreatRoastData[i].hear_us)
  //                         : [],
  //                       emergency_name1: retreatRoastData[i].emergency_name1,
  //                       emergency_relationship1:
  //                         retreatRoastData[i].emergency_relationship1,
  //                       emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                       emergency_name2: retreatRoastData[i].emergency_name2,
  //                       emergency_relationship2:
  //                         retreatRoastData[i].emergency_relationship2,
  //                       emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                       under_age_35:
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].under_age_35 &&
  //                         retreatRoastData[i].under_age_35 !== "" &&
  //                         retreatRoastData[i].under_age_35 !== null
  //                           ? retreatRoastData[i].under_age_35
  //                           : "NO",
  //                       referral_from_a_friend_or_family_member:
  //                         retreatRoastData[i]
  //                           .referral_from_a_friend_or_family_member,
  //                     });
  //                     let oldData = await coupleSchema.findAll({
  //                       where: { hisMobile: retreatRoastData[i].hisMobile },
  //                       raw: true,
  //                     });
  //                     if (oldData[0].parishId) {
  //                       oldP = await parishSchema.findOne({
  //                         where: { id: oldData[0].parishId },
  //                         raw: true,
  //                       });
  //                     }
  //                     let update = await coupleSchema.update(coupleExist[0], {
  //                       where: {
  //                         hisMobile: retreatRoastData[i].hisMobile,
  //                       },
  //                     });

  //                     if (update[0] > 0) {
  //                       // If rows were updated, fetch the updated data
  //                       let updatedData = await coupleSchema.findAll({
  //                         where: {
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                         },
  //                         raw: true,
  //                       });

  //                       if (updatedData[0].parishId) {
  //                         newP = await parishSchema.findOne({
  //                           where: { id: updatedData[0].parishId },
  //                           raw: true,
  //                         });
  //                       }
  //                       updateCouple.push({
  //                         couple:
  //                           `Couple : (${
  //                             updatedData[0].primaryKey
  //                               ? `${updatedData[0].primaryKey} - `
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].hisLastName
  //                               ? `${updatedData[0].hisLastName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].hisFirstName
  //                               ? `, ${updatedData[0].hisFirstName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].herFirstName
  //                               ? ` and  ${updatedData[0].herFirstName}`
  //                               : ""
  //                           })`,

  //                         oldhisFirstName: oldData[0].hisFirstName,
  //                         oldhisLastName: oldData[0].hisLastName,
  //                         oldhisEmail: oldData[0].hisEmail,
  //                         oldhisMobile: oldData[0].hisMobile,
  //                         oldherFirstName: oldData[0].herFirstName,
  //                         oldherLastName: oldData[0].herLastName,
  //                         oldherEmail: oldData[0].herEmail,
  //                         oldherMobile: oldData[0].herMobile,
  //                         oldcity: oldData[0].city,
  //                         oldstate: oldData[0].state,
  //                         oldzip: oldData[0].zip,
  //                         oldnotes: oldData[0].notes,
  //                         oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                         oldaddress: oldData[0].address,
  //                         oldallergies: oldData[0].allergies,
  //                         oldanniversary_date: oldData[0].anniversary_date,
  //                         oldhear_us: oldData[0].hear_us,
  //                         oldemergency_name1: oldData[0].emergency_name1,
  //                         oldemergency_relationship1:
  //                           oldData[0].emergency_relationship1,
  //                         oldemergency_phone1: oldData[0].emergency_phone1,
  //                         oldemergency_name2: oldData[0].emergency_name2,
  //                         oldemergency_relationship2:
  //                           oldData[0].emergency_relationship2,
  //                         oldemergency_phone2: oldData[0].emergency_phone2,
  //                         oldunder_age_35: oldData[0].under_age_35,
  //                         oldreferral_from_a_friend_or_family_member:
  //                           oldData[0].referral_from_a_friend_or_family_member,
  //                         newhisFirstName: updatedData[0].hisFirstName,
  //                         newhisLastName: updatedData[0].hisLastName,
  //                         newhisEmail: updatedData[0].hisEmail,
  //                         newhisMobile: updatedData[0].hisMobile,
  //                         newherFirstName: updatedData[0].herFirstName,
  //                         newherLastName: updatedData[0].herLastName,
  //                         newherEmail: updatedData[0].herEmail,
  //                         newherMobile: updatedData[0].herMobile,
  //                         newcity: updatedData[0].city,
  //                         newstate: updatedData[0].state,
  //                         newzip: updatedData[0].zip,
  //                         newnotes: updatedData[0].notes,
  //                         newparish: newP && newP.parish ? newP.parish : "",
  //                         newaddress: updatedData[0].address,
  //                         newallergies: updatedData[0].allergies,
  //                         newanniversary_date: updatedData[0].anniversary_date,
  //                         newhear_us: updatedData[0].hear_us,
  //                         newemergency_name1: updatedData[0].emergency_name1,
  //                         newemergency_relationship1:
  //                           updatedData[0].emergency_relationship1,
  //                         newemergency_phone1: updatedData[0].emergency_phone1,
  //                         newemergency_name2: updatedData[0].emergency_name2,
  //                         newemergency_relationship2:
  //                           updatedData[0].emergency_relationship2,
  //                         newemergency_phone2: updatedData[0].emergency_phone2,
  //                         newunder_age_35: updatedData[0].under_age_35,
  //                         newreferral_from_a_friend_or_family_member:
  //                           updatedData[0]
  //                             .referral_from_a_friend_or_family_member,
  //                       });
  //                     }
  //                   } else {
  //                     if (retreatRoastData[i].herMobile !== "") {
  //                       // console.log("iffffff hisemail; hai");

  //                       let normalizedHerMobile = retreatRoastData[
  //                         i
  //                       ].herMobile.replace(/\D/g, "");
  //                       let foundRecord = await coupleSchema.findAll({
  //                         where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHerMobile}'
  // `),
  //                         raw: true,
  //                       });
  //                       if (foundRecord.length > 0) {
  //                         let foundInRoaster = await retreatRoastSchema.findOne(
  //                           {
  //                             where: {
  //                               [Op.and]: [
  //                                 { coupleId: foundRecord[0].id },
  //                                 { retreatId: retreatRoastData[i].retreatId },
  //                               ],
  //                             },
  //                             raw: true,
  //                           }
  //                         );
  //                         if (foundInRoaster && foundInRoaster !== null) {
  //                           let retreatToUpdate = {
  //                             roleId: formatObj.roleId
  //                               ? formatObj.roleId
  //                               : null,
  //                             serviceId: formatObj.serviceId
  //                               ? formatObj.serviceId
  //                               : null,
  //                             talkId: formatObj.talkId
  //                               ? formatObj.talkId
  //                               : null,
  //                             attendeeType: retreatRoastData[i].attendeeType,
  //                           };
  //                           // Perform the update
  //                           let updatedData = await retreatRoastSchema.findAll({
  //                             where: {
  //                               [Op.and]: [
  //                                 { id: foundInRoaster.id },
  //                                 { retreatId: retreatRoastData[i].retreatId },
  //                                 { coupleId: foundRecord[0].id },
  //                               ],
  //                             },
  //                             raw: true,
  //                           });
  //                           if (updatedData[0].roleId) {
  //                             roleId = await roleSchema.findOne({
  //                               where: { id: updatedData[0].roleId },
  //                               raw: true,
  //                             });
  //                           }

  //                           // service
  //                           if (updatedData[0].serviceId) {
  //                             serviceId = await serviceSchema.findOne({
  //                               where: { id: updatedData[0].serviceId },
  //                               raw: true,
  //                             });
  //                           }

  //                           // talk
  //                           if (updatedData[0].talkId) {
  //                             talkId = await talkSchema.findOne({
  //                               where: { id: updatedData[0].talkId },
  //                               raw: true,
  //                             });
  //                           }

  //                           let updateOldRec = await retreatRoastSchema.update(
  //                             retreatToUpdate,
  //                             {
  //                               where: {
  //                                 [Op.and]: [
  //                                   { id: foundInRoaster.id },
  //                                   {
  //                                     retreatId: retreatRoastData[i].retreatId,
  //                                   },
  //                                   { coupleId: foundRecord[0].id },
  //                                 ],
  //                               },
  //                             }
  //                           );

  //                           // Check if any rows were affected by the update
  //                           if (updateOldRec[0] > 0) {
  //                             updateRoaster.push({
  //                               couple:
  //                                 `Couple : (${
  //                                   retreatRoastData[i].primaryKey
  //                                     ? `${retreatRoastData[i].primaryKey} - `
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisLastName
  //                                     ? `${retreatRoastData[i].hisLastName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisFirstName
  //                                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].herFirstName
  //                                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                     : ""
  //                                 })`,
  //                               oldRole: roleId ? roleId.role : "",
  //                               oldService: serviceId ? serviceId.service : "",
  //                               oldTalk: talkId ? talkId.talk : "",
  //                               oldAttendeeType:
  //                                 updatedData && updatedData[0].attendeeType,

  //                               newRole: retreatRoastData[i].role,
  //                               newService: retreatRoastData[i].service,
  //                               newTalk: retreatRoastData[i].talk,
  //                               newAttendeeType:
  //                                 retreatRoastData[i].attendeeType,
  //                             });
  //                           } else {
  //                             console.log("No rows were updated.");
  //                           }
  //                         } else {
  //                           let retreatData = {
  //                             roleId: formatObj.roleId
  //                               ? formatObj.roleId
  //                               : null,
  //                             serviceId: formatObj.serviceId
  //                               ? formatObj.serviceId
  //                               : null,
  //                             talkId: formatObj.talkId
  //                               ? formatObj.talkId
  //                               : null,
  //                             attendeeType: retreatRoastData[i].attendeeType,
  //                             coupleId: foundRecord[0].id,
  //                             retreatId: retreatRoastData[i].retreatId,
  //                           };
  //                           let newUser = new retreatRoastSchema(retreatData);
  //                           let succ = await newUser.save();
  //                           createdRoasters.push({
  //                             couple:
  //                               `Couple : (${
  //                                 retreatRoastData[i].primaryKey
  //                                   ? `${retreatRoastData[i].primaryKey} - `
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].hisLastName
  //                                   ? `${retreatRoastData[i].hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].hisFirstName
  //                                   ? `, ${retreatRoastData[i].hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].herFirstName
  //                                   ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                   : ""
  //                               })`,
  //                             Role: retreatRoastData[i].role,
  //                             Service: retreatRoastData[i].service,
  //                             Talk: retreatRoastData[i].talk,
  //                             Parish: retreatRoastData[i].parish,
  //                             AttendeeType: retreatRoastData[i].attendeeType,
  //                           });
  //                         }

  //                         coupleExist.push({
  //                           hisFirstName: retreatRoastData[i].hisFirstName,
  //                           hisLastName: retreatRoastData[i].hisLastName,
  //                           hisEmail: retreatRoastData[i].hisEmail,
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                           herFirstName: retreatRoastData[i].herFirstName,
  //                           herLastName: retreatRoastData[i].herLastName,
  //                           herEmail: retreatRoastData[i].herEmail,
  //                           herMobile: retreatRoastData[i].herMobile,
  //                           city: retreatRoastData[i].city,
  //                           state: retreatRoastData[i].state,
  //                           zip: retreatRoastData[i].zip,
  //                           notes: retreatRoastData[i].notes,
  //                           parishId: parishId && parishId.id,
  //                           address: retreatRoastData[i].address,
  //                           allergies: retreatRoastData[i].allergies,
  //                           anniversary_date:
  //                             retreatRoastData[i].anniversary_date === ""
  //                               ? null
  //                               : retreatRoastData[i].anniversary_date,
  //                           hear_us: retreatRoastData[i].hear_us
  //                             ? JSON.stringify(retreatRoastData[i].hear_us)
  //                             : [],
  //                           emergency_name1:
  //                             retreatRoastData[i].emergency_name1,
  //                           emergency_relationship1:
  //                             retreatRoastData[i].emergency_relationship1,
  //                           emergency_phone1:
  //                             retreatRoastData[i].emergency_phone1,
  //                           emergency_name2:
  //                             retreatRoastData[i].emergency_name2,
  //                           emergency_relationship2:
  //                             retreatRoastData[i].emergency_relationship2,
  //                           emergency_phone2:
  //                             retreatRoastData[i].emergency_phone2,
  //                           under_age_35:
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].under_age_35 &&
  //                             retreatRoastData[i].under_age_35 !== "" &&
  //                             retreatRoastData[i].under_age_35 !== null
  //                               ? retreatRoastData[i].under_age_35
  //                               : "NO",
  //                           referral_from_a_friend_or_family_member:
  //                             retreatRoastData[i]
  //                               .referral_from_a_friend_or_family_member,
  //                         });
  //                         let oldData = await coupleSchema.findAll({
  //                           where: { herMobile: retreatRoastData[i].herMobile },
  //                           raw: true,
  //                         });
  //                         if (oldData[0].parishId) {
  //                           oldP = await parishSchema.findOne({
  //                             where: { id: oldData[0].parishId },
  //                             raw: true,
  //                           });
  //                         }
  //                         let update = await coupleSchema.update(
  //                           coupleExist[0],
  //                           {
  //                             where: {
  //                               herMobile: retreatRoastData[i].herMobile,
  //                             },
  //                           }
  //                         );
  //                         if (update[0] > 0) {
  //                           // If rows were updated, fetch the updated data
  //                           let updatedData = await coupleSchema.findAll({
  //                             where: {
  //                               herMobile: retreatRoastData[i].herMobile,
  //                             },
  //                             raw: true,
  //                           });
  //                           if (updatedData[0].parishId) {
  //                             newP = await parishSchema.findOne({
  //                               where: { id: updatedData[0].parishId },
  //                               raw: true,
  //                             });
  //                           }
  //                           updateCouple.push({
  //                             couple:
  //                               `Couple : (${
  //                                 updatedData[0].primaryKey
  //                                   ? `${updatedData[0].primaryKey} - `
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].hisLastName
  //                                   ? `${updatedData[0].hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].hisFirstName
  //                                   ? `, ${updatedData[0].hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].herFirstName
  //                                   ? ` and  ${updatedData[0].herFirstName}`
  //                                   : ""
  //                               })`,

  //                             oldhisFirstName: oldData[0].hisFirstName,
  //                             oldhisLastName: oldData[0].hisLastName,
  //                             oldhisEmail: oldData[0].hisEmail,
  //                             oldhisMobile: oldData[0].hisMobile,
  //                             oldherFirstName: oldData[0].herFirstName,
  //                             oldherLastName: oldData[0].herLastName,
  //                             oldherEmail: oldData[0].herEmail,
  //                             oldherMobile: oldData[0].herMobile,
  //                             oldcity: oldData[0].city,
  //                             oldstate: oldData[0].state,
  //                             oldzip: oldData[0].zip,
  //                             oldnotes: oldData[0].notes,
  //                             oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                             oldaddress: oldData[0].address,
  //                             oldallergies: oldData[0].allergies,
  //                             oldanniversary_date: oldData[0].anniversary_date,
  //                             oldhear_us: oldData[0].hear_us,
  //                             oldemergency_name1: oldData[0].emergency_name1,
  //                             oldemergency_relationship1:
  //                               oldData[0].emergency_relationship1,
  //                             oldemergency_phone1: oldData[0].emergency_phone1,
  //                             oldemergency_name2: oldData[0].emergency_name2,
  //                             oldemergency_relationship2:
  //                               oldData[0].emergency_relationship2,
  //                             oldemergency_phone2: oldData[0].emergency_phone2,
  //                             oldunder_age_35: oldData[0].under_age_35,
  //                             oldreferral_from_a_friend_or_family_member:
  //                               oldData[0]
  //                                 .referral_from_a_friend_or_family_member,
  //                             newhisFirstName: updatedData[0].hisFirstName,
  //                             newhisLastName: updatedData[0].hisLastName,
  //                             newhisEmail: updatedData[0].hisEmail,
  //                             newhisMobile: updatedData[0].hisMobile,
  //                             newherFirstName: updatedData[0].herFirstName,
  //                             newherLastName: updatedData[0].herLastName,
  //                             newherEmail: updatedData[0].herEmail,
  //                             newherMobile: updatedData[0].herMobile,
  //                             newcity: updatedData[0].city,
  //                             newstate: updatedData[0].state,
  //                             newzip: updatedData[0].zip,
  //                             newnotes: updatedData[0].notes,
  //                             newparish: newP && newP.parish ? newP.parish : "",
  //                             newaddress: updatedData[0].address,
  //                             newallergies: updatedData[0].allergies,
  //                             newanniversary_date:
  //                               updatedData[0].anniversary_date,
  //                             newhear_us: updatedData[0].hear_us,
  //                             newemergency_name1:
  //                               updatedData[0].emergency_name1,
  //                             newemergency_relationship1:
  //                               updatedData[0].emergency_relationship1,
  //                             newemergency_phone1:
  //                               updatedData[0].emergency_phone1,
  //                             newemergency_name2:
  //                               updatedData[0].emergency_name2,
  //                             newemergency_relationship2:
  //                               updatedData[0].emergency_relationship2,
  //                             newemergency_phone2:
  //                               updatedData[0].emergency_phone2,
  //                             newunder_age_35: updatedData[0].under_age_35,
  //                             newreferral_from_a_friend_or_family_member:
  //                               updatedData[0]
  //                                 .referral_from_a_friend_or_family_member,
  //                           });
  //                         }
  //                       } else {
  //                         coupleNotExist.push({
  //                           primaryKey: maxPrimaryKeyDB + 1,
  //                           hisFirstName: retreatRoastData[i].hisFirstName,
  //                           hisLastName: retreatRoastData[i].hisLastName,
  //                           hisEmail: retreatRoastData[i].hisEmail,
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                           herFirstName: retreatRoastData[i].herFirstName,
  //                           herLastName: retreatRoastData[i].herLastName,
  //                           herEmail: retreatRoastData[i].herEmail,
  //                           herMobile: retreatRoastData[i].herMobile,
  //                           city: retreatRoastData[i].city,
  //                           state: retreatRoastData[i].state,
  //                           zip: retreatRoastData[i].zip,
  //                           notes: retreatRoastData[i].notes,
  //                           parishId: parishId && parishId.id,
  //                           address: retreatRoastData[i].address,
  //                           allergies: retreatRoastData[i].allergies,
  //                           anniversary_date:
  //                             retreatRoastData[i].anniversary_date === ""
  //                               ? null
  //                               : retreatRoastData[i].anniversary_date,
  //                           hear_us: retreatRoastData[i].hear_us
  //                             ? JSON.stringify(retreatRoastData[i].hear_us)
  //                             : [],
  //                           emergency_name1:
  //                             retreatRoastData[i].emergency_name1,
  //                           emergency_relationship1:
  //                             retreatRoastData[i].emergency_relationship1,
  //                           emergency_phone1:
  //                             retreatRoastData[i].emergency_phone1,
  //                           emergency_name2:
  //                             retreatRoastData[i].emergency_name2,
  //                           emergency_relationship2:
  //                             retreatRoastData[i].emergency_relationship2,
  //                           emergency_phone2:
  //                             retreatRoastData[i].emergency_phone2,
  //                           under_age_35:
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].under_age_35 &&
  //                             retreatRoastData[i].under_age_35 !== "" &&
  //                             retreatRoastData[i].under_age_35 !== null
  //                               ? retreatRoastData[i].under_age_35
  //                               : "NO",
  //                           referral_from_a_friend_or_family_member:
  //                             retreatRoastData[i]
  //                               .referral_from_a_friend_or_family_member,
  //                         });
  //                         // console.log(
  //                         //   "coupleNotExist- whenb primary key naa mile--->",
  //                         //   coupleNotExist
  //                         // );
  //                         if (
  //                           coupleNotExist.length &&
  //                           coupleNotExist.length > 0 &&
  //                           coupleNotExist !== ""
  //                         ) {
  //                           let newUser = new coupleSchema(coupleNotExist[0]);
  //                           let succ = await newUser.save();
  //                           // console.log("succ-- xcrteate couple->", succ.id);
  //                           if (succ.hisEmail) {
  //                             let randompass1 = Math.random()
  //                               .toString(36)
  //                               .slice(-8);
  //                             paswordForCouple1 =
  //                               randompass1.slice(0, 0) +
  //                               "p" +
  //                               randompass1.slice(1);
  //                           }
  //                           if (succ.herEmail) {
  //                             let randompass2 = Math.random()
  //                               .toString(36)
  //                               .slice(-8);
  //                             paswordForCouple2 =
  //                               randompass2.slice(0, 0) +
  //                               "p" +
  //                               randompass2.slice(1);
  //                           }
  //                           adminObj = [
  //                             {
  //                               userName: succ.hisFirstName,
  //                               email: succ.hisEmail,
  //                               password: paswordForCouple1
  //                                 ? paswordForCouple1
  //                                 : null,
  //                               accountStatus: "ACTIVE",
  //                               userRole: "COUPLE",
  //                               coupleId: succ.id,
  //                               emailVerified: 1,
  //                               firstName: succ.hisFirstName
  //                                 ? succ.hisFirstName
  //                                 : "",
  //                               lastName: succ.hisLastName
  //                                 ? succ.hisLastName
  //                                 : "",
  //                               mobile: succ.hisMobile ? succ.hisMobile : "",
  //                             },
  //                             {
  //                               userName: succ.herFirstName,
  //                               email: succ.herEmail,
  //                               password: paswordForCouple2
  //                                 ? paswordForCouple2
  //                                 : null,
  //                               accountStatus: "ACTIVE",
  //                               userRole: "COUPLE",
  //                               coupleId: succ.id,
  //                               emailVerified: 1,
  //                               firstName: succ.herFirstName
  //                                 ? succ.herFirstName
  //                                 : "",
  //                               lastName: succ.herLastName
  //                                 ? succ.herLastName
  //                                 : "",
  //                               mobile: succ.herMobile ? succ.herMobile : "",
  //                             },
  //                           ];
  //                           adminObj.map((succ) => {
  //                             userSchema.create(succ);
  //                           });
  //                           if (succ.hisEmail) {
  //                             let replData = SENDLOGINDETAILBYADMIN.replace(
  //                               /#firstName#/g,
  //                               succ.hisFirstName ? succ.hisFirstName : ""
  //                             )
  //                               .replace(
  //                                 /#lastName#/g,
  //                                 succ.hisLastName ? succ.hisLastName : ""
  //                               )
  //                               .replace(
  //                                 /#url#/g,
  //                                 `${process.env.SERVERADDRESS}/login`
  //                               )
  //                               .replace(/#email#/g, succ.hisEmail)
  //                               .replace(/#password#/g, paswordForCouple1);
  //                             Email.sendEmail(
  //                               succ.hisEmail,
  //                               "Couple Detail for Login- LOVESTRONG Marriage",
  //                               replData
  //                             );
  //                           }
  //                           if (succ.herEmail) {
  //                             let replData = SENDLOGINDETAILBYADMIN.replace(
  //                               /#firstName#/g,
  //                               succ.herFirstName ? succ.herFirstName : ""
  //                             )
  //                               .replace(
  //                                 /#lastName#/g,
  //                                 succ.herLastName ? succ.herLastName : ""
  //                               )
  //                               .replace(
  //                                 /#url#/g,
  //                                 `${process.env.SERVERADDRESS}/login`
  //                               )

  //                               .replace(/#email#/g, succ.herEmail)
  //                               .replace(/#password#/g, paswordForCouple2);
  //                             Email.sendEmail(
  //                               succ.herEmail,
  //                               "Couple Detail for Login- LOVESTRONG Marriage",
  //                               replData
  //                             );
  //                           }

  //                           createdCouple.push({
  //                             couple:
  //                               `Couple : (${
  //                                 succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.hisLastName
  //                                   ? `${succ.hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.hisFirstName
  //                                   ? `, ${succ.hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.herFirstName
  //                                   ? ` and  ${succ.herFirstName}`
  //                                   : ""
  //                               })`,
  //                             primaryKey: succ.primaryKey,
  //                             hisFirstName: succ.hisFirstName,
  //                             hisLastName: succ.hisLastName,
  //                             hisEmail: succ.hisEmail,
  //                             hisMobile: succ.hisMobile,
  //                             herFirstName: succ.herFirstName,
  //                             herLastName: succ.herLastName,
  //                             herEmail: succ.herEmail,
  //                             herMobile: succ.herMobile,
  //                             city: succ.city,
  //                             state: succ.state,
  //                             zip: succ.zip,
  //                             notes: succ.notes,
  //                             parish: retreatRoastData[i].parish,
  //                             address: succ.address,
  //                             allergies: succ.allergies,
  //                             anniversary_date: succ.anniversary_date,
  //                             hear_us: succ.hear_us,
  //                             emergency_name1: succ.emergency_name1,
  //                             emergency_relationship1:
  //                               succ.emergency_relationship1,
  //                             emergency_phone1: succ.emergency_phone1,
  //                             emergency_name2: succ.emergency_name2,
  //                             emergency_relationship2:
  //                               succ.emergency_relationship2,
  //                             emergency_phone2: succ.emergency_phone2,
  //                             under_age_35: succ.under_age_35,
  //                             referral_from_a_friend_or_family_member:
  //                               succ.referral_from_a_friend_or_family_member,
  //                           });

  //                           let foundRecordForRetreat =
  //                             await coupleSchema.findAll({
  //                               where: {
  //                                 id: succ.id,
  //                               },
  //                               raw: true,
  //                             });
  //                           if (
  //                             foundRecordForRetreat &&
  //                             foundRecordForRetreat.length > 0
  //                           ) {
  //                             let userDataToUpdate = {
  //                               roleId: formatObj.roleId
  //                                 ? formatObj.roleId
  //                                 : null,
  //                               serviceId: formatObj.serviceId
  //                                 ? formatObj.serviceId
  //                                 : null,
  //                               talkId: formatObj.talkId
  //                                 ? formatObj.talkId
  //                                 : null,
  //                               attendeeType: retreatRoastData[i].attendeeType,
  //                               coupleId:
  //                                 foundRecordForRetreat &&
  //                                 foundRecordForRetreat[0].id,
  //                               retreatId: retreatRoastData[i].retreatId,
  //                             };
  //                             let newRoaster = new retreatRoastSchema(
  //                               userDataToUpdate
  //                             );
  //                             let succ1 = await newRoaster.save();
  //                             createdRoasters.push({
  //                               couple:
  //                                 `Couple : (${
  //                                   retreatRoastData[i].primaryKey
  //                                     ? `${retreatRoastData[i].primaryKey} - `
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisLastName
  //                                     ? `${retreatRoastData[i].hisLastName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisFirstName
  //                                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].herFirstName
  //                                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                     : ""
  //                                 })`,
  //                               Role: retreatRoastData[i].role,
  //                               Service: retreatRoastData[i].service,
  //                               Talk: retreatRoastData[i].talk,
  //                               Parish: retreatRoastData[i].parish,
  //                               AttendeeType: retreatRoastData[i].attendeeType,
  //                             });
  //                           }
  //                         }
  //                       }
  //                     }
  //                   }
  //                 }
  //               }
  //             }
  //           }
  //         }
  //       }

  //       if (retreatRoastData[i].herEmail !== "") {
  //         let foundRecord = await coupleSchema.findAll({
  //           where: {
  //             herEmail: retreatRoastData[i].herEmail,
  //           },
  //           raw: true,
  //         });
  //         if (foundRecord.length > 0) {
  //           let foundInRoaster = await retreatRoastSchema.findOne({
  //             where: {
  //               [Op.and]: [
  //                 { coupleId: foundRecord[0].id },
  //                 { retreatId: retreatRoastData[i].retreatId },
  //               ],
  //             },
  //             raw: true,
  //           });

  //           if (foundInRoaster && foundInRoaster !== null) {
  //             let retreatToUpdate = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //             };
  //             // Perform the update
  //             let updatedData = await retreatRoastSchema.findAll({
  //               where: {
  //                 [Op.and]: [
  //                   { id: foundInRoaster.id },
  //                   { retreatId: retreatRoastData[i].retreatId },
  //                   { coupleId: foundRecord[0].id },
  //                 ],
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].roleId) {
  //               roleId = await roleSchema.findOne({
  //                 where: { id: updatedData[0].roleId },
  //                 raw: true,
  //               });
  //             }

  //             // service
  //             if (updatedData[0].serviceId) {
  //               serviceId = await serviceSchema.findOne({
  //                 where: { id: updatedData[0].serviceId },
  //                 raw: true,
  //               });
  //             }

  //             // talk
  //             if (updatedData[0].talkId) {
  //               talkId = await talkSchema.findOne({
  //                 where: { id: updatedData[0].talkId },
  //                 raw: true,
  //               });
  //             }

  //             let updateOldRec = await retreatRoastSchema.update(
  //               retreatToUpdate,
  //               {
  //                 where: {
  //                   [Op.and]: [
  //                     { id: foundInRoaster.id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                     { coupleId: foundRecord[0].id },
  //                   ],
  //                 },
  //               }
  //             );

  //             // Check if any rows were affected by the update
  //             if (updateOldRec[0] > 0) {
  //               updateRoaster.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 oldRole: roleId ? roleId.role : "",
  //                 oldService: serviceId ? serviceId.service : "",
  //                 oldTalk: talkId ? talkId.talk : "",
  //                 oldAttendeeType: updatedData && updatedData[0].attendeeType,

  //                 newRole: retreatRoastData[i].role,
  //                 newService: retreatRoastData[i].service,
  //                 newTalk: retreatRoastData[i].talk,
  //                 newAttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             } else {
  //               console.log("No rows were updated.");
  //             }
  //           } else {
  //             let retreatData = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //               coupleId: foundRecord[0].id,
  //               retreatId: retreatRoastData[i].retreatId,
  //             };
  //             let newUser = new retreatRoastSchema(retreatData);
  //             let succ = await newUser.save();
  //             createdRoasters.push({
  //               couple:
  //                 `Couple : (${
  //                   retreatRoastData[i].primaryKey
  //                     ? `${retreatRoastData[i].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                     ? `${retreatRoastData[i].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                     : ""
  //                 })`,
  //               Role: retreatRoastData[i].role,
  //               Service: retreatRoastData[i].service,
  //               Talk: retreatRoastData[i].talk,
  //               Parish: retreatRoastData[i].parish,
  //               AttendeeType: retreatRoastData[i].attendeeType,
  //             });
  //           }

  //           coupleExist.push({
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,
  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });
  //           let oldData = await coupleSchema.findAll({
  //             where: { herEmail: retreatRoastData[i].herEmail },
  //             raw: true,
  //           });
  //           if (oldData[0].parishId) {
  //             oldP = await parishSchema.findOne({
  //               where: { id: oldData[0].parishId },
  //               raw: true,
  //             });
  //           }
  //           let update = await coupleSchema.update(coupleExist[0], {
  //             where: {
  //               herEmail: retreatRoastData[i].herEmail,
  //             },
  //           });

  //           if (update[0] > 0) {
  //             // If rows were updated, fetch the updated data
  //             let updatedData = await coupleSchema.findAll({
  //               where: {
  //                 herEmail: retreatRoastData[i].herEmail,
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].parishId) {
  //               newP = await parishSchema.findOne({
  //                 where: { id: updatedData[0].parishId },
  //                 raw: true,
  //               });
  //             }
  //             updateCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   updatedData[0].primaryKey
  //                     ? `${updatedData[0].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisLastName
  //                     ? `${updatedData[0].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisFirstName
  //                     ? `, ${updatedData[0].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].herFirstName
  //                     ? ` and  ${updatedData[0].herFirstName}`
  //                     : ""
  //                 })`,

  //               oldhisFirstName: oldData[0].hisFirstName,
  //               oldhisLastName: oldData[0].hisLastName,
  //               oldhisEmail: oldData[0].hisEmail,
  //               oldhisMobile: oldData[0].hisMobile,
  //               oldherFirstName: oldData[0].herFirstName,
  //               oldherLastName: oldData[0].herLastName,
  //               oldherEmail: oldData[0].herEmail,
  //               oldherMobile: oldData[0].herMobile,
  //               oldcity: oldData[0].city,
  //               oldstate: oldData[0].state,
  //               oldzip: oldData[0].zip,
  //               oldnotes: oldData[0].notes,
  //               oldparish: oldP && oldP.parish ? oldP.parish : "",
  //               oldaddress: oldData[0].address,
  //               oldallergies: oldData[0].allergies,
  //               oldanniversary_date: oldData[0].anniversary_date,
  //               oldhear_us: oldData[0].hear_us,
  //               oldemergency_name1: oldData[0].emergency_name1,
  //               oldemergency_relationship1: oldData[0].emergency_relationship1,
  //               oldemergency_phone1: oldData[0].emergency_phone1,
  //               oldemergency_name2: oldData[0].emergency_name2,
  //               oldemergency_relationship2: oldData[0].emergency_relationship2,
  //               oldemergency_phone2: oldData[0].emergency_phone2,
  //               oldunder_age_35: oldData[0].under_age_35,
  //               oldreferral_from_a_friend_or_family_member:
  //                 oldData[0].referral_from_a_friend_or_family_member,
  //               newhisFirstName: updatedData[0].hisFirstName,
  //               newhisLastName: updatedData[0].hisLastName,
  //               newhisEmail: updatedData[0].hisEmail,
  //               newhisMobile: updatedData[0].hisMobile,
  //               newherFirstName: updatedData[0].herFirstName,
  //               newherLastName: updatedData[0].herLastName,
  //               newherEmail: updatedData[0].herEmail,
  //               newherMobile: updatedData[0].herMobile,
  //               newcity: updatedData[0].city,
  //               newstate: updatedData[0].state,
  //               newzip: updatedData[0].zip,
  //               newnotes: updatedData[0].notes,
  //               newparish: newP && newP.parish ? newP.parish : "",
  //               newaddress: updatedData[0].address,
  //               newallergies: updatedData[0].allergies,
  //               newanniversary_date: updatedData[0].anniversary_date,
  //               newhear_us: updatedData[0].hear_us,
  //               newemergency_name1: updatedData[0].emergency_name1,
  //               newemergency_relationship1:
  //                 updatedData[0].emergency_relationship1,
  //               newemergency_phone1: updatedData[0].emergency_phone1,
  //               newemergency_name2: updatedData[0].emergency_name2,
  //               newemergency_relationship2:
  //                 updatedData[0].emergency_relationship2,
  //               newemergency_phone2: updatedData[0].emergency_phone2,
  //               newunder_age_35: updatedData[0].under_age_35,
  //               newreferral_from_a_friend_or_family_member:
  //                 updatedData[0].referral_from_a_friend_or_family_member,
  //             });
  //           }
  //         } else {
  //           if (retreatRoastData[i].hisEmail !== "") {
  //             // console.log("iffffff hisemail; hai");
  //             let foundRecord = await coupleSchema.findAll({
  //               where: {
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //               },
  //               raw: true,
  //             });
  //             if (foundRecord.length > 0) {
  //               let foundInRoaster = await retreatRoastSchema.findOne({
  //                 where: {
  //                   [Op.and]: [
  //                     { coupleId: foundRecord[0].id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                   ],
  //                 },
  //                 raw: true,
  //               });
  //               if (foundInRoaster && foundInRoaster !== null) {
  //                 let retreatToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                 };
  //                 // Perform the update
  //                 let updatedData = await retreatRoastSchema.findAll({
  //                   where: {
  //                     [Op.and]: [
  //                       { id: foundInRoaster.id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                       { coupleId: foundRecord[0].id },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].roleId) {
  //                   roleId = await roleSchema.findOne({
  //                     where: { id: updatedData[0].roleId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // service
  //                 if (updatedData[0].serviceId) {
  //                   serviceId = await serviceSchema.findOne({
  //                     where: { id: updatedData[0].serviceId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // talk
  //                 if (updatedData[0].talkId) {
  //                   talkId = await talkSchema.findOne({
  //                     where: { id: updatedData[0].talkId },
  //                     raw: true,
  //                   });
  //                 }

  //                 let updateOldRec = await retreatRoastSchema.update(
  //                   retreatToUpdate,
  //                   {
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         { retreatId: retreatRoastData[i].retreatId },
  //                         { coupleId: foundRecord[0].id },
  //                       ],
  //                     },
  //                   }
  //                 );

  //                 // Check if any rows were affected by the update
  //                 if (updateOldRec[0] > 0) {
  //                   updateRoaster.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     oldRole: roleId ? roleId.role : "",
  //                     oldService: serviceId ? serviceId.service : "",
  //                     oldTalk: talkId ? talkId.talk : "",
  //                     oldAttendeeType:
  //                       updatedData && updatedData[0].attendeeType,

  //                     newRole: retreatRoastData[i].role,
  //                     newService: retreatRoastData[i].service,
  //                     newTalk: retreatRoastData[i].talk,
  //                     newAttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 } else {
  //                   console.log("No rows were updated.");
  //                 }
  //               } else {
  //                 let retreatData = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId: foundRecord[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newUser = new retreatRoastSchema(retreatData);
  //                 let succ = await newUser.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and  ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }

  //               coupleExist.push({
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               let oldData = await coupleSchema.findAll({
  //                 where: { hisEmail: retreatRoastData[i].hisEmail },
  //                 raw: true,
  //               });
  //               if (oldData[0].parishId) {
  //                 oldP = await parishSchema.findOne({
  //                   where: { id: oldData[0].parishId },
  //                   raw: true,
  //                 });
  //               }
  //               let update = await coupleSchema.update(coupleExist[0], {
  //                 where: {
  //                   hisEmail: retreatRoastData[i].hisEmail,
  //                 },
  //               });
  //               if (update[0] > 0) {
  //                 // If rows were updated, fetch the updated data
  //                 let updatedData = await coupleSchema.findAll({
  //                   where: {
  //                     hisEmail: retreatRoastData[i].hisEmail,
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].parishId) {
  //                   newP = await parishSchema.findOne({
  //                     where: { id: updatedData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 updateCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       updatedData[0].primaryKey
  //                         ? `${updatedData[0].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisLastName
  //                         ? `${updatedData[0].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisFirstName
  //                         ? `, ${updatedData[0].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].herFirstName
  //                         ? ` and  ${updatedData[0].herFirstName}`
  //                         : ""
  //                     })`,

  //                   oldhisFirstName: oldData[0].hisFirstName,
  //                   oldhisLastName: oldData[0].hisLastName,
  //                   oldhisEmail: oldData[0].hisEmail,
  //                   oldhisMobile: oldData[0].hisMobile,
  //                   oldherFirstName: oldData[0].herFirstName,
  //                   oldherLastName: oldData[0].herLastName,
  //                   oldherEmail: oldData[0].herEmail,
  //                   oldherMobile: oldData[0].herMobile,
  //                   oldcity: oldData[0].city,
  //                   oldstate: oldData[0].state,
  //                   oldzip: oldData[0].zip,
  //                   oldnotes: oldData[0].notes,
  //                   oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                   oldaddress: oldData[0].address,
  //                   oldallergies: oldData[0].allergies,
  //                   oldanniversary_date: oldData[0].anniversary_date,
  //                   oldhear_us: oldData[0].hear_us,
  //                   oldemergency_name1: oldData[0].emergency_name1,
  //                   oldemergency_relationship1:
  //                     oldData[0].emergency_relationship1,
  //                   oldemergency_phone1: oldData[0].emergency_phone1,
  //                   oldemergency_name2: oldData[0].emergency_name2,
  //                   oldemergency_relationship2:
  //                     oldData[0].emergency_relationship2,
  //                   oldemergency_phone2: oldData[0].emergency_phone2,
  //                   oldunder_age_35: oldData[0].under_age_35,
  //                   oldreferral_from_a_friend_or_family_member:
  //                     oldData[0].referral_from_a_friend_or_family_member,
  //                   newhisFirstName: updatedData[0].hisFirstName,
  //                   newhisLastName: updatedData[0].hisLastName,
  //                   newhisEmail: updatedData[0].hisEmail,
  //                   newhisMobile: updatedData[0].hisMobile,
  //                   newherFirstName: updatedData[0].herFirstName,
  //                   newherLastName: updatedData[0].herLastName,
  //                   newherEmail: updatedData[0].herEmail,
  //                   newherMobile: updatedData[0].herMobile,
  //                   newcity: updatedData[0].city,
  //                   newstate: updatedData[0].state,
  //                   newzip: updatedData[0].zip,
  //                   newnotes: updatedData[0].notes,
  //                   newparish: newP && newP.parish ? newP.parish : "",
  //                   newaddress: updatedData[0].address,
  //                   newallergies: updatedData[0].allergies,
  //                   newanniversary_date: updatedData[0].anniversary_date,
  //                   newhear_us: updatedData[0].hear_us,
  //                   newemergency_name1: updatedData[0].emergency_name1,
  //                   newemergency_relationship1:
  //                     updatedData[0].emergency_relationship1,
  //                   newemergency_phone1: updatedData[0].emergency_phone1,
  //                   newemergency_name2: updatedData[0].emergency_name2,
  //                   newemergency_relationship2:
  //                     updatedData[0].emergency_relationship2,
  //                   newemergency_phone2: updatedData[0].emergency_phone2,
  //                   newunder_age_35: updatedData[0].under_age_35,
  //                   newreferral_from_a_friend_or_family_member:
  //                     updatedData[0].referral_from_a_friend_or_family_member,
  //                 });
  //               }
  //             } else {
  //               let f = await coupleSchema.findAll({
  //                 where: {
  //                   primaryKey: retreatRoastData[i].primaryKey,
  //                 },
  //                 raw: true,
  //               });
  //               if (f.length > 0) {
  //                 let foundInRoaster = await retreatRoastSchema.findOne({
  //                   where: {
  //                     [Op.and]: [
  //                       { coupleId: f[0].id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });
  //                 if (foundInRoaster && foundInRoaster !== null) {
  //                   let retreatToUpdate = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                   };

  //                   let updatedData = await retreatRoastSchema.findAll({
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         { retreatId: retreatRoastData[i].retreatId },
  //                         { coupleId: f[0].id },
  //                       ],
  //                     },
  //                     raw: true,
  //                   });
  //                   if (updatedData[0].roleId) {
  //                     roleId = await roleSchema.findOne({
  //                       where: { id: updatedData[0].roleId },
  //                       raw: true,
  //                     });
  //                   }

  //                   // service
  //                   if (updatedData[0].serviceId) {
  //                     serviceId = await serviceSchema.findOne({
  //                       where: { id: updatedData[0].serviceId },
  //                       raw: true,
  //                     });
  //                   }

  //                   // talk
  //                   if (updatedData[0].talkId) {
  //                     talkId = await talkSchema.findOne({
  //                       where: { id: updatedData[0].talkId },
  //                       raw: true,
  //                     });
  //                   }

  //                   let updateOldRec = await retreatRoastSchema.update(
  //                     retreatToUpdate,
  //                     {
  //                       where: {
  //                         [Op.and]: [
  //                           { id: foundInRoaster.id },
  //                           { retreatId: retreatRoastData[i].retreatId },
  //                           { coupleId: f[0].id },
  //                         ],
  //                       },
  //                     }
  //                   );

  //                   // Check if any rows were affected by the update
  //                   if (updateOldRec[0] > 0) {
  //                     updateRoaster.push({
  //                       couple:
  //                         `Couple : (${
  //                           retreatRoastData[i].primaryKey
  //                             ? `${retreatRoastData[i].primaryKey} - `
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].hisLastName
  //                             ? `${retreatRoastData[i].hisLastName}`
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].hisFirstName
  //                             ? `, ${retreatRoastData[i].hisFirstName}`
  //                             : ""
  //                         }` +
  //                         `${
  //                           retreatRoastData[i] &&
  //                           retreatRoastData[i].herFirstName
  //                             ? ` and  ${retreatRoastData[i].herFirstName}`
  //                             : ""
  //                         })`,
  //                       oldRole: roleId ? roleId.role : "",
  //                       oldService: serviceId ? serviceId.service : "",
  //                       oldTalk: talkId ? talkId.talk : "",
  //                       oldAttendeeType:
  //                         updatedData && updatedData[0].attendeeType,

  //                       newRole: retreatRoastData[i].role,
  //                       newService: retreatRoastData[i].service,
  //                       newTalk: retreatRoastData[i].talk,
  //                       newAttendeeType: retreatRoastData[i].attendeeType,
  //                     });
  //                   } else {
  //                     console.log("No rows were updated.");
  //                   }
  //                 } else {
  //                   let retreatData = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                     coupleId: f[0].id,
  //                     retreatId: retreatRoastData[i].retreatId,
  //                   };
  //                   let newUser = new retreatRoastSchema(retreatData);
  //                   let succ = await newUser.save();
  //                   createdRoasters.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     Role: retreatRoastData[i].role,
  //                     Service: retreatRoastData[i].service,
  //                     Talk: retreatRoastData[i].talk,
  //                     Parish: retreatRoastData[i].parish,
  //                     AttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 }

  //                 coupleExist.push({
  //                   hisFirstName: retreatRoastData[i].hisFirstName,
  //                   hisLastName: retreatRoastData[i].hisLastName,
  //                   hisEmail: retreatRoastData[i].hisEmail,
  //                   hisMobile: retreatRoastData[i].hisMobile,
  //                   herFirstName: retreatRoastData[i].herFirstName,
  //                   herLastName: retreatRoastData[i].herLastName,
  //                   herEmail: retreatRoastData[i].herEmail,
  //                   herMobile: retreatRoastData[i].herMobile,
  //                   city: retreatRoastData[i].city,
  //                   state: retreatRoastData[i].state,
  //                   zip: retreatRoastData[i].zip,
  //                   notes: retreatRoastData[i].notes,
  //                   parishId: parishId && parishId.id,
  //                   address: retreatRoastData[i].address,
  //                   allergies: retreatRoastData[i].allergies,
  //                   anniversary_date:
  //                     retreatRoastData[i].anniversary_date === ""
  //                       ? null
  //                       : retreatRoastData[i].anniversary_date,
  //                   hear_us: retreatRoastData[i].hear_us
  //                     ? JSON.stringify(retreatRoastData[i].hear_us)
  //                     : [],
  //                   emergency_name1: retreatRoastData[i].emergency_name1,
  //                   emergency_relationship1:
  //                     retreatRoastData[i].emergency_relationship1,
  //                   emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                   emergency_name2: retreatRoastData[i].emergency_name2,
  //                   emergency_relationship2:
  //                     retreatRoastData[i].emergency_relationship2,
  //                   emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                   under_age_35:
  //                     retreatRoastData[i] &&
  //                     retreatRoastData[i].under_age_35 &&
  //                     retreatRoastData[i].under_age_35 !== "" &&
  //                     retreatRoastData[i].under_age_35 !== null
  //                       ? retreatRoastData[i].under_age_35
  //                       : "NO",
  //                   referral_from_a_friend_or_family_member:
  //                     retreatRoastData[i]
  //                       .referral_from_a_friend_or_family_member,
  //                 });

  //                 let oldData = await coupleSchema.findAll({
  //                   where: { primaryKey: retreatRoastData[i].primaryKey },
  //                   raw: true,
  //                 });
  //                 if (oldData[0].parishId) {
  //                   oldP = await parishSchema.findOne({
  //                     where: { id: oldData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 let update = await coupleSchema.update(coupleExist[0], {
  //                   where: { primaryKey: retreatRoastData[i].primaryKey },
  //                 });

  //                 if (update[0] > 0) {
  //                   // If rows were updated, fetch the updated data
  //                   let updatedData = await coupleSchema.findAll({
  //                     where: { primaryKey: retreatRoastData[i].primaryKey },
  //                     raw: true,
  //                   });

  //                   if (updatedData[0].parishId) {
  //                     newP = await parishSchema.findOne({
  //                       where: { id: updatedData[0].parishId },
  //                       raw: true,
  //                     });
  //                   }
  //                   updateCouple.push({
  //                     couple:
  //                       `Couple : (${
  //                         updatedData[0].primaryKey
  //                           ? `${updatedData[0].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].hisLastName
  //                           ? `${updatedData[0].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].hisFirstName
  //                           ? `, ${updatedData[0].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         updatedData[0] && updatedData[0].herFirstName
  //                           ? ` and  ${updatedData[0].herFirstName}`
  //                           : ""
  //                       })`,

  //                     oldhisFirstName: oldData[0].hisFirstName,
  //                     oldhisLastName: oldData[0].hisLastName,
  //                     oldhisEmail: oldData[0].hisEmail,
  //                     oldhisMobile: oldData[0].hisMobile,
  //                     oldherFirstName: oldData[0].herFirstName,
  //                     oldherLastName: oldData[0].herLastName,
  //                     oldherEmail: oldData[0].herEmail,
  //                     oldherMobile: oldData[0].herMobile,
  //                     oldcity: oldData[0].city,
  //                     oldstate: oldData[0].state,
  //                     oldzip: oldData[0].zip,
  //                     oldnotes: oldData[0].notes,
  //                     oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                     oldaddress: oldData[0].address,
  //                     oldallergies: oldData[0].allergies,
  //                     oldanniversary_date: oldData[0].anniversary_date,
  //                     oldhear_us: oldData[0].hear_us,
  //                     oldemergency_name1: oldData[0].emergency_name1,
  //                     oldemergency_relationship1:
  //                       oldData[0].emergency_relationship1,
  //                     oldemergency_phone1: oldData[0].emergency_phone1,
  //                     oldemergency_name2: oldData[0].emergency_name2,
  //                     oldemergency_relationship2:
  //                       oldData[0].emergency_relationship2,
  //                     oldemergency_phone2: oldData[0].emergency_phone2,
  //                     oldunder_age_35: oldData[0].under_age_35,
  //                     oldreferral_from_a_friend_or_family_member:
  //                       oldData[0].referral_from_a_friend_or_family_member,
  //                     newhisFirstName: updatedData[0].hisFirstName,
  //                     newhisLastName: updatedData[0].hisLastName,
  //                     newhisEmail: updatedData[0].hisEmail,
  //                     newhisMobile: updatedData[0].hisMobile,
  //                     newherFirstName: updatedData[0].herFirstName,
  //                     newherLastName: updatedData[0].herLastName,
  //                     newherEmail: updatedData[0].herEmail,
  //                     newherMobile: updatedData[0].herMobile,
  //                     newcity: updatedData[0].city,
  //                     newstate: updatedData[0].state,
  //                     newzip: updatedData[0].zip,
  //                     newnotes: updatedData[0].notes,
  //                     newparish: newP && newP.parish ? newP.parish : "",
  //                     newaddress: updatedData[0].address,
  //                     newallergies: updatedData[0].allergies,
  //                     newanniversary_date: updatedData[0].anniversary_date,
  //                     newhear_us: updatedData[0].hear_us,
  //                     newemergency_name1: updatedData[0].emergency_name1,
  //                     newemergency_relationship1:
  //                       updatedData[0].emergency_relationship1,
  //                     newemergency_phone1: updatedData[0].emergency_phone1,
  //                     newemergency_name2: updatedData[0].emergency_name2,
  //                     newemergency_relationship2:
  //                       updatedData[0].emergency_relationship2,
  //                     newemergency_phone2: updatedData[0].emergency_phone2,
  //                     newunder_age_35: updatedData[0].under_age_35,
  //                     newreferral_from_a_friend_or_family_member:
  //                       updatedData[0].referral_from_a_friend_or_family_member,
  //                   });
  //                 }
  //               } else {
  //                 if (retreatRoastData[i].hisMobile !== "") {
  //                   let normalizedHisMobile = retreatRoastData[
  //                     i
  //                   ].hisMobile.replace(/\D/g, "");
  //                   let foundRecord = await coupleSchema.findAll({
  //                     where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHisMobile}'
  // `),
  //                     raw: true,
  //                   });
  //                   if (foundRecord.length > 0) {
  //                     let foundInRoaster = await retreatRoastSchema.findOne({
  //                       where: {
  //                         [Op.and]: [
  //                           { coupleId: foundRecord[0].id },
  //                           { retreatId: retreatRoastData[i].retreatId },
  //                         ],
  //                       },
  //                       raw: true,
  //                     });

  //                     if (foundInRoaster && foundInRoaster !== null) {
  //                       let retreatToUpdate = {
  //                         roleId: formatObj.roleId ? formatObj.roleId : null,
  //                         serviceId: formatObj.serviceId
  //                           ? formatObj.serviceId
  //                           : null,
  //                         talkId: formatObj.talkId ? formatObj.talkId : null,
  //                         attendeeType: retreatRoastData[i].attendeeType,
  //                       };
  //                       // Perform the update
  //                       let updatedData = await retreatRoastSchema.findAll({
  //                         where: {
  //                           [Op.and]: [
  //                             { id: foundInRoaster.id },
  //                             { retreatId: retreatRoastData[i].retreatId },
  //                             { coupleId: foundRecord[0].id },
  //                           ],
  //                         },
  //                         raw: true,
  //                       });

  //                       if (updatedData[0].roleId) {
  //                         roleId = await roleSchema.findOne({
  //                           where: { id: updatedData[0].roleId },
  //                           raw: true,
  //                         });
  //                       }

  //                       // service
  //                       if (updatedData[0].serviceId) {
  //                         serviceId = await serviceSchema.findOne({
  //                           where: { id: updatedData[0].serviceId },
  //                           raw: true,
  //                         });
  //                       }

  //                       // talk
  //                       if (updatedData[0].talkId) {
  //                         talkId = await talkSchema.findOne({
  //                           where: { id: updatedData[0].talkId },
  //                           raw: true,
  //                         });
  //                       }

  //                       let updateOldRec = await retreatRoastSchema.update(
  //                         retreatToUpdate,
  //                         {
  //                           where: {
  //                             [Op.and]: [
  //                               { id: foundInRoaster.id },
  //                               { retreatId: retreatRoastData[i].retreatId },
  //                               { coupleId: foundRecord[0].id },
  //                             ],
  //                           },
  //                         }
  //                       );

  //                       // Check if any rows were affected by the update
  //                       if (updateOldRec[0] > 0) {
  //                         updateRoaster.push({
  //                           couple:
  //                             `Couple : (${
  //                               retreatRoastData[i].primaryKey
  //                                 ? `${retreatRoastData[i].primaryKey} - `
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].hisLastName
  //                                 ? `${retreatRoastData[i].hisLastName}`
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].hisFirstName
  //                                 ? `, ${retreatRoastData[i].hisFirstName}`
  //                                 : ""
  //                             }` +
  //                             `${
  //                               retreatRoastData[i] &&
  //                               retreatRoastData[i].herFirstName
  //                                 ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                 : ""
  //                             })`,
  //                           oldRole: roleId ? roleId.role : "",
  //                           oldService: serviceId ? serviceId.service : "",
  //                           oldTalk: talkId ? talkId.talk : "",
  //                           oldAttendeeType:
  //                             updatedData && updatedData[0].attendeeType,

  //                           newRole: retreatRoastData[i].role,
  //                           newService: retreatRoastData[i].service,
  //                           newTalk: retreatRoastData[i].talk,
  //                           newAttendeeType: retreatRoastData[i].attendeeType,
  //                         });
  //                       } else {
  //                         console.log("No rows were updated.");
  //                       }
  //                     } else {
  //                       let retreatData = {
  //                         roleId: formatObj.roleId ? formatObj.roleId : null,
  //                         serviceId: formatObj.serviceId
  //                           ? formatObj.serviceId
  //                           : null,
  //                         talkId: formatObj.talkId ? formatObj.talkId : null,
  //                         attendeeType: retreatRoastData[i].attendeeType,
  //                         coupleId: foundRecord[0].id,
  //                         retreatId: retreatRoastData[i].retreatId,
  //                       };
  //                       let newUser = new retreatRoastSchema(retreatData);
  //                       let succ = await newUser.save();
  //                       createdRoasters.push({
  //                         couple:
  //                           `Couple : (${
  //                             retreatRoastData[i].primaryKey
  //                               ? `${retreatRoastData[i].primaryKey} - `
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].hisLastName
  //                               ? `${retreatRoastData[i].hisLastName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].hisFirstName
  //                               ? `, ${retreatRoastData[i].hisFirstName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].herFirstName
  //                               ? ` and  ${retreatRoastData[i].herFirstName}`
  //                               : ""
  //                           })`,
  //                         Role: retreatRoastData[i].role,
  //                         Service: retreatRoastData[i].service,
  //                         Talk: retreatRoastData[i].talk,
  //                         Parish: retreatRoastData[i].parish,
  //                         AttendeeType: retreatRoastData[i].attendeeType,
  //                       });
  //                     }

  //                     coupleExist.push({
  //                       hisFirstName: retreatRoastData[i].hisFirstName,
  //                       hisLastName: retreatRoastData[i].hisLastName,
  //                       hisEmail: retreatRoastData[i].hisEmail,
  //                       hisMobile: retreatRoastData[i].hisMobile,
  //                       herFirstName: retreatRoastData[i].herFirstName,
  //                       herLastName: retreatRoastData[i].herLastName,
  //                       herEmail: retreatRoastData[i].herEmail,
  //                       herMobile: retreatRoastData[i].herMobile,
  //                       city: retreatRoastData[i].city,
  //                       state: retreatRoastData[i].state,
  //                       zip: retreatRoastData[i].zip,
  //                       notes: retreatRoastData[i].notes,
  //                       parishId: parishId && parishId.id,
  //                       address: retreatRoastData[i].address,
  //                       allergies: retreatRoastData[i].allergies,
  //                       anniversary_date:
  //                         retreatRoastData[i].anniversary_date === ""
  //                           ? null
  //                           : retreatRoastData[i].anniversary_date,
  //                       hear_us: retreatRoastData[i].hear_us
  //                         ? JSON.stringify(retreatRoastData[i].hear_us)
  //                         : [],
  //                       emergency_name1: retreatRoastData[i].emergency_name1,
  //                       emergency_relationship1:
  //                         retreatRoastData[i].emergency_relationship1,
  //                       emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                       emergency_name2: retreatRoastData[i].emergency_name2,
  //                       emergency_relationship2:
  //                         retreatRoastData[i].emergency_relationship2,
  //                       emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                       under_age_35:
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].under_age_35 &&
  //                         retreatRoastData[i].under_age_35 !== "" &&
  //                         retreatRoastData[i].under_age_35 !== null
  //                           ? retreatRoastData[i].under_age_35
  //                           : "NO",
  //                       referral_from_a_friend_or_family_member:
  //                         retreatRoastData[i]
  //                           .referral_from_a_friend_or_family_member,
  //                     });
  //                     let oldData = await coupleSchema.findAll({
  //                       where: { hisMobile: retreatRoastData[i].hisMobile },
  //                       raw: true,
  //                     });
  //                     if (oldData[0].parishId) {
  //                       oldP = await parishSchema.findOne({
  //                         where: { id: oldData[0].parishId },
  //                         raw: true,
  //                       });
  //                     }
  //                     let update = await coupleSchema.update(coupleExist[0], {
  //                       where: {
  //                         hisMobile: retreatRoastData[i].hisMobile,
  //                       },
  //                     });

  //                     if (update[0] > 0) {
  //                       // If rows were updated, fetch the updated data
  //                       let updatedData = await coupleSchema.findAll({
  //                         where: {
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                         },
  //                         raw: true,
  //                       });

  //                       if (updatedData[0].parishId) {
  //                         newP = await parishSchema.findOne({
  //                           where: { id: updatedData[0].parishId },
  //                           raw: true,
  //                         });
  //                       }
  //                       updateCouple.push({
  //                         couple:
  //                           `Couple : (${
  //                             updatedData[0].primaryKey
  //                               ? `${updatedData[0].primaryKey} - `
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].hisLastName
  //                               ? `${updatedData[0].hisLastName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].hisFirstName
  //                               ? `, ${updatedData[0].hisFirstName}`
  //                               : ""
  //                           }` +
  //                           `${
  //                             updatedData[0] && updatedData[0].herFirstName
  //                               ? ` and  ${updatedData[0].herFirstName}`
  //                               : ""
  //                           })`,

  //                         oldhisFirstName: oldData[0].hisFirstName,
  //                         oldhisLastName: oldData[0].hisLastName,
  //                         oldhisEmail: oldData[0].hisEmail,
  //                         oldhisMobile: oldData[0].hisMobile,
  //                         oldherFirstName: oldData[0].herFirstName,
  //                         oldherLastName: oldData[0].herLastName,
  //                         oldherEmail: oldData[0].herEmail,
  //                         oldherMobile: oldData[0].herMobile,
  //                         oldcity: oldData[0].city,
  //                         oldstate: oldData[0].state,
  //                         oldzip: oldData[0].zip,
  //                         oldnotes: oldData[0].notes,
  //                         oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                         oldaddress: oldData[0].address,
  //                         oldallergies: oldData[0].allergies,
  //                         oldanniversary_date: oldData[0].anniversary_date,
  //                         oldhear_us: oldData[0].hear_us,
  //                         oldemergency_name1: oldData[0].emergency_name1,
  //                         oldemergency_relationship1:
  //                           oldData[0].emergency_relationship1,
  //                         oldemergency_phone1: oldData[0].emergency_phone1,
  //                         oldemergency_name2: oldData[0].emergency_name2,
  //                         oldemergency_relationship2:
  //                           oldData[0].emergency_relationship2,
  //                         oldemergency_phone2: oldData[0].emergency_phone2,
  //                         oldunder_age_35: oldData[0].under_age_35,
  //                         oldreferral_from_a_friend_or_family_member:
  //                           oldData[0].referral_from_a_friend_or_family_member,
  //                         newhisFirstName: updatedData[0].hisFirstName,
  //                         newhisLastName: updatedData[0].hisLastName,
  //                         newhisEmail: updatedData[0].hisEmail,
  //                         newhisMobile: updatedData[0].hisMobile,
  //                         newherFirstName: updatedData[0].herFirstName,
  //                         newherLastName: updatedData[0].herLastName,
  //                         newherEmail: updatedData[0].herEmail,
  //                         newherMobile: updatedData[0].herMobile,
  //                         newcity: updatedData[0].city,
  //                         newstate: updatedData[0].state,
  //                         newzip: updatedData[0].zip,
  //                         newnotes: updatedData[0].notes,
  //                         newparish: newP && newP.parish ? newP.parish : "",
  //                         newaddress: updatedData[0].address,
  //                         newallergies: updatedData[0].allergies,
  //                         newanniversary_date: updatedData[0].anniversary_date,
  //                         newhear_us: updatedData[0].hear_us,
  //                         newemergency_name1: updatedData[0].emergency_name1,
  //                         newemergency_relationship1:
  //                           updatedData[0].emergency_relationship1,
  //                         newemergency_phone1: updatedData[0].emergency_phone1,
  //                         newemergency_name2: updatedData[0].emergency_name2,
  //                         newemergency_relationship2:
  //                           updatedData[0].emergency_relationship2,
  //                         newemergency_phone2: updatedData[0].emergency_phone2,
  //                         newunder_age_35: updatedData[0].under_age_35,
  //                         newreferral_from_a_friend_or_family_member:
  //                           updatedData[0]
  //                             .referral_from_a_friend_or_family_member,
  //                       });
  //                     }
  //                   } else {
  //                     if (retreatRoastData[i].herMobile !== "") {
  //                       // console.log("iffffff hisemail; hai");
  //                       let normalizedHerMobile = retreatRoastData[
  //                         i
  //                       ].herMobile.replace(/\D/g, "");
  //                       let foundRecord = await coupleSchema.findAll({
  //                         where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHerMobile}'
  // `),
  //                         raw: true,
  //                       });
  //                       if (foundRecord.length > 0) {
  //                         let foundInRoaster = await retreatRoastSchema.findOne(
  //                           {
  //                             where: {
  //                               [Op.and]: [
  //                                 { coupleId: foundRecord[0].id },
  //                                 { retreatId: retreatRoastData[i].retreatId },
  //                               ],
  //                             },
  //                             raw: true,
  //                           }
  //                         );
  //                         if (foundInRoaster && foundInRoaster !== null) {
  //                           let retreatToUpdate = {
  //                             roleId: formatObj.roleId
  //                               ? formatObj.roleId
  //                               : null,
  //                             serviceId: formatObj.serviceId
  //                               ? formatObj.serviceId
  //                               : null,
  //                             talkId: formatObj.talkId
  //                               ? formatObj.talkId
  //                               : null,
  //                             attendeeType: retreatRoastData[i].attendeeType,
  //                           };
  //                           // Perform the update
  //                           let updatedData = await retreatRoastSchema.findAll({
  //                             where: {
  //                               [Op.and]: [
  //                                 { id: foundInRoaster.id },
  //                                 { retreatId: retreatRoastData[i].retreatId },
  //                                 { coupleId: foundRecord[0].id },
  //                               ],
  //                             },
  //                             raw: true,
  //                           });
  //                           if (updatedData[0].roleId) {
  //                             roleId = await roleSchema.findOne({
  //                               where: { id: updatedData[0].roleId },
  //                               raw: true,
  //                             });
  //                           }

  //                           // service
  //                           if (updatedData[0].serviceId) {
  //                             serviceId = await serviceSchema.findOne({
  //                               where: { id: updatedData[0].serviceId },
  //                               raw: true,
  //                             });
  //                           }

  //                           // talk
  //                           if (updatedData[0].talkId) {
  //                             talkId = await talkSchema.findOne({
  //                               where: { id: updatedData[0].talkId },
  //                               raw: true,
  //                             });
  //                           }

  //                           let updateOldRec = await retreatRoastSchema.update(
  //                             retreatToUpdate,
  //                             {
  //                               where: {
  //                                 [Op.and]: [
  //                                   { id: foundInRoaster.id },
  //                                   {
  //                                     retreatId: retreatRoastData[i].retreatId,
  //                                   },
  //                                   { coupleId: foundRecord[0].id },
  //                                 ],
  //                               },
  //                             }
  //                           );

  //                           // Check if any rows were affected by the update
  //                           if (updateOldRec[0] > 0) {
  //                             updateRoaster.push({
  //                               couple:
  //                                 `Couple : (${
  //                                   retreatRoastData[i].primaryKey
  //                                     ? `${retreatRoastData[i].primaryKey} - `
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisLastName
  //                                     ? `${retreatRoastData[i].hisLastName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisFirstName
  //                                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].herFirstName
  //                                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                     : ""
  //                                 })`,
  //                               oldRole: roleId ? roleId.role : "",
  //                               oldService: serviceId ? serviceId.service : "",
  //                               oldTalk: talkId ? talkId.talk : "",
  //                               oldAttendeeType:
  //                                 updatedData && updatedData[0].attendeeType,

  //                               newRole: retreatRoastData[i].role,
  //                               newService: retreatRoastData[i].service,
  //                               newTalk: retreatRoastData[i].talk,
  //                               newAttendeeType:
  //                                 retreatRoastData[i].attendeeType,
  //                             });
  //                           } else {
  //                             console.log("No rows were updated.");
  //                           }
  //                         } else {
  //                           let retreatData = {
  //                             roleId: formatObj.roleId
  //                               ? formatObj.roleId
  //                               : null,
  //                             serviceId: formatObj.serviceId
  //                               ? formatObj.serviceId
  //                               : null,
  //                             talkId: formatObj.talkId
  //                               ? formatObj.talkId
  //                               : null,
  //                             attendeeType: retreatRoastData[i].attendeeType,
  //                             coupleId: foundRecord[0].id,
  //                             retreatId: retreatRoastData[i].retreatId,
  //                           };
  //                           let newUser = new retreatRoastSchema(retreatData);
  //                           let succ = await newUser.save();
  //                           createdRoasters.push({
  //                             couple:
  //                               `Couple : (${
  //                                 retreatRoastData[i].primaryKey
  //                                   ? `${retreatRoastData[i].primaryKey} - `
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].hisLastName
  //                                   ? `${retreatRoastData[i].hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].hisFirstName
  //                                   ? `, ${retreatRoastData[i].hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 retreatRoastData[i] &&
  //                                 retreatRoastData[i].herFirstName
  //                                   ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                   : ""
  //                               })`,
  //                             Role: retreatRoastData[i].role,
  //                             Service: retreatRoastData[i].service,
  //                             Talk: retreatRoastData[i].talk,
  //                             Parish: retreatRoastData[i].parish,
  //                             AttendeeType: retreatRoastData[i].attendeeType,
  //                           });
  //                         }

  //                         coupleExist.push({
  //                           hisFirstName: retreatRoastData[i].hisFirstName,
  //                           hisLastName: retreatRoastData[i].hisLastName,
  //                           hisEmail: retreatRoastData[i].hisEmail,
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                           herFirstName: retreatRoastData[i].herFirstName,
  //                           herLastName: retreatRoastData[i].herLastName,
  //                           herEmail: retreatRoastData[i].herEmail,
  //                           herMobile: retreatRoastData[i].herMobile,
  //                           city: retreatRoastData[i].city,
  //                           state: retreatRoastData[i].state,
  //                           zip: retreatRoastData[i].zip,
  //                           notes: retreatRoastData[i].notes,
  //                           parishId: parishId && parishId.id,
  //                           address: retreatRoastData[i].address,
  //                           allergies: retreatRoastData[i].allergies,
  //                           anniversary_date:
  //                             retreatRoastData[i].anniversary_date === ""
  //                               ? null
  //                               : retreatRoastData[i].anniversary_date,
  //                           hear_us: retreatRoastData[i].hear_us
  //                             ? JSON.stringify(retreatRoastData[i].hear_us)
  //                             : [],
  //                           emergency_name1:
  //                             retreatRoastData[i].emergency_name1,
  //                           emergency_relationship1:
  //                             retreatRoastData[i].emergency_relationship1,
  //                           emergency_phone1:
  //                             retreatRoastData[i].emergency_phone1,
  //                           emergency_name2:
  //                             retreatRoastData[i].emergency_name2,
  //                           emergency_relationship2:
  //                             retreatRoastData[i].emergency_relationship2,
  //                           emergency_phone2:
  //                             retreatRoastData[i].emergency_phone2,
  //                           under_age_35:
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].under_age_35 &&
  //                             retreatRoastData[i].under_age_35 !== "" &&
  //                             retreatRoastData[i].under_age_35 !== null
  //                               ? retreatRoastData[i].under_age_35
  //                               : "NO",
  //                           referral_from_a_friend_or_family_member:
  //                             retreatRoastData[i]
  //                               .referral_from_a_friend_or_family_member,
  //                         });
  //                         let oldData = await coupleSchema.findAll({
  //                           where: { herMobile: retreatRoastData[i].herMobile },
  //                           raw: true,
  //                         });
  //                         if (oldData[0].parishId) {
  //                           oldP = await parishSchema.findOne({
  //                             where: { id: oldData[0].parishId },
  //                             raw: true,
  //                           });
  //                         }
  //                         let update = await coupleSchema.update(
  //                           coupleExist[0],
  //                           {
  //                             where: {
  //                               herMobile: retreatRoastData[i].herMobile,
  //                             },
  //                           }
  //                         );
  //                         if (update[0] > 0) {
  //                           // If rows were updated, fetch the updated data
  //                           let updatedData = await coupleSchema.findAll({
  //                             where: {
  //                               herMobile: retreatRoastData[i].herMobile,
  //                             },
  //                             raw: true,
  //                           });
  //                           if (updatedData[0].parishId) {
  //                             newP = await parishSchema.findOne({
  //                               where: { id: updatedData[0].parishId },
  //                               raw: true,
  //                             });
  //                           }
  //                           updateCouple.push({
  //                             couple:
  //                               `Couple : (${
  //                                 updatedData[0].primaryKey
  //                                   ? `${updatedData[0].primaryKey} - `
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].hisLastName
  //                                   ? `${updatedData[0].hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].hisFirstName
  //                                   ? `, ${updatedData[0].hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 updatedData[0] && updatedData[0].herFirstName
  //                                   ? ` and  ${updatedData[0].herFirstName}`
  //                                   : ""
  //                               })`,

  //                             oldhisFirstName: oldData[0].hisFirstName,
  //                             oldhisLastName: oldData[0].hisLastName,
  //                             oldhisEmail: oldData[0].hisEmail,
  //                             oldhisMobile: oldData[0].hisMobile,
  //                             oldherFirstName: oldData[0].herFirstName,
  //                             oldherLastName: oldData[0].herLastName,
  //                             oldherEmail: oldData[0].herEmail,
  //                             oldherMobile: oldData[0].herMobile,
  //                             oldcity: oldData[0].city,
  //                             oldstate: oldData[0].state,
  //                             oldzip: oldData[0].zip,
  //                             oldnotes: oldData[0].notes,
  //                             oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                             oldaddress: oldData[0].address,
  //                             oldallergies: oldData[0].allergies,
  //                             oldanniversary_date: oldData[0].anniversary_date,
  //                             oldhear_us: oldData[0].hear_us,
  //                             oldemergency_name1: oldData[0].emergency_name1,
  //                             oldemergency_relationship1:
  //                               oldData[0].emergency_relationship1,
  //                             oldemergency_phone1: oldData[0].emergency_phone1,
  //                             oldemergency_name2: oldData[0].emergency_name2,
  //                             oldemergency_relationship2:
  //                               oldData[0].emergency_relationship2,
  //                             oldemergency_phone2: oldData[0].emergency_phone2,
  //                             oldunder_age_35: oldData[0].under_age_35,
  //                             oldreferral_from_a_friend_or_family_member:
  //                               oldData[0]
  //                                 .referral_from_a_friend_or_family_member,
  //                             newhisFirstName: updatedData[0].hisFirstName,
  //                             newhisLastName: updatedData[0].hisLastName,
  //                             newhisEmail: updatedData[0].hisEmail,
  //                             newhisMobile: updatedData[0].hisMobile,
  //                             newherFirstName: updatedData[0].herFirstName,
  //                             newherLastName: updatedData[0].herLastName,
  //                             newherEmail: updatedData[0].herEmail,
  //                             newherMobile: updatedData[0].herMobile,
  //                             newcity: updatedData[0].city,
  //                             newstate: updatedData[0].state,
  //                             newzip: updatedData[0].zip,
  //                             newnotes: updatedData[0].notes,
  //                             newparish: newP && newP.parish ? newP.parish : "",
  //                             newaddress: updatedData[0].address,
  //                             newallergies: updatedData[0].allergies,
  //                             newanniversary_date:
  //                               updatedData[0].anniversary_date,
  //                             newhear_us: updatedData[0].hear_us,
  //                             newemergency_name1:
  //                               updatedData[0].emergency_name1,
  //                             newemergency_relationship1:
  //                               updatedData[0].emergency_relationship1,
  //                             newemergency_phone1:
  //                               updatedData[0].emergency_phone1,
  //                             newemergency_name2:
  //                               updatedData[0].emergency_name2,
  //                             newemergency_relationship2:
  //                               updatedData[0].emergency_relationship2,
  //                             newemergency_phone2:
  //                               updatedData[0].emergency_phone2,
  //                             newunder_age_35: updatedData[0].under_age_35,
  //                             newreferral_from_a_friend_or_family_member:
  //                               updatedData[0]
  //                                 .referral_from_a_friend_or_family_member,
  //                           });
  //                         }
  //                       } else {
  //                         coupleNotExist.push({
  //                           primaryKey: maxPrimaryKeyDB + 1,
  //                           hisFirstName: retreatRoastData[i].hisFirstName,
  //                           hisLastName: retreatRoastData[i].hisLastName,
  //                           hisEmail: retreatRoastData[i].hisEmail,
  //                           hisMobile: retreatRoastData[i].hisMobile,
  //                           herFirstName: retreatRoastData[i].herFirstName,
  //                           herLastName: retreatRoastData[i].herLastName,
  //                           herEmail: retreatRoastData[i].herEmail,
  //                           herMobile: retreatRoastData[i].herMobile,
  //                           city: retreatRoastData[i].city,
  //                           state: retreatRoastData[i].state,
  //                           zip: retreatRoastData[i].zip,
  //                           notes: retreatRoastData[i].notes,
  //                           parishId: parishId && parishId.id,
  //                           address: retreatRoastData[i].address,
  //                           allergies: retreatRoastData[i].allergies,
  //                           anniversary_date:
  //                             retreatRoastData[i].anniversary_date === ""
  //                               ? null
  //                               : retreatRoastData[i].anniversary_date,
  //                           hear_us: retreatRoastData[i].hear_us
  //                             ? JSON.stringify(retreatRoastData[i].hear_us)
  //                             : [],
  //                           emergency_name1:
  //                             retreatRoastData[i].emergency_name1,
  //                           emergency_relationship1:
  //                             retreatRoastData[i].emergency_relationship1,
  //                           emergency_phone1:
  //                             retreatRoastData[i].emergency_phone1,
  //                           emergency_name2:
  //                             retreatRoastData[i].emergency_name2,
  //                           emergency_relationship2:
  //                             retreatRoastData[i].emergency_relationship2,
  //                           emergency_phone2:
  //                             retreatRoastData[i].emergency_phone2,
  //                           under_age_35:
  //                             retreatRoastData[i] &&
  //                             retreatRoastData[i].under_age_35 &&
  //                             retreatRoastData[i].under_age_35 !== "" &&
  //                             retreatRoastData[i].under_age_35 !== null
  //                               ? retreatRoastData[i].under_age_35
  //                               : "NO",
  //                           referral_from_a_friend_or_family_member:
  //                             retreatRoastData[i]
  //                               .referral_from_a_friend_or_family_member,
  //                         });
  //                         // console.log(
  //                         //   "coupleNotExist- whenb primary key naa mile--->",
  //                         //   coupleNotExist
  //                         // );
  //                         if (
  //                           coupleNotExist.length &&
  //                           coupleNotExist.length > 0 &&
  //                           coupleNotExist !== ""
  //                         ) {
  //                           let newUser = new coupleSchema(coupleNotExist[0]);
  //                           let succ = await newUser.save();
  //                           // console.log("succ-- xcrteate couple->", succ.id);
  //                           if (succ.hisEmail) {
  //                             let randompass1 = Math.random()
  //                               .toString(36)
  //                               .slice(-8);
  //                             paswordForCouple1 =
  //                               randompass1.slice(0, 0) +
  //                               "p" +
  //                               randompass1.slice(1);
  //                           }
  //                           if (succ.herEmail) {
  //                             let randompass2 = Math.random()
  //                               .toString(36)
  //                               .slice(-8);
  //                             paswordForCouple2 =
  //                               randompass2.slice(0, 0) +
  //                               "p" +
  //                               randompass2.slice(1);
  //                           }
  //                           adminObj = [
  //                             {
  //                               userName: succ.hisFirstName,
  //                               email: succ.hisEmail,
  //                               password: paswordForCouple1
  //                                 ? paswordForCouple1
  //                                 : null,
  //                               accountStatus: "ACTIVE",
  //                               userRole: "COUPLE",
  //                               coupleId: succ.id,
  //                               emailVerified: 1,
  //                               firstName: succ.hisFirstName
  //                                 ? succ.hisFirstName
  //                                 : "",
  //                               lastName: succ.hisLastName
  //                                 ? succ.hisLastName
  //                                 : "",
  //                               mobile: succ.hisMobile ? succ.hisMobile : "",
  //                             },
  //                             {
  //                               userName: succ.herFirstName,
  //                               email: succ.herEmail,
  //                               password: paswordForCouple2
  //                                 ? paswordForCouple2
  //                                 : null,
  //                               accountStatus: "ACTIVE",
  //                               userRole: "COUPLE",
  //                               coupleId: succ.id,
  //                               emailVerified: 1,
  //                               firstName: succ.herFirstName
  //                                 ? succ.herFirstName
  //                                 : "",
  //                               lastName: succ.herLastName
  //                                 ? succ.herLastName
  //                                 : "",
  //                               mobile: succ.herMobile ? succ.herMobile : "",
  //                             },
  //                           ];
  //                           adminObj.map((succ) => {
  //                             userSchema.create(succ);
  //                           });
  //                           if (succ.hisEmail) {
  //                             let replData = SENDLOGINDETAILBYADMIN.replace(
  //                               /#firstName#/g,
  //                               succ.hisFirstName ? succ.hisFirstName : ""
  //                             )
  //                               .replace(
  //                                 /#lastName#/g,
  //                                 succ.hisLastName ? succ.hisLastName : ""
  //                               )
  //                               .replace(
  //                                 /#url#/g,
  //                                 `${process.env.SERVERADDRESS}/login`
  //                               )
  //                               .replace(/#email#/g, succ.hisEmail)
  //                               .replace(/#password#/g, paswordForCouple1);
  //                             Email.sendEmail(
  //                               succ.hisEmail,
  //                               "Couple Detail for Login- LOVESTRONG Marriage",
  //                               replData
  //                             );
  //                           }
  //                           if (succ.herEmail) {
  //                             let replData = SENDLOGINDETAILBYADMIN.replace(
  //                               /#firstName#/g,
  //                               succ.herFirstName ? succ.herFirstName : ""
  //                             )
  //                               .replace(
  //                                 /#lastName#/g,
  //                                 succ.herLastName ? succ.herLastName : ""
  //                               )
  //                               .replace(
  //                                 /#url#/g,
  //                                 `${process.env.SERVERADDRESS}/login`
  //                               )

  //                               .replace(/#email#/g, succ.herEmail)
  //                               .replace(/#password#/g, paswordForCouple2);
  //                             Email.sendEmail(
  //                               succ.herEmail,
  //                               "Couple Detail for Login- LOVESTRONG Marriage",
  //                               replData
  //                             );
  //                           }

  //                           createdCouple.push({
  //                             couple:
  //                               `Couple : (${
  //                                 succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.hisLastName
  //                                   ? `${succ.hisLastName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.hisFirstName
  //                                   ? `, ${succ.hisFirstName}`
  //                                   : ""
  //                               }` +
  //                               `${
  //                                 succ && succ.herFirstName
  //                                   ? ` and  ${succ.herFirstName}`
  //                                   : ""
  //                               })`,
  //                             primaryKey: succ.primaryKey,
  //                             hisFirstName: succ.hisFirstName,
  //                             hisLastName: succ.hisLastName,
  //                             hisEmail: succ.hisEmail,
  //                             hisMobile: succ.hisMobile,
  //                             herFirstName: succ.herFirstName,
  //                             herLastName: succ.herLastName,
  //                             herEmail: succ.herEmail,
  //                             herMobile: succ.herMobile,
  //                             city: succ.city,
  //                             state: succ.state,
  //                             zip: succ.zip,
  //                             notes: succ.notes,
  //                             parish: succ.parish,
  //                             address: succ.address,
  //                             allergies: succ.allergies,
  //                             anniversary_date: succ.anniversary_date,
  //                             hear_us: succ.hear_us,
  //                             emergency_name1: succ.emergency_name1,
  //                             emergency_relationship1:
  //                               succ.emergency_relationship1,
  //                             emergency_phone1: succ.emergency_phone1,
  //                             emergency_name2: succ.emergency_name2,
  //                             emergency_relationship2:
  //                               succ.emergency_relationship2,
  //                             emergency_phone2: succ.emergency_phone2,
  //                             under_age_35: succ.under_age_35,
  //                             referral_from_a_friend_or_family_member:
  //                               succ.referral_from_a_friend_or_family_member,
  //                           });

  //                           let foundRecordForRetreat =
  //                             await coupleSchema.findAll({
  //                               where: {
  //                                 id: succ.id,
  //                               },
  //                               raw: true,
  //                             });
  //                           if (
  //                             foundRecordForRetreat &&
  //                             foundRecordForRetreat.length > 0
  //                           ) {
  //                             let userDataToUpdate = {
  //                               roleId: formatObj.roleId
  //                                 ? formatObj.roleId
  //                                 : null,
  //                               serviceId: formatObj.serviceId
  //                                 ? formatObj.serviceId
  //                                 : null,
  //                               talkId: formatObj.talkId
  //                                 ? formatObj.talkId
  //                                 : null,
  //                               attendeeType: retreatRoastData[i].attendeeType,
  //                               coupleId:
  //                                 foundRecordForRetreat &&
  //                                 foundRecordForRetreat[0].id,
  //                               retreatId: retreatRoastData[i].retreatId,
  //                             };
  //                             let newRoaster = new retreatRoastSchema(
  //                               userDataToUpdate
  //                             );
  //                             let succ1 = await newRoaster.save();
  //                             createdRoasters.push({
  //                               couple:
  //                                 `Couple : (${
  //                                   retreatRoastData[i].primaryKey
  //                                     ? `${retreatRoastData[i].primaryKey} - `
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisLastName
  //                                     ? `${retreatRoastData[i].hisLastName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].hisFirstName
  //                                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                                     : ""
  //                                 }` +
  //                                 `${
  //                                   retreatRoastData[i] &&
  //                                   retreatRoastData[i].herFirstName
  //                                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                                     : ""
  //                                 })`,
  //                               Role: retreatRoastData[i].role,
  //                               Service: retreatRoastData[i].service,
  //                               Talk: retreatRoastData[i].talk,
  //                               Parish: retreatRoastData[i].parish,
  //                               AttendeeType: retreatRoastData[i].attendeeType,
  //                             });
  //                           }
  //                         }
  //                       }
  //                     }
  //                   }
  //                 }
  //               }
  //             }
  //           }
  //         }
  //       }

  //       if (retreatRoastData[i].hisMobile !== "") {
  //         let normalizedHisMobile = retreatRoastData[i].hisMobile.replace(
  //           /\D/g,
  //           ""
  //         );
  //         let foundRecord = await coupleSchema.findAll({
  //           where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHisMobile}'
  // `),
  //           raw: true,
  //         });

  //         if (foundRecord.length > 0) {
  //           let foundInRoaster = await retreatRoastSchema.findOne({
  //             where: {
  //               [Op.and]: [
  //                 { coupleId: foundRecord[0].id },
  //                 { retreatId: retreatRoastData[i].retreatId },
  //               ],
  //             },
  //             raw: true,
  //           });

  //           if (foundInRoaster && foundInRoaster !== null) {
  //             let retreatToUpdate = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //             };
  //             // Perform the update
  //             let updatedData = await retreatRoastSchema.findAll({
  //               where: {
  //                 [Op.and]: [
  //                   { id: foundInRoaster.id },
  //                   { retreatId: retreatRoastData[i].retreatId },
  //                   { coupleId: foundRecord[0].id },
  //                 ],
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].roleId) {
  //               roleId = await roleSchema.findOne({
  //                 where: { id: updatedData[0].roleId },
  //                 raw: true,
  //               });
  //             }

  //             // service
  //             if (updatedData[0].serviceId) {
  //               serviceId = await serviceSchema.findOne({
  //                 where: { id: updatedData[0].serviceId },
  //                 raw: true,
  //               });
  //             }

  //             // talk
  //             if (updatedData[0].talkId) {
  //               talkId = await talkSchema.findOne({
  //                 where: { id: updatedData[0].talkId },
  //                 raw: true,
  //               });
  //             }

  //             let updateOldRec = await retreatRoastSchema.update(
  //               retreatToUpdate,
  //               {
  //                 where: {
  //                   [Op.and]: [
  //                     { id: foundInRoaster.id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                     { coupleId: foundRecord[0].id },
  //                   ],
  //                 },
  //               }
  //             );

  //             // Check if any rows were affected by the update
  //             if (updateOldRec[0] > 0) {
  //               updateRoaster.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 oldRole: roleId ? roleId.role : "",
  //                 oldService: serviceId ? serviceId.service : "",
  //                 oldTalk: talkId ? talkId.talk : "",
  //                 oldAttendeeType: updatedData && updatedData[0].attendeeType,

  //                 newRole: retreatRoastData[i].role,
  //                 newService: retreatRoastData[i].service,
  //                 newTalk: retreatRoastData[i].talk,
  //                 newAttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             } else {
  //               console.log("No rows were updated.");
  //             }
  //           } else {
  //             let retreatData = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //               coupleId: foundRecord[0].id,
  //               retreatId: retreatRoastData[i].retreatId,
  //             };
  //             let newUser = new retreatRoastSchema(retreatData);
  //             let succ = await newUser.save();
  //             createdRoasters.push({
  //               couple:
  //                 `Couple : (${
  //                   retreatRoastData[i].primaryKey
  //                     ? `${retreatRoastData[i].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                     ? `${retreatRoastData[i].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                     : ""
  //                 })`,
  //               Role: retreatRoastData[i].role,
  //               Service: retreatRoastData[i].service,
  //               Talk: retreatRoastData[i].talk,
  //               Parish: retreatRoastData[i].parish,
  //               AttendeeType: retreatRoastData[i].attendeeType,
  //             });
  //           }

  //           coupleExist.push({
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,
  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });
  //           let oldData = await coupleSchema.findAll({
  //             where: { hisMobile: retreatRoastData[i].hisMobile },
  //             raw: true,
  //           });
  //           if (oldData[0].parishId) {
  //             oldP = await parishSchema.findOne({
  //               where: { id: oldData[0].parishId },
  //               raw: true,
  //             });
  //           }
  //           let update = await coupleSchema.update(coupleExist[0], {
  //             where: {
  //               hisMobile: retreatRoastData[i].hisMobile,
  //             },
  //           });

  //           if (update[0] > 0) {
  //             // If rows were updated, fetch the updated data
  //             let updatedData = await coupleSchema.findAll({
  //               where: {
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].parishId) {
  //               newP = await parishSchema.findOne({
  //                 where: { id: updatedData[0].parishId },
  //                 raw: true,
  //               });
  //             }
  //             updateCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   updatedData[0].primaryKey
  //                     ? `${updatedData[0].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisLastName
  //                     ? `${updatedData[0].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisFirstName
  //                     ? `, ${updatedData[0].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].herFirstName
  //                     ? ` and  ${updatedData[0].herFirstName}`
  //                     : ""
  //                 })`,

  //               oldhisFirstName: oldData[0].hisFirstName,
  //               oldhisLastName: oldData[0].hisLastName,
  //               oldhisEmail: oldData[0].hisEmail,
  //               oldhisMobile: oldData[0].hisMobile,
  //               oldherFirstName: oldData[0].herFirstName,
  //               oldherLastName: oldData[0].herLastName,
  //               oldherEmail: oldData[0].herEmail,
  //               oldherMobile: oldData[0].herMobile,
  //               oldcity: oldData[0].city,
  //               oldstate: oldData[0].state,
  //               oldzip: oldData[0].zip,
  //               oldnotes: oldData[0].notes,
  //               oldparish: oldP && oldP.parish ? oldP.parish : "",
  //               oldaddress: oldData[0].address,
  //               oldallergies: oldData[0].allergies,
  //               oldanniversary_date: oldData[0].anniversary_date,
  //               oldhear_us: oldData[0].hear_us,
  //               oldemergency_name1: oldData[0].emergency_name1,
  //               oldemergency_relationship1: oldData[0].emergency_relationship1,
  //               oldemergency_phone1: oldData[0].emergency_phone1,
  //               oldemergency_name2: oldData[0].emergency_name2,
  //               oldemergency_relationship2: oldData[0].emergency_relationship2,
  //               oldemergency_phone2: oldData[0].emergency_phone2,
  //               oldunder_age_35: oldData[0].under_age_35,
  //               oldreferral_from_a_friend_or_family_member:
  //                 oldData[0].referral_from_a_friend_or_family_member,
  //               newhisFirstName: updatedData[0].hisFirstName,
  //               newhisLastName: updatedData[0].hisLastName,
  //               newhisEmail: updatedData[0].hisEmail,
  //               newhisMobile: updatedData[0].hisMobile,
  //               newherFirstName: updatedData[0].herFirstName,
  //               newherLastName: updatedData[0].herLastName,
  //               newherEmail: updatedData[0].herEmail,
  //               newherMobile: updatedData[0].herMobile,
  //               newcity: updatedData[0].city,
  //               newstate: updatedData[0].state,
  //               newzip: updatedData[0].zip,
  //               newnotes: updatedData[0].notes,
  //               newparish: newP && newP.parish ? newP.parish : "",
  //               newaddress: updatedData[0].address,
  //               newallergies: updatedData[0].allergies,
  //               newanniversary_date: updatedData[0].anniversary_date,
  //               newhear_us: updatedData[0].hear_us,
  //               newemergency_name1: updatedData[0].emergency_name1,
  //               newemergency_relationship1:
  //                 updatedData[0].emergency_relationship1,
  //               newemergency_phone1: updatedData[0].emergency_phone1,
  //               newemergency_name2: updatedData[0].emergency_name2,
  //               newemergency_relationship2:
  //                 updatedData[0].emergency_relationship2,
  //               newemergency_phone2: updatedData[0].emergency_phone2,
  //               newunder_age_35: updatedData[0].under_age_35,
  //               newreferral_from_a_friend_or_family_member:
  //                 updatedData[0].referral_from_a_friend_or_family_member,
  //             });
  //           }
  //         } else {
  //           if (retreatRoastData[i].herMobile !== "") {
  //             // console.log("iffffff hisemail; hai");
  //             let normalizedHerMobile = retreatRoastData[i].herMobile.replace(
  //               /\D/g,
  //               ""
  //             );
  //             let foundRecord = await coupleSchema.findAll({
  //               where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHerMobile}'
  // `),
  //               raw: true,
  //             });

  //             if (foundRecord.length > 0) {
  //               let foundInRoaster = await retreatRoastSchema.findOne({
  //                 where: {
  //                   [Op.and]: [
  //                     { coupleId: foundRecord[0].id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                   ],
  //                 },
  //                 raw: true,
  //               });
  //               if (foundInRoaster && foundInRoaster !== null) {
  //                 let retreatToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                 };
  //                 // Perform the update
  //                 let updatedData = await retreatRoastSchema.findAll({
  //                   where: {
  //                     [Op.and]: [
  //                       { id: foundInRoaster.id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                       { coupleId: foundRecord[0].id },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].roleId) {
  //                   roleId = await roleSchema.findOne({
  //                     where: { id: updatedData[0].roleId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // service
  //                 if (updatedData[0].serviceId) {
  //                   serviceId = await serviceSchema.findOne({
  //                     where: { id: updatedData[0].serviceId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // talk
  //                 if (updatedData[0].talkId) {
  //                   talkId = await talkSchema.findOne({
  //                     where: { id: updatedData[0].talkId },
  //                     raw: true,
  //                   });
  //                 }

  //                 let updateOldRec = await retreatRoastSchema.update(
  //                   retreatToUpdate,
  //                   {
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         {
  //                           retreatId: retreatRoastData[i].retreatId,
  //                         },
  //                         { coupleId: foundRecord[0].id },
  //                       ],
  //                     },
  //                   }
  //                 );

  //                 // Check if any rows were affected by the update
  //                 if (updateOldRec[0] > 0) {
  //                   updateRoaster.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     oldRole: roleId ? roleId.role : "",
  //                     oldService: serviceId ? serviceId.service : "",
  //                     oldTalk: talkId ? talkId.talk : "",
  //                     oldAttendeeType:
  //                       updatedData && updatedData[0].attendeeType,

  //                     newRole: retreatRoastData[i].role,
  //                     newService: retreatRoastData[i].service,
  //                     newTalk: retreatRoastData[i].talk,
  //                     newAttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 } else {
  //                   console.log("No rows were updated.");
  //                 }
  //               } else {
  //                 let retreatData = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId: foundRecord[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newUser = new retreatRoastSchema(retreatData);
  //                 let succ = await newUser.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and  ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }

  //               coupleExist.push({
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               let oldData = await coupleSchema.findAll({
  //                 where: { herMobile: retreatRoastData[i].herMobile },
  //                 raw: true,
  //               });
  //               if (oldData[0].parishId) {
  //                 oldP = await parishSchema.findOne({
  //                   where: { id: oldData[0].parishId },
  //                   raw: true,
  //                 });
  //               }
  //               let update = await coupleSchema.update(coupleExist[0], {
  //                 where: {
  //                   herMobile: retreatRoastData[i].herMobile,
  //                 },
  //               });
  //               if (update[0] > 0) {
  //                 // If rows were updated, fetch the updated data
  //                 let updatedData = await coupleSchema.findAll({
  //                   where: {
  //                     herMobile: retreatRoastData[i].herMobile,
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].parishId) {
  //                   newP = await parishSchema.findOne({
  //                     where: { id: updatedData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 updateCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       updatedData[0].primaryKey
  //                         ? `${updatedData[0].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisLastName
  //                         ? `${updatedData[0].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisFirstName
  //                         ? `, ${updatedData[0].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].herFirstName
  //                         ? ` and  ${updatedData[0].herFirstName}`
  //                         : ""
  //                     })`,

  //                   oldhisFirstName: oldData[0].hisFirstName,
  //                   oldhisLastName: oldData[0].hisLastName,
  //                   oldhisEmail: oldData[0].hisEmail,
  //                   oldhisMobile: oldData[0].hisMobile,
  //                   oldherFirstName: oldData[0].herFirstName,
  //                   oldherLastName: oldData[0].herLastName,
  //                   oldherEmail: oldData[0].herEmail,
  //                   oldherMobile: oldData[0].herMobile,
  //                   oldcity: oldData[0].city,
  //                   oldstate: oldData[0].state,
  //                   oldzip: oldData[0].zip,
  //                   oldnotes: oldData[0].notes,
  //                   oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                   oldaddress: oldData[0].address,
  //                   oldallergies: oldData[0].allergies,
  //                   oldanniversary_date: oldData[0].anniversary_date,
  //                   oldhear_us: oldData[0].hear_us,
  //                   oldemergency_name1: oldData[0].emergency_name1,
  //                   oldemergency_relationship1:
  //                     oldData[0].emergency_relationship1,
  //                   oldemergency_phone1: oldData[0].emergency_phone1,
  //                   oldemergency_name2: oldData[0].emergency_name2,
  //                   oldemergency_relationship2:
  //                     oldData[0].emergency_relationship2,
  //                   oldemergency_phone2: oldData[0].emergency_phone2,
  //                   oldunder_age_35: oldData[0].under_age_35,
  //                   oldreferral_from_a_friend_or_family_member:
  //                     oldData[0].referral_from_a_friend_or_family_member,
  //                   newhisFirstName: updatedData[0].hisFirstName,
  //                   newhisLastName: updatedData[0].hisLastName,
  //                   newhisEmail: updatedData[0].hisEmail,
  //                   newhisMobile: updatedData[0].hisMobile,
  //                   newherFirstName: updatedData[0].herFirstName,
  //                   newherLastName: updatedData[0].herLastName,
  //                   newherEmail: updatedData[0].herEmail,
  //                   newherMobile: updatedData[0].herMobile,
  //                   newcity: updatedData[0].city,
  //                   newstate: updatedData[0].state,
  //                   newzip: updatedData[0].zip,
  //                   newnotes: updatedData[0].notes,
  //                   newparish: newP && newP.parish ? newP.parish : "",
  //                   newaddress: updatedData[0].address,
  //                   newallergies: updatedData[0].allergies,
  //                   newanniversary_date: updatedData[0].anniversary_date,
  //                   newhear_us: updatedData[0].hear_us,
  //                   newemergency_name1: updatedData[0].emergency_name1,
  //                   newemergency_relationship1:
  //                     updatedData[0].emergency_relationship1,
  //                   newemergency_phone1: updatedData[0].emergency_phone1,
  //                   newemergency_name2: updatedData[0].emergency_name2,
  //                   newemergency_relationship2:
  //                     updatedData[0].emergency_relationship2,
  //                   newemergency_phone2: updatedData[0].emergency_phone2,
  //                   newunder_age_35: updatedData[0].under_age_35,
  //                   newreferral_from_a_friend_or_family_member:
  //                     updatedData[0].referral_from_a_friend_or_family_member,
  //                 });
  //               }
  //             } else {
  //               coupleNotExist.push({
  //                 primaryKey: maxPrimaryKeyDB + 1,
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               // console.log(
  //               //   "coupleNotExist- whenb primary key naa mile--->",
  //               //   coupleNotExist
  //               // );
  //               if (
  //                 coupleNotExist.length &&
  //                 coupleNotExist.length > 0 &&
  //                 coupleNotExist !== ""
  //               ) {
  //                 let newUser = new coupleSchema(coupleNotExist[0]);
  //                 let succ = await newUser.save();
  //                 // console.log("succ-- xcrteate couple->", succ.id);
  //                 if (succ.hisEmail) {
  //                   let randompass1 = Math.random().toString(36).slice(-8);
  //                   paswordForCouple1 =
  //                     randompass1.slice(0, 0) + "p" + randompass1.slice(1);
  //                 }
  //                 if (succ.herEmail) {
  //                   let randompass2 = Math.random().toString(36).slice(-8);
  //                   paswordForCouple2 =
  //                     randompass2.slice(0, 0) + "p" + randompass2.slice(1);
  //                 }
  //                 adminObj = [
  //                   {
  //                     userName: succ.hisFirstName,
  //                     email: succ.hisEmail,
  //                     password: paswordForCouple1 ? paswordForCouple1 : null,
  //                     accountStatus: "ACTIVE",
  //                     userRole: "COUPLE",
  //                     coupleId: succ.id,
  //                     emailVerified: 1,
  //                     firstName: succ.hisFirstName ? succ.hisFirstName : "",
  //                     lastName: succ.hisLastName ? succ.hisLastName : "",
  //                     mobile: succ.hisMobile ? succ.hisMobile : "",
  //                   },
  //                   {
  //                     userName: succ.herFirstName,
  //                     email: succ.herEmail,
  //                     password: paswordForCouple2 ? paswordForCouple2 : null,
  //                     accountStatus: "ACTIVE",
  //                     userRole: "COUPLE",
  //                     coupleId: succ.id,
  //                     emailVerified: 1,
  //                     firstName: succ.herFirstName ? succ.herFirstName : "",
  //                     lastName: succ.herLastName ? succ.herLastName : "",
  //                     mobile: succ.herMobile ? succ.herMobile : "",
  //                   },
  //                 ];
  //                 adminObj.map((succ) => {
  //                   userSchema.create(succ);
  //                 });
  //                 if (succ.hisEmail) {
  //                   let replData = SENDLOGINDETAILBYADMIN.replace(
  //                     /#firstName#/g,
  //                     succ.hisFirstName ? succ.hisFirstName : ""
  //                   )
  //                     .replace(
  //                       /#lastName#/g,
  //                       succ.hisLastName ? succ.hisLastName : ""
  //                     )
  //                     .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
  //                     .replace(/#email#/g, succ.hisEmail)
  //                     .replace(/#password#/g, paswordForCouple1);
  //                   Email.sendEmail(
  //                     succ.hisEmail,
  //                     "Couple Detail for Login- LOVESTRONG Marriage",
  //                     replData
  //                   );
  //                 }
  //                 if (succ.herEmail) {
  //                   let replData = SENDLOGINDETAILBYADMIN.replace(
  //                     /#firstName#/g,
  //                     succ.herFirstName ? succ.herFirstName : ""
  //                   )
  //                     .replace(
  //                       /#lastName#/g,
  //                       succ.herLastName ? succ.herLastName : ""
  //                     )
  //                     .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

  //                     .replace(/#email#/g, succ.herEmail)
  //                     .replace(/#password#/g, paswordForCouple2);
  //                   Email.sendEmail(
  //                     succ.herEmail,
  //                     "Couple Detail for Login- LOVESTRONG Marriage",
  //                     replData
  //                   );
  //                 }

  //                 createdCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                     }` +
  //                     `${
  //                       succ && succ.hisLastName ? `${succ.hisLastName}` : ""
  //                     }` +
  //                     `${
  //                       succ && succ.hisFirstName
  //                         ? `, ${succ.hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       succ && succ.herFirstName
  //                         ? ` and  ${succ.herFirstName}`
  //                         : ""
  //                     })`,
  //                   primaryKey: succ.primaryKey,
  //                   hisFirstName: succ.hisFirstName,
  //                   hisLastName: succ.hisLastName,
  //                   hisEmail: succ.hisEmail,
  //                   hisMobile: succ.hisMobile,
  //                   herFirstName: succ.herFirstName,
  //                   herLastName: succ.herLastName,
  //                   herEmail: succ.herEmail,
  //                   herMobile: succ.herMobile,
  //                   city: succ.city,
  //                   state: succ.state,
  //                   zip: succ.zip,
  //                   notes: succ.notes,
  //                   parish: succ.parish,
  //                   address: succ.address,
  //                   allergies: succ.allergies,
  //                   anniversary_date: succ.anniversary_date,
  //                   hear_us: succ.hear_us,
  //                   emergency_name1: succ.emergency_name1,
  //                   emergency_relationship1: succ.emergency_relationship1,
  //                   emergency_phone1: succ.emergency_phone1,
  //                   emergency_name2: succ.emergency_name2,
  //                   emergency_relationship2: succ.emergency_relationship2,
  //                   emergency_phone2: succ.emergency_phone2,
  //                   under_age_35: succ.under_age_35,
  //                   referral_from_a_friend_or_family_member:
  //                     succ.referral_from_a_friend_or_family_member,
  //                 });

  //                 let foundRecordForRetreat = await coupleSchema.findAll({
  //                   where: {
  //                     id: succ.id,
  //                   },
  //                   raw: true,
  //                 });
  //                 if (
  //                   foundRecordForRetreat &&
  //                   foundRecordForRetreat.length > 0
  //                 ) {
  //                   let userDataToUpdate = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                     coupleId:
  //                       foundRecordForRetreat && foundRecordForRetreat[0].id,
  //                     retreatId: retreatRoastData[i].retreatId,
  //                   };
  //                   let newRoaster = new retreatRoastSchema(userDataToUpdate);
  //                   let succ1 = await newRoaster.save();
  //                   createdRoasters.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     Role: retreatRoastData[i].role,
  //                     Service: retreatRoastData[i].service,
  //                     Talk: retreatRoastData[i].talk,
  //                     Parish: retreatRoastData[i].parish,
  //                     AttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 }
  //               }
  //             }
  //           } else {
  //             coupleNotExist.push({
  //               primaryKey: maxPrimaryKeyDB + 1,
  //               hisFirstName: retreatRoastData[i].hisFirstName,
  //               hisLastName: retreatRoastData[i].hisLastName,
  //               hisEmail: retreatRoastData[i].hisEmail,
  //               hisMobile: retreatRoastData[i].hisMobile,
  //               herFirstName: retreatRoastData[i].herFirstName,
  //               herLastName: retreatRoastData[i].herLastName,
  //               herEmail: retreatRoastData[i].herEmail,
  //               herMobile: retreatRoastData[i].herMobile,
  //               city: retreatRoastData[i].city,
  //               state: retreatRoastData[i].state,
  //               zip: retreatRoastData[i].zip,
  //               notes: retreatRoastData[i].notes,
  //               parishId: parishId && parishId.id,
  //               address: retreatRoastData[i].address,
  //               allergies: retreatRoastData[i].allergies,
  //               anniversary_date:
  //                 retreatRoastData[i].anniversary_date === ""
  //                   ? null
  //                   : retreatRoastData[i].anniversary_date,
  //               hear_us: retreatRoastData[i].hear_us
  //                 ? JSON.stringify(retreatRoastData[i].hear_us)
  //                 : [],
  //               emergency_name1: retreatRoastData[i].emergency_name1,
  //               emergency_relationship1:
  //                 retreatRoastData[i].emergency_relationship1,
  //               emergency_phone1: retreatRoastData[i].emergency_phone1,
  //               emergency_name2: retreatRoastData[i].emergency_name2,
  //               emergency_relationship2:
  //                 retreatRoastData[i].emergency_relationship2,
  //               emergency_phone2: retreatRoastData[i].emergency_phone2,
  //               under_age_35:
  //                 retreatRoastData[i] &&
  //                 retreatRoastData[i].under_age_35 &&
  //                 retreatRoastData[i].under_age_35 !== "" &&
  //                 retreatRoastData[i].under_age_35 !== null
  //                   ? retreatRoastData[i].under_age_35
  //                   : "NO",
  //               referral_from_a_friend_or_family_member:
  //                 retreatRoastData[i].referral_from_a_friend_or_family_member,
  //             });
  //             // console.log(
  //             //   "coupleNotExist- whenb primary key naa mile--->",
  //             //   coupleNotExist
  //             // );
  //             if (
  //               coupleNotExist.length &&
  //               coupleNotExist.length > 0 &&
  //               coupleNotExist !== ""
  //             ) {
  //               let newUser = new coupleSchema(coupleNotExist[0]);
  //               let succ = await newUser.save();
  //               // console.log("succ-- xcrteate couple->", succ.id);
  //               if (succ.hisEmail) {
  //                 let randompass1 = Math.random().toString(36).slice(-8);
  //                 paswordForCouple1 =
  //                   randompass1.slice(0, 0) + "p" + randompass1.slice(1);
  //               }
  //               if (succ.herEmail) {
  //                 let randompass2 = Math.random().toString(36).slice(-8);
  //                 paswordForCouple2 =
  //                   randompass2.slice(0, 0) + "p" + randompass2.slice(1);
  //               }
  //               adminObj = [
  //                 {
  //                   userName: succ.hisFirstName,
  //                   email: succ.hisEmail,
  //                   password: paswordForCouple1 ? paswordForCouple1 : null,
  //                   accountStatus: "ACTIVE",
  //                   userRole: "COUPLE",
  //                   coupleId: succ.id,
  //                   emailVerified: 1,
  //                   firstName: succ.hisFirstName ? succ.hisFirstName : "",
  //                   lastName: succ.hisLastName ? succ.hisLastName : "",
  //                   mobile: succ.hisMobile ? succ.hisMobile : "",
  //                 },
  //                 {
  //                   userName: succ.herFirstName,
  //                   email: succ.herEmail,
  //                   password: paswordForCouple2 ? paswordForCouple2 : null,
  //                   accountStatus: "ACTIVE",
  //                   userRole: "COUPLE",
  //                   coupleId: succ.id,
  //                   emailVerified: 1,
  //                   firstName: succ.herFirstName ? succ.herFirstName : "",
  //                   lastName: succ.herLastName ? succ.herLastName : "",
  //                   mobile: succ.herMobile ? succ.herMobile : "",
  //                 },
  //               ];
  //               adminObj.map((succ) => {
  //                 userSchema.create(succ);
  //               });
  //               if (succ.hisEmail) {
  //                 let replData = SENDLOGINDETAILBYADMIN.replace(
  //                   /#firstName#/g,
  //                   succ.hisFirstName ? succ.hisFirstName : ""
  //                 )
  //                   .replace(
  //                     /#lastName#/g,
  //                     succ.hisLastName ? succ.hisLastName : ""
  //                   )
  //                   .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
  //                   .replace(/#email#/g, succ.hisEmail)
  //                   .replace(/#password#/g, paswordForCouple1);
  //                 Email.sendEmail(
  //                   succ.hisEmail,
  //                   "Couple Detail for Login- LOVESTRONG Marriage",
  //                   replData
  //                 );
  //               }
  //               if (succ.herEmail) {
  //                 let replData = SENDLOGINDETAILBYADMIN.replace(
  //                   /#firstName#/g,
  //                   succ.herFirstName ? succ.herFirstName : ""
  //                 )
  //                   .replace(
  //                     /#lastName#/g,
  //                     succ.herLastName ? succ.herLastName : ""
  //                   )
  //                   .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

  //                   .replace(/#email#/g, succ.herEmail)
  //                   .replace(/#password#/g, paswordForCouple2);
  //                 Email.sendEmail(
  //                   succ.herEmail,
  //                   "Couple Detail for Login- LOVESTRONG Marriage",
  //                   replData
  //                 );
  //               }

  //               createdCouple.push({
  //                 couple:
  //                   `Couple : (${
  //                     succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                   }` +
  //                   `${succ && succ.hisLastName ? `${succ.hisLastName}` : ""}` +
  //                   `${
  //                     succ && succ.hisFirstName ? `, ${succ.hisFirstName}` : ""
  //                   }` +
  //                   `${
  //                     succ && succ.herFirstName
  //                       ? ` and  ${succ.herFirstName}`
  //                       : ""
  //                   })`,
  //                 primaryKey: succ.primaryKey,
  //                 hisFirstName: succ.hisFirstName,
  //                 hisLastName: succ.hisLastName,
  //                 hisEmail: succ.hisEmail,
  //                 hisMobile: succ.hisMobile,
  //                 herFirstName: succ.herFirstName,
  //                 herLastName: succ.herLastName,
  //                 herEmail: succ.herEmail,
  //                 herMobile: succ.herMobile,
  //                 city: succ.city,
  //                 state: succ.state,
  //                 zip: succ.zip,
  //                 notes: succ.notes,
  //                 parish: succ.parish,
  //                 address: succ.address,
  //                 allergies: succ.allergies,
  //                 anniversary_date: succ.anniversary_date,
  //                 hear_us: succ.hear_us,
  //                 emergency_name1: succ.emergency_name1,
  //                 emergency_relationship1: succ.emergency_relationship1,
  //                 emergency_phone1: succ.emergency_phone1,
  //                 emergency_name2: succ.emergency_name2,
  //                 emergency_relationship2: succ.emergency_relationship2,
  //                 emergency_phone2: succ.emergency_phone2,
  //                 under_age_35: succ.under_age_35,
  //                 referral_from_a_friend_or_family_member:
  //                   succ.referral_from_a_friend_or_family_member,
  //               });

  //               let foundRecordForRetreat = await coupleSchema.findAll({
  //                 where: {
  //                   id: succ.id,
  //                 },
  //                 raw: true,
  //               });
  //               if (foundRecordForRetreat && foundRecordForRetreat.length > 0) {
  //                 let userDataToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId:
  //                     foundRecordForRetreat && foundRecordForRetreat[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newRoaster = new retreatRoastSchema(userDataToUpdate);
  //                 let succ1 = await newRoaster.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and  ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }
  //             }
  //           }
  //         }
  //       }

  //       if (retreatRoastData[i].herMobile !== "") {
  //         let normalizedHerMobile = retreatRoastData[i].herMobile.replace(
  //           /\D/g,
  //           ""
  //         );
  //         let foundRecord = await coupleSchema.findAll({
  //           where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHerMobile}'
  // `),
  //           raw: true,
  //         });
  //         if (foundRecord.length > 0) {
  //           let foundInRoaster = await retreatRoastSchema.findOne({
  //             where: {
  //               [Op.and]: [
  //                 { coupleId: foundRecord[0].id },
  //                 { retreatId: retreatRoastData[i].retreatId },
  //               ],
  //             },
  //             raw: true,
  //           });

  //           if (foundInRoaster && foundInRoaster !== null) {
  //             let retreatToUpdate = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //             };
  //             // Perform the update
  //             let updatedData = await retreatRoastSchema.findAll({
  //               where: {
  //                 [Op.and]: [
  //                   { id: foundInRoaster.id },
  //                   { retreatId: retreatRoastData[i].retreatId },
  //                   { coupleId: foundRecord[0].id },
  //                 ],
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].roleId) {
  //               roleId = await roleSchema.findOne({
  //                 where: { id: updatedData[0].roleId },
  //                 raw: true,
  //               });
  //             }

  //             // service
  //             if (updatedData[0].serviceId) {
  //               serviceId = await serviceSchema.findOne({
  //                 where: { id: updatedData[0].serviceId },
  //                 raw: true,
  //               });
  //             }

  //             // talk
  //             if (updatedData[0].talkId) {
  //               talkId = await talkSchema.findOne({
  //                 where: { id: updatedData[0].talkId },
  //                 raw: true,
  //               });
  //             }

  //             let updateOldRec = await retreatRoastSchema.update(
  //               retreatToUpdate,
  //               {
  //                 where: {
  //                   [Op.and]: [
  //                     { id: foundInRoaster.id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                     { coupleId: foundRecord[0].id },
  //                   ],
  //                 },
  //               }
  //             );

  //             // Check if any rows were affected by the update
  //             if (updateOldRec[0] > 0) {
  //               updateRoaster.push({
  //                 couple:
  //                   `Couple : (${
  //                     retreatRoastData[i].primaryKey
  //                       ? `${retreatRoastData[i].primaryKey} - `
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                       ? `${retreatRoastData[i].hisLastName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                       ? `, ${retreatRoastData[i].hisFirstName}`
  //                       : ""
  //                   }` +
  //                   `${
  //                     retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                       ? ` and  ${retreatRoastData[i].herFirstName}`
  //                       : ""
  //                   })`,
  //                 oldRole: roleId ? roleId.role : "",
  //                 oldService: serviceId ? serviceId.service : "",
  //                 oldTalk: talkId ? talkId.talk : "",
  //                 oldAttendeeType: updatedData && updatedData[0].attendeeType,

  //                 newRole: retreatRoastData[i].role,
  //                 newService: retreatRoastData[i].service,
  //                 newTalk: retreatRoastData[i].talk,
  //                 newAttendeeType: retreatRoastData[i].attendeeType,
  //               });
  //             } else {
  //               console.log("No rows were updated.");
  //             }
  //           } else {
  //             let retreatData = {
  //               roleId: formatObj.roleId ? formatObj.roleId : null,
  //               serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //               talkId: formatObj.talkId ? formatObj.talkId : null,
  //               attendeeType: retreatRoastData[i].attendeeType,
  //               coupleId: foundRecord[0].id,
  //               retreatId: retreatRoastData[i].retreatId,
  //             };
  //             let newUser = new retreatRoastSchema(retreatData);
  //             let succ = await newUser.save();
  //             createdRoasters.push({
  //               couple:
  //                 `Couple : (${
  //                   retreatRoastData[i].primaryKey
  //                     ? `${retreatRoastData[i].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                     ? `${retreatRoastData[i].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                     ? `, ${retreatRoastData[i].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                     ? ` and  ${retreatRoastData[i].herFirstName}`
  //                     : ""
  //                 })`,
  //               Role: retreatRoastData[i].role,
  //               Service: retreatRoastData[i].service,
  //               Talk: retreatRoastData[i].talk,
  //               Parish: retreatRoastData[i].parish,
  //               AttendeeType: retreatRoastData[i].attendeeType,
  //             });
  //           }

  //           coupleExist.push({
  //             hisFirstName: retreatRoastData[i].hisFirstName,
  //             hisLastName: retreatRoastData[i].hisLastName,
  //             hisEmail: retreatRoastData[i].hisEmail,
  //             hisMobile: retreatRoastData[i].hisMobile,
  //             herFirstName: retreatRoastData[i].herFirstName,
  //             herLastName: retreatRoastData[i].herLastName,
  //             herEmail: retreatRoastData[i].herEmail,
  //             herMobile: retreatRoastData[i].herMobile,
  //             city: retreatRoastData[i].city,
  //             state: retreatRoastData[i].state,
  //             zip: retreatRoastData[i].zip,
  //             notes: retreatRoastData[i].notes,
  //             parishId: parishId && parishId.id,
  //             address: retreatRoastData[i].address,
  //             allergies: retreatRoastData[i].allergies,
  //             anniversary_date:
  //               retreatRoastData[i].anniversary_date === ""
  //                 ? null
  //                 : retreatRoastData[i].anniversary_date,
  //             hear_us: retreatRoastData[i].hear_us
  //               ? JSON.stringify(retreatRoastData[i].hear_us)
  //               : [],
  //             emergency_name1: retreatRoastData[i].emergency_name1,
  //             emergency_relationship1:
  //               retreatRoastData[i].emergency_relationship1,
  //             emergency_phone1: retreatRoastData[i].emergency_phone1,
  //             emergency_name2: retreatRoastData[i].emergency_name2,
  //             emergency_relationship2:
  //               retreatRoastData[i].emergency_relationship2,
  //             emergency_phone2: retreatRoastData[i].emergency_phone2,
  //             under_age_35:
  //               retreatRoastData[i] &&
  //               retreatRoastData[i].under_age_35 &&
  //               retreatRoastData[i].under_age_35 !== "" &&
  //               retreatRoastData[i].under_age_35 !== null
  //                 ? retreatRoastData[i].under_age_35
  //                 : "NO",
  //             referral_from_a_friend_or_family_member:
  //               retreatRoastData[i].referral_from_a_friend_or_family_member,
  //           });
  //           let oldData = await coupleSchema.findAll({
  //             where: { herMobile: retreatRoastData[i].herMobile },
  //             raw: true,
  //           });
  //           if (oldData[0].parishId) {
  //             oldP = await parishSchema.findOne({
  //               where: { id: oldData[0].parishId },
  //               raw: true,
  //             });
  //           }
  //           let update = await coupleSchema.update(coupleExist[0], {
  //             where: {
  //               herMobile: retreatRoastData[i].herMobile,
  //             },
  //           });

  //           if (update[0] > 0) {
  //             // If rows were updated, fetch the updated data
  //             let updatedData = await coupleSchema.findAll({
  //               where: {
  //                 herMobile: retreatRoastData[i].herMobile,
  //               },
  //               raw: true,
  //             });

  //             if (updatedData[0].parishId) {
  //               newP = await parishSchema.findOne({
  //                 where: { id: updatedData[0].parishId },
  //                 raw: true,
  //               });
  //             }
  //             updateCouple.push({
  //               couple:
  //                 `Couple : (${
  //                   updatedData[0].primaryKey
  //                     ? `${updatedData[0].primaryKey} - `
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisLastName
  //                     ? `${updatedData[0].hisLastName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].hisFirstName
  //                     ? `, ${updatedData[0].hisFirstName}`
  //                     : ""
  //                 }` +
  //                 `${
  //                   updatedData[0] && updatedData[0].herFirstName
  //                     ? ` and  ${updatedData[0].herFirstName}`
  //                     : ""
  //                 })`,

  //               oldhisFirstName: oldData[0].hisFirstName,
  //               oldhisLastName: oldData[0].hisLastName,
  //               oldhisEmail: oldData[0].hisEmail,
  //               oldhisMobile: oldData[0].hisMobile,
  //               oldherFirstName: oldData[0].herFirstName,
  //               oldherLastName: oldData[0].herLastName,
  //               oldherEmail: oldData[0].herEmail,
  //               oldherMobile: oldData[0].herMobile,
  //               oldcity: oldData[0].city,
  //               oldstate: oldData[0].state,
  //               oldzip: oldData[0].zip,
  //               oldnotes: oldData[0].notes,
  //               oldparish: oldP && oldP.parish ? oldP.parish : "",
  //               oldaddress: oldData[0].address,
  //               oldallergies: oldData[0].allergies,
  //               oldanniversary_date: oldData[0].anniversary_date,
  //               oldhear_us: oldData[0].hear_us,
  //               oldemergency_name1: oldData[0].emergency_name1,
  //               oldemergency_relationship1: oldData[0].emergency_relationship1,
  //               oldemergency_phone1: oldData[0].emergency_phone1,
  //               oldemergency_name2: oldData[0].emergency_name2,
  //               oldemergency_relationship2: oldData[0].emergency_relationship2,
  //               oldemergency_phone2: oldData[0].emergency_phone2,
  //               oldunder_age_35: oldData[0].under_age_35,
  //               oldreferral_from_a_friend_or_family_member:
  //                 oldData[0].referral_from_a_friend_or_family_member,
  //               newhisFirstName: updatedData[0].hisFirstName,
  //               newhisLastName: updatedData[0].hisLastName,
  //               newhisEmail: updatedData[0].hisEmail,
  //               newhisMobile: updatedData[0].hisMobile,
  //               newherFirstName: updatedData[0].herFirstName,
  //               newherLastName: updatedData[0].herLastName,
  //               newherEmail: updatedData[0].herEmail,
  //               newherMobile: updatedData[0].herMobile,
  //               newcity: updatedData[0].city,
  //               newstate: updatedData[0].state,
  //               newzip: updatedData[0].zip,
  //               newnotes: updatedData[0].notes,
  //               newparish: newP && newP.parish ? newP.parish : "",
  //               newaddress: updatedData[0].address,
  //               newallergies: updatedData[0].allergies,
  //               newanniversary_date: updatedData[0].anniversary_date,
  //               newhear_us: updatedData[0].hear_us,
  //               newemergency_name1: updatedData[0].emergency_name1,
  //               newemergency_relationship1:
  //                 updatedData[0].emergency_relationship1,
  //               newemergency_phone1: updatedData[0].emergency_phone1,
  //               newemergency_name2: updatedData[0].emergency_name2,
  //               newemergency_relationship2:
  //                 updatedData[0].emergency_relationship2,
  //               newemergency_phone2: updatedData[0].emergency_phone2,
  //               newunder_age_35: updatedData[0].under_age_35,
  //               newreferral_from_a_friend_or_family_member:
  //                 updatedData[0].referral_from_a_friend_or_family_member,
  //             });
  //           }
  //         } else {
  //           if (retreatRoastData[i].hisMobile !== "") {
  //             let normalizedHisMobile = retreatRoastData[i].hisMobile.replace(
  //               /\D/g,
  //               ""
  //             );
  //             let foundRecord = await coupleSchema.findAll({
  //               where: Sequelize.literal(`
  //   REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', ''), '.', ''), '+', '') = '${normalizedHisMobile}'
  // `),
  //               raw: true,
  //             });

  //             if (foundRecord.length > 0) {
  //               let foundInRoaster = await retreatRoastSchema.findOne({
  //                 where: {
  //                   [Op.and]: [
  //                     { coupleId: foundRecord[0].id },
  //                     { retreatId: retreatRoastData[i].retreatId },
  //                   ],
  //                 },
  //                 raw: true,
  //               });
  //               if (foundInRoaster && foundInRoaster !== null) {
  //                 let retreatToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                 };
  //                 // Perform the update
  //                 let updatedData = await retreatRoastSchema.findAll({
  //                   where: {
  //                     [Op.and]: [
  //                       { id: foundInRoaster.id },
  //                       { retreatId: retreatRoastData[i].retreatId },
  //                       { coupleId: foundRecord[0].id },
  //                     ],
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].roleId) {
  //                   roleId = await roleSchema.findOne({
  //                     where: { id: updatedData[0].roleId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // service
  //                 if (updatedData[0].serviceId) {
  //                   serviceId = await serviceSchema.findOne({
  //                     where: { id: updatedData[0].serviceId },
  //                     raw: true,
  //                   });
  //                 }

  //                 // talk
  //                 if (updatedData[0].talkId) {
  //                   talkId = await talkSchema.findOne({
  //                     where: { id: updatedData[0].talkId },
  //                     raw: true,
  //                   });
  //                 }

  //                 let updateOldRec = await retreatRoastSchema.update(
  //                   retreatToUpdate,
  //                   {
  //                     where: {
  //                       [Op.and]: [
  //                         { id: foundInRoaster.id },
  //                         {
  //                           retreatId: retreatRoastData[i].retreatId,
  //                         },
  //                         { coupleId: foundRecord[0].id },
  //                       ],
  //                     },
  //                   }
  //                 );

  //                 // Check if any rows were affected by the update
  //                 if (updateOldRec[0] > 0) {
  //                   updateRoaster.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     oldRole: roleId ? roleId.role : "",
  //                     oldService: serviceId ? serviceId.service : "",
  //                     oldTalk: talkId ? talkId.talk : "",
  //                     oldAttendeeType:
  //                       updatedData && updatedData[0].attendeeType,

  //                     newRole: retreatRoastData[i].role,
  //                     newService: retreatRoastData[i].service,
  //                     newTalk: retreatRoastData[i].talk,
  //                     newAttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 } else {
  //                   console.log("No rows were updated.");
  //                 }
  //               } else {
  //                 let retreatData = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId: foundRecord[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newUser = new retreatRoastSchema(retreatData);
  //                 let succ = await newUser.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and  ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }

  //               coupleExist.push({
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               let oldData = await coupleSchema.findAll({
  //                 where: { hisMobile: retreatRoastData[i].hisMobile },
  //                 raw: true,
  //               });
  //               if (oldData[0].parishId) {
  //                 oldP = await parishSchema.findOne({
  //                   where: { id: oldData[0].parishId },
  //                   raw: true,
  //                 });
  //               }
  //               let update = await coupleSchema.update(coupleExist[0], {
  //                 where: {
  //                   hisMobile: retreatRoastData[i].hisMobile,
  //                 },
  //               });
  //               if (update[0] > 0) {
  //                 // If rows were updated, fetch the updated data
  //                 let updatedData = await coupleSchema.findAll({
  //                   where: {
  //                     hisMobile: retreatRoastData[i].hisMobile,
  //                   },
  //                   raw: true,
  //                 });
  //                 if (updatedData[0].parishId) {
  //                   newP = await parishSchema.findOne({
  //                     where: { id: updatedData[0].parishId },
  //                     raw: true,
  //                   });
  //                 }
  //                 updateCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       updatedData[0].primaryKey
  //                         ? `${updatedData[0].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisLastName
  //                         ? `${updatedData[0].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].hisFirstName
  //                         ? `, ${updatedData[0].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       updatedData[0] && updatedData[0].herFirstName
  //                         ? ` and  ${updatedData[0].herFirstName}`
  //                         : ""
  //                     })`,

  //                   oldhisFirstName: oldData[0].hisFirstName,
  //                   oldhisLastName: oldData[0].hisLastName,
  //                   oldhisEmail: oldData[0].hisEmail,
  //                   oldhisMobile: oldData[0].hisMobile,
  //                   oldherFirstName: oldData[0].herFirstName,
  //                   oldherLastName: oldData[0].herLastName,
  //                   oldherEmail: oldData[0].herEmail,
  //                   oldherMobile: oldData[0].herMobile,
  //                   oldcity: oldData[0].city,
  //                   oldstate: oldData[0].state,
  //                   oldzip: oldData[0].zip,
  //                   oldnotes: oldData[0].notes,
  //                   oldparish: oldP && oldP.parish ? oldP.parish : "",
  //                   oldaddress: oldData[0].address,
  //                   oldallergies: oldData[0].allergies,
  //                   oldanniversary_date: oldData[0].anniversary_date,
  //                   oldhear_us: oldData[0].hear_us,
  //                   oldemergency_name1: oldData[0].emergency_name1,
  //                   oldemergency_relationship1:
  //                     oldData[0].emergency_relationship1,
  //                   oldemergency_phone1: oldData[0].emergency_phone1,
  //                   oldemergency_name2: oldData[0].emergency_name2,
  //                   oldemergency_relationship2:
  //                     oldData[0].emergency_relationship2,
  //                   oldemergency_phone2: oldData[0].emergency_phone2,
  //                   oldunder_age_35: oldData[0].under_age_35,
  //                   oldreferral_from_a_friend_or_family_member:
  //                     oldData[0].referral_from_a_friend_or_family_member,
  //                   newhisFirstName: updatedData[0].hisFirstName,
  //                   newhisLastName: updatedData[0].hisLastName,
  //                   newhisEmail: updatedData[0].hisEmail,
  //                   newhisMobile: updatedData[0].hisMobile,
  //                   newherFirstName: updatedData[0].herFirstName,
  //                   newherLastName: updatedData[0].herLastName,
  //                   newherEmail: updatedData[0].herEmail,
  //                   newherMobile: updatedData[0].herMobile,
  //                   newcity: updatedData[0].city,
  //                   newstate: updatedData[0].state,
  //                   newzip: updatedData[0].zip,
  //                   newnotes: updatedData[0].notes,
  //                   newparish: newP && newP.parish ? newP.parish : "",
  //                   newaddress: updatedData[0].address,
  //                   newallergies: updatedData[0].allergies,
  //                   newanniversary_date: updatedData[0].anniversary_date,
  //                   newhear_us: updatedData[0].hear_us,
  //                   newemergency_name1: updatedData[0].emergency_name1,
  //                   newemergency_relationship1:
  //                     updatedData[0].emergency_relationship1,
  //                   newemergency_phone1: updatedData[0].emergency_phone1,
  //                   newemergency_name2: updatedData[0].emergency_name2,
  //                   newemergency_relationship2:
  //                     updatedData[0].emergency_relationship2,
  //                   newemergency_phone2: updatedData[0].emergency_phone2,
  //                   newunder_age_35: updatedData[0].under_age_35,
  //                   newreferral_from_a_friend_or_family_member:
  //                     updatedData[0].referral_from_a_friend_or_family_member,
  //                 });
  //               }
  //             } else {
  //               coupleNotExist.push({
  //                 primaryKey: maxPrimaryKeyDB + 1,
  //                 hisFirstName: retreatRoastData[i].hisFirstName,
  //                 hisLastName: retreatRoastData[i].hisLastName,
  //                 hisEmail: retreatRoastData[i].hisEmail,
  //                 hisMobile: retreatRoastData[i].hisMobile,
  //                 herFirstName: retreatRoastData[i].herFirstName,
  //                 herLastName: retreatRoastData[i].herLastName,
  //                 herEmail: retreatRoastData[i].herEmail,
  //                 herMobile: retreatRoastData[i].herMobile,
  //                 city: retreatRoastData[i].city,
  //                 state: retreatRoastData[i].state,
  //                 zip: retreatRoastData[i].zip,
  //                 notes: retreatRoastData[i].notes,
  //                 parishId: parishId && parishId.id,
  //                 address: retreatRoastData[i].address,
  //                 allergies: retreatRoastData[i].allergies,
  //                 anniversary_date:
  //                   retreatRoastData[i].anniversary_date === ""
  //                     ? null
  //                     : retreatRoastData[i].anniversary_date,
  //                 hear_us: retreatRoastData[i].hear_us
  //                   ? JSON.stringify(retreatRoastData[i].hear_us)
  //                   : [],
  //                 emergency_name1: retreatRoastData[i].emergency_name1,
  //                 emergency_relationship1:
  //                   retreatRoastData[i].emergency_relationship1,
  //                 emergency_phone1: retreatRoastData[i].emergency_phone1,
  //                 emergency_name2: retreatRoastData[i].emergency_name2,
  //                 emergency_relationship2:
  //                   retreatRoastData[i].emergency_relationship2,
  //                 emergency_phone2: retreatRoastData[i].emergency_phone2,
  //                 under_age_35:
  //                   retreatRoastData[i] &&
  //                   retreatRoastData[i].under_age_35 &&
  //                   retreatRoastData[i].under_age_35 !== "" &&
  //                   retreatRoastData[i].under_age_35 !== null
  //                     ? retreatRoastData[i].under_age_35
  //                     : "NO",
  //                 referral_from_a_friend_or_family_member:
  //                   retreatRoastData[i].referral_from_a_friend_or_family_member,
  //               });
  //               // console.log(
  //               //   "coupleNotExist- whenb primary key naa mile--->",
  //               //   coupleNotExist
  //               // );
  //               if (
  //                 coupleNotExist.length &&
  //                 coupleNotExist.length > 0 &&
  //                 coupleNotExist !== ""
  //               ) {
  //                 let newUser = new coupleSchema(coupleNotExist[0]);
  //                 let succ = await newUser.save();
  //                 // console.log("succ-- xcrteate couple->", succ.id);
  //                 if (succ.hisEmail) {
  //                   let randompass1 = Math.random().toString(36).slice(-8);
  //                   paswordForCouple1 =
  //                     randompass1.slice(0, 0) + "p" + randompass1.slice(1);
  //                 }
  //                 if (succ.herEmail) {
  //                   let randompass2 = Math.random().toString(36).slice(-8);
  //                   paswordForCouple2 =
  //                     randompass2.slice(0, 0) + "p" + randompass2.slice(1);
  //                 }
  //                 adminObj = [
  //                   {
  //                     userName: succ.hisFirstName,
  //                     email: succ.hisEmail,
  //                     password: paswordForCouple1 ? paswordForCouple1 : null,
  //                     accountStatus: "ACTIVE",
  //                     userRole: "COUPLE",
  //                     coupleId: succ.id,
  //                     emailVerified: 1,
  //                     firstName: succ.hisFirstName ? succ.hisFirstName : "",
  //                     lastName: succ.hisLastName ? succ.hisLastName : "",
  //                     mobile: succ.hisMobile ? succ.hisMobile : "",
  //                   },
  //                   {
  //                     userName: succ.herFirstName,
  //                     email: succ.herEmail,
  //                     password: paswordForCouple2 ? paswordForCouple2 : null,
  //                     accountStatus: "ACTIVE",
  //                     userRole: "COUPLE",
  //                     coupleId: succ.id,
  //                     emailVerified: 1,
  //                     firstName: succ.herFirstName ? succ.herFirstName : "",
  //                     lastName: succ.herLastName ? succ.herLastName : "",
  //                     mobile: succ.herMobile ? succ.herMobile : "",
  //                   },
  //                 ];
  //                 adminObj.map((succ) => {
  //                   userSchema.create(succ);
  //                 });
  //                 if (succ.hisEmail) {
  //                   let replData = SENDLOGINDETAILBYADMIN.replace(
  //                     /#firstName#/g,
  //                     succ.hisFirstName ? succ.hisFirstName : ""
  //                   )
  //                     .replace(
  //                       /#lastName#/g,
  //                       succ.hisLastName ? succ.hisLastName : ""
  //                     )
  //                     .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
  //                     .replace(/#email#/g, succ.hisEmail)
  //                     .replace(/#password#/g, paswordForCouple1);
  //                   Email.sendEmail(
  //                     succ.hisEmail,
  //                     "Couple Detail for Login- LOVESTRONG Marriage",
  //                     replData
  //                   );
  //                 }
  //                 if (succ.herEmail) {
  //                   let replData = SENDLOGINDETAILBYADMIN.replace(
  //                     /#firstName#/g,
  //                     succ.herFirstName ? succ.herFirstName : ""
  //                   )
  //                     .replace(
  //                       /#lastName#/g,
  //                       succ.herLastName ? succ.herLastName : ""
  //                     )
  //                     .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

  //                     .replace(/#email#/g, succ.herEmail)
  //                     .replace(/#password#/g, paswordForCouple2);
  //                   Email.sendEmail(
  //                     succ.herEmail,
  //                     "Couple Detail for Login- LOVESTRONG Marriage",
  //                     replData
  //                   );
  //                 }

  //                 createdCouple.push({
  //                   couple:
  //                     `Couple : (${
  //                       succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                     }` +
  //                     `${
  //                       succ && succ.hisLastName ? `${succ.hisLastName}` : ""
  //                     }` +
  //                     `${
  //                       succ && succ.hisFirstName
  //                         ? `, ${succ.hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       succ && succ.herFirstName
  //                         ? ` and  ${succ.herFirstName}`
  //                         : ""
  //                     })`,
  //                   primaryKey: succ.primaryKey,
  //                   hisFirstName: succ.hisFirstName,
  //                   hisLastName: succ.hisLastName,
  //                   hisEmail: succ.hisEmail,
  //                   hisMobile: succ.hisMobile,
  //                   herFirstName: succ.herFirstName,
  //                   herLastName: succ.herLastName,
  //                   herEmail: succ.herEmail,
  //                   herMobile: succ.herMobile,
  //                   city: succ.city,
  //                   state: succ.state,
  //                   zip: succ.zip,
  //                   notes: succ.notes,
  //                   parish: retreatRoastData[i].parish,
  //                   address: succ.address,
  //                   allergies: succ.allergies,
  //                   anniversary_date: succ.anniversary_date,
  //                   hear_us: succ.hear_us,
  //                   emergency_name1: succ.emergency_name1,
  //                   emergency_relationship1: succ.emergency_relationship1,
  //                   emergency_phone1: succ.emergency_phone1,
  //                   emergency_name2: succ.emergency_name2,
  //                   emergency_relationship2: succ.emergency_relationship2,
  //                   emergency_phone2: succ.emergency_phone2,
  //                   under_age_35: succ.under_age_35,
  //                   referral_from_a_friend_or_family_member:
  //                     succ.referral_from_a_friend_or_family_member,
  //                 });

  //                 let foundRecordForRetreat = await coupleSchema.findAll({
  //                   where: {
  //                     id: succ.id,
  //                   },
  //                   raw: true,
  //                 });
  //                 if (
  //                   foundRecordForRetreat &&
  //                   foundRecordForRetreat.length > 0
  //                 ) {
  //                   let userDataToUpdate = {
  //                     roleId: formatObj.roleId ? formatObj.roleId : null,
  //                     serviceId: formatObj.serviceId
  //                       ? formatObj.serviceId
  //                       : null,
  //                     talkId: formatObj.talkId ? formatObj.talkId : null,
  //                     attendeeType: retreatRoastData[i].attendeeType,
  //                     coupleId:
  //                       foundRecordForRetreat && foundRecordForRetreat[0].id,
  //                     retreatId: retreatRoastData[i].retreatId,
  //                   };
  //                   let newRoaster = new retreatRoastSchema(userDataToUpdate);
  //                   let succ1 = await newRoaster.save();
  //                   createdRoasters.push({
  //                     couple:
  //                       `Couple : (${
  //                         retreatRoastData[i].primaryKey
  //                           ? `${retreatRoastData[i].primaryKey} - `
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                           ? `${retreatRoastData[i].hisLastName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].hisFirstName
  //                           ? `, ${retreatRoastData[i].hisFirstName}`
  //                           : ""
  //                       }` +
  //                       `${
  //                         retreatRoastData[i] &&
  //                         retreatRoastData[i].herFirstName
  //                           ? ` and  ${retreatRoastData[i].herFirstName}`
  //                           : ""
  //                       })`,
  //                     Role: retreatRoastData[i].role,
  //                     Service: retreatRoastData[i].service,
  //                     Talk: retreatRoastData[i].talk,
  //                     Parish: retreatRoastData[i].parish,
  //                     AttendeeType: retreatRoastData[i].attendeeType,
  //                   });
  //                 }
  //               }
  //             }
  //           } else {
  //             coupleNotExist.push({
  //               primaryKey: maxPrimaryKeyDB + 1,
  //               hisFirstName: retreatRoastData[i].hisFirstName,
  //               hisLastName: retreatRoastData[i].hisLastName,
  //               hisEmail: retreatRoastData[i].hisEmail,
  //               hisMobile: retreatRoastData[i].hisMobile,
  //               herFirstName: retreatRoastData[i].herFirstName,
  //               herLastName: retreatRoastData[i].herLastName,
  //               herEmail: retreatRoastData[i].herEmail,
  //               herMobile: retreatRoastData[i].herMobile,
  //               city: retreatRoastData[i].city,
  //               state: retreatRoastData[i].state,
  //               zip: retreatRoastData[i].zip,
  //               notes: retreatRoastData[i].notes,
  //               parishId: parishId && parishId.id,
  //               address: retreatRoastData[i].address,
  //               allergies: retreatRoastData[i].allergies,
  //               anniversary_date:
  //                 retreatRoastData[i].anniversary_date === ""
  //                   ? null
  //                   : retreatRoastData[i].anniversary_date,
  //               hear_us: retreatRoastData[i].hear_us
  //                 ? JSON.stringify(retreatRoastData[i].hear_us)
  //                 : [],
  //               emergency_name1: retreatRoastData[i].emergency_name1,
  //               emergency_relationship1:
  //                 retreatRoastData[i].emergency_relationship1,
  //               emergency_phone1: retreatRoastData[i].emergency_phone1,
  //               emergency_name2: retreatRoastData[i].emergency_name2,
  //               emergency_relationship2:
  //                 retreatRoastData[i].emergency_relationship2,
  //               emergency_phone2: retreatRoastData[i].emergency_phone2,
  //               under_age_35:
  //                 retreatRoastData[i] &&
  //                 retreatRoastData[i].under_age_35 &&
  //                 retreatRoastData[i].under_age_35 !== "" &&
  //                 retreatRoastData[i].under_age_35 !== null
  //                   ? retreatRoastData[i].under_age_35
  //                   : "NO",
  //               referral_from_a_friend_or_family_member:
  //                 retreatRoastData[i].referral_from_a_friend_or_family_member,
  //             });
  //             // console.log(
  //             //   "coupleNotExist- whenb primary key naa mile--->",
  //             //   coupleNotExist
  //             // );
  //             if (
  //               coupleNotExist.length &&
  //               coupleNotExist.length > 0 &&
  //               coupleNotExist !== ""
  //             ) {
  //               let newUser = new coupleSchema(coupleNotExist[0]);
  //               let succ = await newUser.save();
  //               // console.log("succ-- xcrteate couple->", succ.id);
  //               if (succ.hisEmail) {
  //                 let randompass1 = Math.random().toString(36).slice(-8);
  //                 paswordForCouple1 =
  //                   randompass1.slice(0, 0) + "p" + randompass1.slice(1);
  //               }
  //               if (succ.herEmail) {
  //                 let randompass2 = Math.random().toString(36).slice(-8);
  //                 paswordForCouple2 =
  //                   randompass2.slice(0, 0) + "p" + randompass2.slice(1);
  //               }
  //               adminObj = [
  //                 {
  //                   userName: succ.hisFirstName,
  //                   email: succ.hisEmail,
  //                   password: paswordForCouple1 ? paswordForCouple1 : null,
  //                   accountStatus: "ACTIVE",
  //                   userRole: "COUPLE",
  //                   coupleId: succ.id,
  //                   emailVerified: 1,
  //                   firstName: succ.hisFirstName ? succ.hisFirstName : "",
  //                   lastName: succ.hisLastName ? succ.hisLastName : "",
  //                   mobile: succ.hisMobile ? succ.hisMobile : "",
  //                 },
  //                 {
  //                   userName: succ.herFirstName,
  //                   email: succ.herEmail,
  //                   password: paswordForCouple2 ? paswordForCouple2 : null,
  //                   accountStatus: "ACTIVE",
  //                   userRole: "COUPLE",
  //                   coupleId: succ.id,
  //                   emailVerified: 1,
  //                   firstName: succ.herFirstName ? succ.herFirstName : "",
  //                   lastName: succ.herLastName ? succ.herLastName : "",
  //                   mobile: succ.herMobile ? succ.herMobile : "",
  //                 },
  //               ];
  //               adminObj.map((succ) => {
  //                 userSchema.create(succ);
  //               });
  //               if (succ.hisEmail) {
  //                 let replData = SENDLOGINDETAILBYADMIN.replace(
  //                   /#firstName#/g,
  //                   succ.hisFirstName ? succ.hisFirstName : ""
  //                 )
  //                   .replace(
  //                     /#lastName#/g,
  //                     succ.hisLastName ? succ.hisLastName : ""
  //                   )
  //                   .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
  //                   .replace(/#email#/g, succ.hisEmail)
  //                   .replace(/#password#/g, paswordForCouple1);
  //                 Email.sendEmail(
  //                   succ.hisEmail,
  //                   "Couple Detail for Login- LOVESTRONG Marriage",
  //                   replData
  //                 );
  //               }
  //               if (succ.herEmail) {
  //                 let replData = SENDLOGINDETAILBYADMIN.replace(
  //                   /#firstName#/g,
  //                   succ.herFirstName ? succ.herFirstName : ""
  //                 )
  //                   .replace(
  //                     /#lastName#/g,
  //                     succ.herLastName ? succ.herLastName : ""
  //                   )
  //                   .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

  //                   .replace(/#email#/g, succ.herEmail)
  //                   .replace(/#password#/g, paswordForCouple2);
  //                 Email.sendEmail(
  //                   succ.herEmail,
  //                   "Couple Detail for Login- LOVESTRONG Marriage",
  //                   replData
  //                 );
  //               }

  //               createdCouple.push({
  //                 couple:
  //                   `Couple : (${
  //                     succ.primaryKey ? `${succ.primaryKey} - ` : ""
  //                   }` +
  //                   `${succ && succ.hisLastName ? `${succ.hisLastName}` : ""}` +
  //                   `${
  //                     succ && succ.hisFirstName ? `, ${succ.hisFirstName}` : ""
  //                   }` +
  //                   `${
  //                     succ && succ.herFirstName
  //                       ? ` and  ${succ.herFirstName}`
  //                       : ""
  //                   })`,
  //                 primaryKey: succ.primaryKey,
  //                 hisFirstName: succ.hisFirstName,
  //                 hisLastName: succ.hisLastName,
  //                 hisEmail: succ.hisEmail,
  //                 hisMobile: succ.hisMobile,
  //                 herFirstName: succ.herFirstName,
  //                 herLastName: succ.herLastName,
  //                 herEmail: succ.herEmail,
  //                 herMobile: succ.herMobile,
  //                 city: succ.city,
  //                 state: succ.state,
  //                 zip: succ.zip,
  //                 notes: succ.notes,
  //                 parish: retreatRoastData[i].parish,
  //                 address: succ.address,
  //                 allergies: succ.allergies,
  //                 anniversary_date: succ.anniversary_date,
  //                 hear_us: succ.hear_us,
  //                 emergency_name1: succ.emergency_name1,
  //                 emergency_relationship1: succ.emergency_relationship1,
  //                 emergency_phone1: succ.emergency_phone1,
  //                 emergency_name2: succ.emergency_name2,
  //                 emergency_relationship2: succ.emergency_relationship2,
  //                 emergency_phone2: succ.emergency_phone2,
  //                 under_age_35: succ.under_age_35,
  //                 referral_from_a_friend_or_family_member:
  //                   succ.referral_from_a_friend_or_family_member,
  //               });

  //               let foundRecordForRetreat = await coupleSchema.findAll({
  //                 where: {
  //                   id: succ.id,
  //                 },
  //                 raw: true,
  //               });
  //               if (foundRecordForRetreat && foundRecordForRetreat.length > 0) {
  //                 let userDataToUpdate = {
  //                   roleId: formatObj.roleId ? formatObj.roleId : null,
  //                   serviceId: formatObj.serviceId ? formatObj.serviceId : null,
  //                   talkId: formatObj.talkId ? formatObj.talkId : null,
  //                   attendeeType: retreatRoastData[i].attendeeType,
  //                   coupleId:
  //                     foundRecordForRetreat && foundRecordForRetreat[0].id,
  //                   retreatId: retreatRoastData[i].retreatId,
  //                 };
  //                 let newRoaster = new retreatRoastSchema(userDataToUpdate);
  //                 let succ1 = await newRoaster.save();
  //                 createdRoasters.push({
  //                   couple:
  //                     `Couple : (${
  //                       retreatRoastData[i].primaryKey
  //                         ? `${retreatRoastData[i].primaryKey} - `
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisLastName
  //                         ? `${retreatRoastData[i].hisLastName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].hisFirstName
  //                         ? `, ${retreatRoastData[i].hisFirstName}`
  //                         : ""
  //                     }` +
  //                     `${
  //                       retreatRoastData[i] && retreatRoastData[i].herFirstName
  //                         ? ` and  ${retreatRoastData[i].herFirstName}`
  //                         : ""
  //                     })`,
  //                   Role: retreatRoastData[i].role,
  //                   Service: retreatRoastData[i].service,
  //                   Talk: retreatRoastData[i].talk,
  //                   Parish: retreatRoastData[i].parish,
  //                   AttendeeType: retreatRoastData[i].attendeeType,
  //                 });
  //               }
  //             }
  //           }
  //         }
  //       }
  //     }
  //     return res.send({
  //       success: true,
  //       updateRoaster: updateRoaster,
  //       createdRoasters: createdRoasters,
  //       updateCouple: updateCouple,
  //       createdCouple: createdCouple,
  //       msg: "Retreat Roster data imported successfully.",
  //     });
  //   } catch (err) {
  //     console.log("error: ", err);
  //     return res.send({
  //       success: false,
  //       msg: "Something went wrong.",
  //     });
  //   }
  // }

  async getRetreatLead(req, res) {
    try {
      let select = `SELECT retreats.freeze,
      retreatroasters.retreatId,
      retreatroasters.createdAt, 
      couples.hisLastName, couples.hisFirstName,
      couples.herFirstName, couples.herLastName,
      couples.hisEmail, 
      couples.herEmail, couples.hisMobile,roles.role `;
      let from = ` FROM retreatroasters
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id 
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      INNER JOIN retreats on retreatroasters.retreatId = retreats.id WHERE roles.role = "Lead" AND retreatroasters.stage = "VERIFIED"`;
      let groupBy = ` GROUP BY retreatroasters.retreatId`;
      let queryRecords = select + from + groupBy;
      let responseData1 = await db.sequelize
        .query(queryRecords)
        .then(([results, metadata]) => results);
      return res.status(200).send({
        success: true,
        data: responseData1,
      });
    } catch (error) {
      console.log("error:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async CreateRoaster(req, res) {
    let {
      primaryKey,
      hisFirstName,
      hisLastName,
      hisEmail,
      hisMobile,
      herFirstName,
      herLastName,
      herEmail,
      herMobile,
      city,
      state,
      notes,
      parishId,
      attendeeType,
      roleId,
      serviceId,
      talkId,
      retreatId,
      zip,
      imageUrl,
      coupleId,
      address,
      allergies,
      anniversary_date,
      under_age_35,
      emergency_name1,
      emergency_relationship1,
      emergency_phone1,
      emergency_name2,
      emergency_relationship2,
      emergency_phone2,
      hear_us,
      referral_from_a_friend_or_family_member,
    } = JSON.parse(`${req.body.fromData}`);
    try {
      if (!(hisFirstName || hisLastName || herFirstName || herLastName)) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "Please add at least one name.",
        });
      }
      if (!hisEmail && !herEmail) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "Please add at least one email ID.",
        });
      }
      // if (hisEmail === herEmail) {
      //   return res.status(200).json({
      //     success: false,
      //     error: true,
      //     msg: "His Email and Her Email should be different.",
      //   });
      // }
      let checkMailInDb = await coupleSchema.findOne({
        where: {
          primaryKey: { [Op.ne]: primaryKey },
          [Op.or]: [
            hisEmail && {
              // Only check if hisEmail is not an empty string
              [Op.and]: [
                { hisEmail: { [Op.ne]: null } },
                { hisEmail: { [Op.ne]: "" } },
                { [Op.or]: [{ hisEmail: hisEmail }, { hisEmail: herEmail }] },
              ],
            },
            herEmail && {
              // Only check if herEmail is not an empty string
              [Op.and]: [
                { herEmail: { [Op.ne]: null } },
                { herEmail: { [Op.ne]: "" } },
                { [Op.or]: [{ herEmail: hisEmail }, { herEmail: herEmail }] },
              ],
            },
          ],
        },
      });
      if (checkMailInDb) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "This email is already exist.",
        });
      }
      const normalizedHisMobile = hisMobile && hisMobile.replace(/\D/g, "");
      const normalizedHerMobile = herMobile && herMobile.replace(/\D/g, "");
      let checkMobileInDb = await coupleSchema.findOne({
        where: {
          primaryKey: { [Op.ne]: primaryKey },
          [Op.or]: [
            {
              [Op.and]: [
                { hisMobile: { [Op.ne]: null } },
                { hisMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHisMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { hisMobile: { [Op.ne]: null } },
                { hisMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHerMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { herMobile: { [Op.ne]: null } },
                { herMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHisMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { herMobile: { [Op.ne]: null } },
                { herMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHerMobile}'`
                ),
              ],
            },
          ],
        },
      });
      if (checkMobileInDb) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "This mobile number already exists.",
        });
      }
      let jsonObj = JSON.parse(req.body.fromData);
      ["hisEmail", "herEmail", "hisMobile", "herMobile"].forEach((field) => {
        if (jsonObj[field] === "") {
          jsonObj[field] = null;
        }
      });
      if (jsonObj.anniversary_date === "") {
        jsonObj.anniversary_date = null;
      }
      if (hear_us) {
        // Convert hear_us array to a JSON string
        jsonObj.hear_us = JSON.stringify(hear_us);
      }
      let randompass = Math.random().toString(36).slice(-8);
      let paswordForCouple = randompass.slice(0, 0) + "p" + randompass.slice(1);
      jsonObj.accountStatus = "ACTIVE";
      delete jsonObj.id;
      if (primaryKey && paswordForCouple) {
        let primaryKeyCheck = await coupleSchema.findAll({
          where: { primaryKey: primaryKey },
        });
        if (primaryKeyCheck.length) {
          let userDataToUpdate1 = {
            hisFirstName: hisFirstName,
            hisLastName: hisLastName,
            hisEmail: hisEmail !== "" ? hisEmail : null,
            hisMobile: hisMobile !== "" ? hisMobile : null,
            herFirstName: herFirstName,
            herLastName: herLastName,
            herEmail: herEmail !== "" ? herEmail : null,
            herMobile: herMobile !== "" ? herMobile : null,
            city: city,
            state: state,
            zip: zip,
            notes: notes,
            parishId: parishId && parishId !== "" ? parishId : null,
            // imageUrl: imageUrl,
            address: address,
            allergies: allergies,
            anniversary_date: anniversary_date === "" ? null : anniversary_date,
            under_age_35: under_age_35,
            emergency_name1: emergency_name1,
            emergency_relationship1: emergency_relationship1,
            emergency_phone1: emergency_phone1,
            emergency_name2: emergency_name2,
            emergency_relationship2: emergency_relationship2,
            emergency_phone2: emergency_phone2,
            hear_us: JSON.stringify(hear_us),
            referral_from_a_friend_or_family_member:
              referral_from_a_friend_or_family_member,
            // linkverify: linkverify,
          };
          let imageSavedCouple;
          let update = await coupleSchema.update(userDataToUpdate1, {
            where: { id: coupleId },
          });
          let userDataToUpdate = {
            attendeeType: attendeeType,
            roleId: roleId !== "" ? roleId : null,
            serviceId: serviceId !== "" ? serviceId : null,
            talkId: talkId !== "" ? talkId : null,
            coupleId: coupleId,
            retreatId: retreatId,
          };
          let newRoaster = new retreatRoastSchema(userDataToUpdate);
          let succ1 = await newRoaster.save();
          if (req.files) {
            imageSavedCouple = await this.saveImageForCouple(
              req.files,
              coupleId
            );
          } else {
            if (imageUrl === "") {
              let removeImage = await coupleSchema.update(
                { imageUrl: "" },
                { where: { id: coupleId } }
              );
            }
          }
          let findCouple = await coupleSchema.findAll({
            where: { id: coupleId },
            raw: true,
          });
          let adminObj = [
            {
              coupleId: findCouple[0].id,
              userName: findCouple[0].hisFirstName,
              firstName: findCouple[0].hisFirstName
                ? findCouple[0].hisFirstName
                : "",
              lastName: findCouple[0].hisLastName
                ? findCouple[0].hisLastName
                : "",
              mobile: findCouple[0].hisMobile ? findCouple[0].hisMobile : "",
            },

            {
              coupleId: findCouple[0].id,
              userName: findCouple[0].herFirstName,
              firstName: findCouple[0].herFirstName
                ? findCouple[0].herFirstName
                : "",
              lastName: findCouple[0].herLastName
                ? findCouple[0].herLastName
                : "",
              mobile: findCouple[0].herMobile ? findCouple[0].herMobile : "",
            },
          ];
          if (
            findCouple &&
            findCouple[0].hisEmail &&
            findCouple[0].hisEmail !== "" &&
            findCouple[0].hisEmail !== null
          ) {
            let upDate = await userSchema.update(adminObj[0], {
              where: { email: findCouple[0].hisEmail },
            });
          }
          if (
            findCouple &&
            findCouple[0].herEmail &&
            findCouple[0].herEmail !== "" &&
            findCouple[0].herEmail !== null
          ) {
            let upDate1 = await userSchema.update(adminObj[1], {
              where: { email: findCouple[0].herEmail },
            });
          }
          if (succ1) {
            return res.status(200).send({
              success: true,
              msg: "Couple added successfully.",
              data: succ1,
            });
          } else {
            return res.status(200).send({
              msg: `Something went wrong. Please try again.`,
              error: true,
              success: false,
            });
          }
        } else {
          let newUser = new coupleSchema(jsonObj);
          let succ = await newUser.save();
          let paswordForCouple1;
          let paswordForCouple2;
          if (hisEmail) {
            let randompass1 = Math.random().toString(36).slice(-8);
            paswordForCouple1 =
              randompass1.slice(0, 0) + "p" + randompass1.slice(1);
          }
          if (herEmail) {
            let randompass2 = Math.random().toString(36).slice(-8);
            paswordForCouple2 =
              randompass2.slice(0, 0) + "p" + randompass2.slice(1);
          }
          let userDataToUpdate = {
            attendeeType: attendeeType,
            roleId: roleId !== "" ? roleId : null,
            serviceId: serviceId !== "" ? serviceId : null,
            talkId: talkId !== "" ? talkId : null,
            coupleId: succ && succ.id,
            retreatId: retreatId,
          };
          let newRoaster = new retreatRoastSchema(userDataToUpdate);
          let succ1 = await newRoaster.save();
          let imageSavedCouple;
          if (req.files) {
            imageSavedCouple = await this.saveImageForCouple(
              req.files,
              succ.id
            );
          }
          let coupleValues = await coupleSchema.findOne({
            where: {
              id: succ.id,
            },
            raw: true,
          });
          let adminObj = [
            {
              userName: coupleValues.hisFirstName,
              email: coupleValues.hisEmail,
              password: paswordForCouple1 ? paswordForCouple1 : null,
              accountStatus: "ACTIVE",
              userRole: "COUPLE",
              coupleId: coupleValues.id,
              emailVerified: 1,
              firstName: coupleValues.hisFirstName
                ? coupleValues.hisFirstName
                : "",
              lastName: coupleValues.hisLastName
                ? coupleValues.hisLastName
                : "",
              mobile: coupleValues.hisMobile ? coupleValues.hisMobile : "",
              imageUpload: coupleValues.imageUrl,
            },
            {
              userName: coupleValues.herFirstName,
              email: coupleValues.herEmail,
              password: paswordForCouple2 ? paswordForCouple2 : null,
              accountStatus: "ACTIVE",
              userRole: "COUPLE",
              coupleId: coupleValues.id,
              emailVerified: 1,
              firstName: coupleValues.herFirstName
                ? coupleValues.herFirstName
                : "",
              lastName: coupleValues.herLastName
                ? coupleValues.herLastName
                : "",
              mobile: coupleValues.herMobile ? coupleValues.herMobile : "",
              imageUpload: coupleValues.imageUrl,
            },
          ];
          adminObj.map((item) => {
            userSchema.create(item);
          });
          if (succ.hisEmail) {
            let replData = SENDLOGINDETAILBYADMIN.replace(
              /#firstName#/g,
              succ.hisFirstName ? succ.hisFirstName : ""
            )
              .replace(/#lastName#/g, succ.hisLastName ? succ.hisLastName : "")
              .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

              .replace(/#email#/g, hisEmail)

              .replace(/#password#/g, paswordForCouple1);
            Email.sendEmail(
              succ.hisEmail,
              "Couple Detail for Login- LOVESTRONG Marriage",
              replData
            );
          }
          if (succ.herEmail) {
            let replData = SENDLOGINDETAILBYADMIN.replace(
              /#firstName#/g,
              succ.herFirstName ? succ.herFirstName : ""
            )
              .replace(/#lastName#/g, succ.herLastName ? succ.herLastName : "")
              .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

              .replace(/#email#/g, herEmail)
              .replace(/#password#/g, paswordForCouple1);
            Email.sendEmail(
              succ.herEmail,
              "Couple Detail for Login- LOVESTRONG Marriage",
              replData
            );
          }
          if (succ1) {
            return res.status(200).send({
              success: true,
              msg: "Couple added successfully.",
              data: succ,
            });
          } else {
            return res.status(200).send({
              msg: `Something went wrong. Please try again.`,
              error: true,
              success: false,
            });
          }
        }
      } else {
        return res.status(200).send({
          msg: "Primary Key is required.",
          error: true,
          success: false,
        });
      }
    } catch (error) {
      console.log(
        `CreateRoaster [${new Date().toLocaleString("en-US", {
          timeZoneName: "short",
        })}] error -->`,
        error
      );
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async updateRoaster(req, res) {
    let {
      primaryKey,
      hisFirstName,
      hisLastName,
      hisEmail,
      hisMobile,
      herFirstName,
      herLastName,
      herEmail,
      herMobile,
      city,
      state,
      notes,
      parishId,
      attendeeType,
      roleId,
      serviceId,
      talkId,
      retreatId,
      id,
      zip,
      linkverify,
      imageUrl,
      // marriageImageUrl,
      coupleId,
      address,
      allergies,
      anniversary_date,
      under_age_35,
      emergency_name1,
      emergency_relationship1,
      emergency_phone1,
      emergency_name2,
      emergency_relationship2,
      emergency_phone2,
      hear_us,
      referral_from_a_friend_or_family_member,
    } = JSON.parse(`${req.body.fromData}`);
    try {
      if (!(hisFirstName || hisLastName || herFirstName || herLastName)) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "Please add at least one name.",
        });
      }
      if (!hisEmail && !herEmail) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "Please add at least one email ID.",
        });
      }
      // if (hisEmail === herEmail) {
      //   return res.status(200).json({
      //     success: false,
      //     error: true,
      //     msg: "His Email and Her Email should be different.",
      //   });
      // }
      let checkMailInDb = await coupleSchema.findOne({
        where: {
          primaryKey: { [Op.ne]: primaryKey },
          [Op.or]: [
            hisEmail && {
              // Only check if hisEmail is not an empty string
              [Op.and]: [
                { hisEmail: { [Op.ne]: null } },
                { hisEmail: { [Op.ne]: "" } },
                { [Op.or]: [{ hisEmail: hisEmail }, { hisEmail: herEmail }] },
              ],
            },
            herEmail && {
              // Only check if herEmail is not an empty string
              [Op.and]: [
                { herEmail: { [Op.ne]: null } },
                { herEmail: { [Op.ne]: "" } },
                { [Op.or]: [{ herEmail: hisEmail }, { herEmail: herEmail }] },
              ],
            },
          ],
        },
      });
      if (checkMailInDb) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "This email is already exist.",
        });
      }
      const normalizedHisMobile = hisMobile && hisMobile.replace(/\D/g, "");
      const normalizedHerMobile = herMobile && herMobile.replace(/\D/g, "");
      let checkMobileInDb = await coupleSchema.findOne({
        where: {
          primaryKey: { [Op.ne]: primaryKey },
          [Op.or]: [
            {
              [Op.and]: [
                { hisMobile: { [Op.ne]: null } },
                { hisMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHisMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { hisMobile: { [Op.ne]: null } },
                { hisMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(hisMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHerMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { herMobile: { [Op.ne]: null } },
                { herMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHisMobile}'`
                ),
              ],
            },
            {
              [Op.and]: [
                { herMobile: { [Op.ne]: null } },
                { herMobile: { [Op.ne]: "" } },
                Sequelize.literal(
                  `REPLACE(REPLACE(REPLACE(REPLACE(herMobile, '-', ''), '(', ''), ')', ''), ' ', '') = '${normalizedHerMobile}'`
                ),
              ],
            },
          ],
        },
      });
      if (checkMobileInDb) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "This mobile number already exists.",
        });
      }

      let userDataToUpdate = {
        hisFirstName: hisFirstName,
        hisLastName: hisLastName,
        hisEmail: hisEmail !== "" ? hisEmail : null,
        hisMobile: hisMobile !== "" ? hisMobile : null,
        herFirstName: herFirstName,
        herLastName: herLastName,
        herEmail: herEmail !== "" ? herEmail : null,
        herMobile: herMobile !== "" ? herMobile : null,
        city: city,
        state: state,
        zip: zip,
        notes: notes,
        parishId: parishId && parishId !== "" ? parishId : null,
        linkverify: linkverify,
        address: address,
        allergies: allergies,
        anniversary_date: anniversary_date === "" ? null : anniversary_date,
        under_age_35: under_age_35,
        emergency_name1: emergency_name1,
        emergency_relationship1: emergency_relationship1,
        emergency_phone1: emergency_phone1,
        emergency_name2: emergency_name2,
        emergency_relationship2: emergency_relationship2,
        emergency_phone2: emergency_phone2,
        hear_us: JSON.stringify(hear_us),
        referral_from_a_friend_or_family_member:
          referral_from_a_friend_or_family_member,
      };
      let update = await coupleSchema.update(userDataToUpdate, {
        where: { id: coupleId },
      });
      let userDataToUpdate1 = {
        attendeeType: attendeeType,
        roleId: roleId !== "" ? roleId : null,
        serviceId: serviceId !== "" ? serviceId : null,
        talkId: talkId !== "" ? talkId : null,
        coupleId: coupleId,
        retreatId: retreatId,
      };
      let updateRoast = {};
      if (id) {
        let succ = await retreatRoastSchema.update(userDataToUpdate1, {
          where: { id: id },
        });
        updateRoast = await retreatRoastSchema.findOne({
          where: { id: id },
          raw: true,
        });
      } else {
        let newRoaster = new retreatRoastSchema(userDataToUpdate1);
        let succ1 = await newRoaster.save();
        updateRoast = await retreatRoastSchema.findOne({
          where: { id: succ1.id },
          raw: true,
        });
      }
      const coupleDetail = await coupleSchema.findOne({
        where: { id: coupleId },
        raw: true,
      });
      let imageSavedCouple;
      if (req.files) {
                if (coupleDetail) {
            if (coupleDetail.imageUrl) {
              const profileImagePath = path.join(
                PROJECT_DIR,
                "profileImage",
                path.basename(coupleDetail.imageUrl)
              );
              if (fs.existsSync(profileImagePath)) {
                fs.unlinkSync(profileImagePath);
              }
            }
          }
        imageSavedCouple = await this.saveImageForCouple(req.files, coupleId);
      } else {
        if (imageUrl === "") {
          if (coupleDetail) {
            if (coupleDetail.imageUrl) {
              const profileImagePath = path.join(
                PROJECT_DIR,
                "profileImage",
                path.basename(coupleDetail.imageUrl)
              );
              if (fs.existsSync(profileImagePath)) {
                fs.unlinkSync(profileImagePath);
              }
            }
          }
          let removeImage = await coupleSchema.update(
            { imageUrl: "" },
            { where: { id: coupleId } }
          );
        }
        // if (marriageImageUrl === "") {
        //   let removeImage = await coupleSchema.update(
        //     { marriageImageUrl: "" },
        //     { where: { id: coupleId } }
        //   );
        // }
      }
      let findCouple = await coupleSchema.findAll({
        where: { id: updateRoast && updateRoast.coupleId },
        raw: true,
      });
      let adminObj = [
        {
          coupleId: findCouple[0].id,
          userName: findCouple[0].hisFirstName,
          firstName: findCouple[0].hisFirstName
            ? findCouple[0].hisFirstName
            : "",
          lastName: findCouple[0].hisLastName ? findCouple[0].hisLastName : "",
          mobile: findCouple[0].hisMobile ? findCouple[0].hisMobile : "",
          imageUpload: findCouple[0].imageUrl,
        },

        {
          coupleId: findCouple[0].id,
          userName: findCouple[0].herFirstName,
          firstName: findCouple[0].herFirstName
            ? findCouple[0].herFirstName
            : "",
          lastName: findCouple[0].herLastName ? findCouple[0].herLastName : "",
          mobile: findCouple[0].herMobile ? findCouple[0].herMobile : "",
          imageUpload: findCouple[0].imageUrl,
        },
      ];
      if (
        findCouple &&
        findCouple[0].hisEmail &&
        findCouple[0].hisEmail !== "" &&
        findCouple[0].hisEmail !== null
      ) {
        let upDate = await userSchema.update(adminObj[0], {
          where: { email: findCouple[0].hisEmail },
        });
      }
      if (
        findCouple &&
        findCouple[0].herEmail &&
        findCouple[0].herEmail !== "" &&
        findCouple[0].herEmail !== null
      ) {
        let upDate1 = await userSchema.update(adminObj[1], {
          where: { email: findCouple[0].herEmail },
        });
      }
      if (findCouple) {
        return res.status(200).send({
          success: true,
          msg: "Updated successfully.",
          data: findCouple,
          roastData: updateRoast,
        });
      } else {
        return res.status(200).send({
          msg: `Something went wrong. Please try again.`,
          error: true,
          success: false,
        });
      }
    } catch (error) {
      console.log(
        `updateRoaster [${new Date().toLocaleString("en-US", {
          timeZoneName: "short",
        })}] error -->`,
        error
      );
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async saveImageForCouple(files, coupleId) {
    try {
      let updateData = {};
      let coupleDetail = await coupleSchema.findOne({
        where: { id: coupleId },
        raw: true,
      });
      if (files.file) {
        const file = files.file;
        const now = new Date();
        const currentDate = now.toISOString().split("T")[0];
        const fileExtension = file.name.substring(file.name.lastIndexOf("."));
        const fileName = `${coupleDetail.hisLastName}, ${coupleDetail.hisFirstName} & ${coupleDetail.herFirstName}_${currentDate}${fileExtension}`;
        const fileUrl = PROJECT_DIR + "/profileImage/" + fileName;
        const imgUrl =
          process.env.SERVERADDRESS + "/public/profileImage/" + fileName;
        updateData.imageUrl = imgUrl;
        file.mv(fileUrl, async function (err) {
          if (err) {
            return false;
          }
        });
      }

      if (Object.keys(updateData).length > 0) {
        await coupleSchema.update(updateData, { where: { id: coupleId } });
        await userSchema.update(updateData, {
          where: {
            [Op.or]: [
              { email: coupleDetail[0].hisEmail },
              { email: coupleDetail[0].herEmail },
            ],
          },
        });
      }
    } catch (error) {
      return error;
    }
  }

  async getCouplesInRetreat(req, res) {
    try {
      let query = `SELECT couples.* FROM  couples WHERE couples.id NOT IN (SELECT retreatroasters.coupleId FROM  retreatroasters WHERE retreatroasters.retreatId = ${req.body.retreatId} )  GROUP BY couples.id`;
      let allCouples = await db.sequelize
        .query(query)
        .then(([results, metadata]) => results);
      return res.status(200).send({
        success: true,
        msg: "All Couples",
        data: allCouples,
      });
    } catch (error) {
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async createLeads(req, res) {
    try {
      let { roleId, coupleId, retreatId } = req.body;
      if (!roleId && !coupleId) {
        return res.send({
          error: true,
          msg: "Please select Role and Couple.",
        });
      } else if (!roleId && coupleId) {
        return res.send({
          error: true,
          msg: "Please select Role.",
        });
      } else if (roleId && !coupleId) {
        return res.send({
          error: true,
          msg: "Please select Couple Role.",
        });
      } else {
        let roasterCount = await retreatRoastSchema.count({
          where: {
            [Op.and]: {
              retreatId: retreatId,
              roleId: roleId,
            },
          },
        });
        if (roasterCount >= 1) {
          let role = await roleSchema.findOne({
            where: {
              id: roleId,
            },
          });
          return res.status(200).json({
            success: false,
            msg: `You can add only one ${role.role}. Please delete the existing ${role.role} before adding a new one.`,
          });
        } else {
          let RoasterCheck = await retreatRoastSchema.findOne({
            where: {
              [Op.and]: {
                coupleId: coupleId,
                retreatId: retreatId,
              },
            },
          });
          if (RoasterCheck && RoasterCheck !== null) {
            let upDateRetreat = await retreatRoastSchema.update(
              { roleId: roleId, stage: "VERIFIED" },
              {
                where: {
                  [Op.and]: {
                    coupleId: coupleId,
                    retreatId: retreatId,
                  },
                },
              }
            );
            return res.status(200).send({
              success: true,
              msg: "New Retreat Team added.",
            });
          } else {
            let roasterData = {
              roleId: roleId,
              coupleId: coupleId,
              retreatId: retreatId,
              stage: "VERIFIED",
              attendeeType: "Team",
            };
            let newRoaster = new retreatRoastSchema(roasterData);
            let succ = await newRoaster.save();
            if (succ) {
              return res.status(200).send({
                success: true,
                msg: "New Retreat Team added.",
              });
            } else {
              return res.status(200).send({
                msg: `Something went wrong. Please try again`,
                error: true,
                success: false,
              });
            }
          }
        }
      }
    } catch (error) {
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async removeCoupleFromRetreat(req, res) {
    try {
      let { coupleId, retreatId } = req.body;
      let result = await retreatRoastSchema.destroy({
        where: {
          [Op.and]: [{ coupleId: coupleId }, { retreatId: retreatId }],
        },
      });

      if (result) {
        return res.status(200).send({
          success: true,
          msg: "Couple Removed.",
        });
      } else {
        return res.status(404).send({
          success: false,
          msg: "Couple not found.",
        });
      }
    } catch (error) {
      console.log("error", error);
      return res.status(500).json({
        success: false,
        msg: "An error occurred while removing the couple.",
      });
    }
  }
}

module.exports = new RetreatRoasterController();
