import React, { useState, useEffect, useRef } from "react";
import { toast } from "react-toastify";
import {
  OverlayTrigger,
  Tooltip,
  Form,
  InputGroup,
  Button,
  Row,
  Col,
  Card,
} from "react-bootstrap";
import CustomDataTable from "../../components/CustomDataTable";
import DeleteModel from "../../components/DeleteModel";
import HelpApi from "../../api/helpApi";
import Helpers from "../../components/Helpers";
import TicketAccess from "./TicketAccess";
import MessagesModal from "./MessagesModal";

const Tickets = () => {
  // main data
  const [tickets, setTickets] = useState([]);
  const [total, setTotal] = useState(0);
  const [LoadData, setLoadData] = useState(false);

  // modal / selected
  const [show, setShow] = useState(false);
  const [selectedTicket, setSelectedTicket] = useState(null);

  // filters / search
  const [searchKey, setSearchKey] = useState("");
  const [categoryFilter, setCategoryFilter] = useState("");
  const [statusFilter, setStatusFilter] = useState("");
  const [priorityFilter, setPriorityFilter] = useState("");

  // popup states
  const [showRolesPopUp, setShowRolesPopUp] = useState(false);
  const [messagesOpen, setMessagesOpen] = useState(false);
  const [newAssign, setNewAssign] = useState(false);

  // pagination / sorting (kept simple — your CustomDataTable calls getAllTickets)
  const pageSize = 20;

  const helpApi = useRef(new HelpApi()).current;
  const searchDebounceRef = useRef(null);

  useEffect(() => {
    getAllTickets();
    // cleanup on unmount
    return () => clearTimeout(searchDebounceRef.current);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const renderTooltip = (props) => (
    <Tooltip id="button-tooltip">{props}</Tooltip>
  );

  // fetcher
  const getAllTickets = async (
    pageNo = 1,
    size = pageSize,
    searchKeyword = searchKey,
    sortBy = {}
  ) => {
    setLoadData(true);
    const payload = {
      pageNo,
      size,
      searchKeyword,
      sortBy,
      filters: {
        category: categoryFilter,
        status: statusFilter,
        priority: priorityFilter,
      },
    };

    try {
      const res = await helpApi.getAllTickets(payload);
      if (res.success) {
        setTickets(res.data);
        setTotal(res.totalRecords || 0);
      } else {
        toast.error(res.msg || "Failed to fetch tickets");
      }
    } catch (err) {
      console.error(err);
      toast.error("Failed to fetch tickets");
    } finally {
      setLoadData(false);
    }
  };

  // Search with debounce to avoid too many requests
  const handleSearch = (e) => {
    const value = e.target.value;
    setSearchKey(value);
    if (searchDebounceRef.current) clearTimeout(searchDebounceRef.current);
    searchDebounceRef.current = setTimeout(() => {
      getAllTickets(1, pageSize, value);
    }, 350);
  };

  // 👇 replace handleFilterChange
  const handleFilterChange = (type, value) => {
    if (type === "category") setCategoryFilter(value);
    if (type === "status") setStatusFilter(value);
    if (type === "priority") setPriorityFilter(value);
  };

  // 👇 replace clearFilters
  const clearFilters = () => {
    setCategoryFilter("");
    setStatusFilter("");
    setPriorityFilter("");
    setSearchKey("");
  };

  // 👇 add this useEffect (below useEffect for mount)
  useEffect(() => {
    getAllTickets(1, pageSize, searchKey);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [categoryFilter, statusFilter, priorityFilter]);

  // modals and actions
  const handleShowDelete = (ticket) => {
    setSelectedTicket(ticket);
    setShow(true);
  };
  const handleClose = () => setShow(false);

  const deleteTicket = async () => {
    if (!selectedTicket) return;
    try {
      const res = await helpApi.deleteTicket({ ticket_id: selectedTicket.id });
      if (res.success) {
        toast.success(res.msg || "Ticket deleted");
        setShow(false);
        getAllTickets();
      } else {
        toast.error(res.msg || "Failed to delete");
      }
    } catch (err) {
      toast.error("Failed to delete ticket");
    }
  };

  const handleStatusChange = async (id, newStatus) => {
    try {
      await helpApi.updateStatus({
        id,
        status: newStatus,
      });
      toast.success("Ticket updated successfully");
      getAllTickets();
    } catch (error) {
      console.error("Update status error:", error);
    }
  };

  const rolesPopUpOpen = (ticket) => {
    setSelectedTicket(ticket);
    setShowRolesPopUp(true);
  };
  const rolesPopUpClose = () => {
    setShowRolesPopUp(false);
    getAllTickets();
  };

  const mesagesPopUpOpen = (ticket) => {
    setSelectedTicket(ticket);
    setMessagesOpen(true);
  };
  const mesagesPopUpClose = () => {
    setMessagesOpen(false);
    getAllTickets();
  };

  // Columns (kept consistent with your previous implementation)
  const columns = [
    {
      id: "ticketNumber",
      name: "Ticket Number",
      selector: (row) => row.ticketNumber,
      cell: (row) => (
        <div className="d-flex flex-column">
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(row.ticketNumber)}
          >
            <span>{row.ticketNumber}</span>
          </OverlayTrigger>
        </div>
      ),
      sortable: true,
      width: "150px",
    },
    {
      id: "date",
      name: "Date",
      selector: (row) =>
        Helpers.convertUtcToTimezone(row.createdAt, "M/D/YYYY"),
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(
            Helpers.convertUtcToTimezone(row.createdAt, "M/D/YYYY")
          )}
        >
          <span>
            {row.createdAt
              ? Helpers.convertUtcToTimezone(row.createdAt, "M/D/YYYY")
              : "--"}
          </span>
        </OverlayTrigger>
      ),
      reorder: true,
      sortable: true,
      style: { cursor: "text" },
      width: "120px",
    },
    {
      id: "category",
      name: "Category",
      selector: (row) => row.category,
      cell: (row) => {
        const text = row.category || "--";
        const displayText = text.length > 20 ? text.slice(0, 20) + "..." : text;
        return (
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(text)}
          >
            <span>{displayText}</span>
          </OverlayTrigger>
        );
      },
      sortable: true,
      width: "150px",
    },
    {
      id: "subject",
      name: "Subject",
      selector: (row) => row.subject,
      cell: (row) => {
        const text = row.subject || "--";
        const displayText = text.length > 30 ? text.slice(0, 30) + "..." : text;
        return (
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(text)}
          >
            <span
              onClick={() => mesagesPopUpOpen(row)}
              className="subjectTicket"
            >
              {displayText}
            </span>
          </OverlayTrigger>
        );
      },
      sortable: true,
      width: "150px",
    },
    {
      id: "message",
      name: "Message",
      selector: (row) => row.message,
      cell: (row) => {
        const text = row.message || "--";
        const displayText = text.length > 60 ? text.slice(0, 60) + "..." : text;
        return (
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(text)}
          >
            <span>{displayText}</span>
          </OverlayTrigger>
        );
      },
      sortable: true,
      width: "260px",
    },
    {
      id: "firstName",
      name: "Posted By",
      selector: (row) => row.firstName,
      cell: (row) => (
        <OverlayTrigger
          placement="bottom"
          delay={{ show: 250, hide: 400 }}
          overlay={renderTooltip(`${row.firstName} ${row.lastName}`)}
        >
          <span>{`${row.firstName} ${row.lastName}` || "--"}</span>
        </OverlayTrigger>
      ),
      sortable: true,
      width: "150px",
    },
    {
      id: "email",
      name: "Contact Detail",
      selector: (row) => row.email,
      cell: (row) => (
        <div className="d-flex flex-column">
          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(row.email)}
          >
            <a
              href={`mailto:${row.email}`}
              className="d-flex align-items-center text-decoration-none"
            >
              <i className="mdi mdi-email text-primary me-1"></i>
              <span>
                {row.email.length > 20
                  ? row.email.slice(0, 20) + "..."
                  : row.email}{" "}
              </span>
            </a>
          </OverlayTrigger>

          <OverlayTrigger
            placement="bottom"
            delay={{ show: 250, hide: 400 }}
            overlay={renderTooltip(row.mobile)}
          >
            <div className="d-flex align-items-center">
              <i className="mdi mdi-phone text-success me-1"></i>
              <span>{row.mobile || "--"}</span>
            </div>
          </OverlayTrigger>
        </div>
      ),
      sortable: true,
      width: "230px",
    },
    {
      id: "Priority",
      name: "Priority",
      cell: (row) =>
        row.priority ? (
          <select
            className="form-select form-select-sm text-white"
            value={row.priority || "LOW"}
            disabled
            style={{
              backgroundColor:
                row.priority === "HIGH"
                  ? "#F44336" // Red
                  : row.priority === "MEDIUM"
                  ? "#FFC107" // Yellow
                  : "#4CAF50", // Green for LOW or default
              color: row.priority === "MEDIUM" ? "black" : "white", // contrast for orange
            }}
          >
            <option value="LOW">Low</option>
            <option value="MEDIUM">Medium</option>
            <option value="HIGH">High</option>
          </select>
        ) : (
          "--"
        ),
      sortable: false,
      width: "135px",
    },
    {
      id: "Status",
      name: "Status",
      cell: (row) => (
        <select
          className="form-select form-select-sm text-white"
          style={{
            backgroundColor:
              row.status === "NEW"
                ? "#52cdff"
                : row.status === "INPROGRESS"
                ? "#ffa725"
                : row.status === "CLOSED"
                ? "#9E9E9E"
                : "",
            color: row.status === "INPROGRESS" ? "black" : "white", // better text contrast
          }}
          value={row.status}
          onChange={(e) => handleStatusChange(row.id, e.target.value)}
        >
          <option value="NEW">New</option>
          <option value="INPROGRESS">In Progress</option>
          <option value="CLOSED">Closed</option>
        </select>
      ),
      sortable: false,
      width: "150px",
    },
    {
      id: "assignedTo",
      name: "Assigned To",
      selector: (row) => row.assignedTo,
      cell: (row) =>
        row.assignedTo ? (
          <div className="d-flex align-items-center gap-2">
            <OverlayTrigger
              placement="bottom"
              delay={{ show: 250, hide: 400 }}
              overlay={renderTooltip(row.assignedTo)}
            >
              <span
                style={{ cursor: "pointer", textDecoration: "underline" }}
                onClick={() => mesagesPopUpOpen(row)}
              >
                {row.assignedUserFirstName &&
                row.assignedUserLastName &&
                row.assignedUserEmail
                  ? `${row.assignedUserFirstName} ${row.assignedUserLastName} - ${row.assignedUserEmail}`
                  : row.assignedUserEmail
                  ? row.assignedUserEmail
                  : row.assignedTo
                  ? row.assignedTo
                  : "--"}
              </span>
            </OverlayTrigger>
            <button
              className="btn btn-sm btn-outline-primary p-1"
              onClick={() => {
                rolesPopUpOpen(row);
                setNewAssign(true);
              }}
            >
              <i
                className="mdi mdi-account-plus"
                title="Re-assign to other user"
              ></i>
            </button>
          </div>
        ) : (
          <button
            className="btn btn-sm btn-outline-primary"
            onClick={() => rolesPopUpOpen(row)}
          >
            <i className="mdi mdi-account-tie"></i>
          </button>
        ),
      sortable: true,
      width: "220px",
    },
    {
      id: "action",
      name: "Action",
      button: true,
      cell: (row) => (
        <div className="d-flex align-items-center">
          <button className="font-size" onClick={() => handleShowDelete(row)}>
            <i
              className="mdi mdi-delete-forever text-md text-danger mx-2 font-size"
              title="Delete"
            ></i>
          </button>
        </div>
      ),
      width: "50px",
    },
  ];

  return (
    <div className="row ext-space">
      <div className="col-12">
        <Card className="mb-3 shadow-sm border-0">
          <Card.Body className="p-3">
            <h4 className="card-title">New Tickets</h4>
            <Row className="align-items-center g-2">
              {/* Filters Section */}
              <Col lg={7} md={12}>
                <div className="d-flex flex-wrap gap-2 align-items-center">
                  <Form.Select
                    value={categoryFilter}
                    onChange={(e) =>
                      handleFilterChange("category", e.target.value)
                    }
                    size="sm"
                    className="w-auto rounded-pill"
                  >
                    <option value="">Category</option>
                    <option value="Retreat Registration">
                      Retreat Registration
                    </option>
                    <option value="Retreat Financials">
                      Retreat Financials
                    </option>
                    <option value="Retreat Roster">Retreat Roster</option>
                    <option value="Retreat Closure">Retreat Closure</option>
                    <option value="Database Issues">Database Issues</option>
                    <option value="Retreat Reports">Retreat Reports</option>
                    <option value="Request to Feature">
                      Request to Feature
                    </option>
                    <option value="How do I...">How do I...</option>
                    <option value="Other">Other</option>
                  </Form.Select>

                  <Form.Select
                    value={statusFilter}
                    onChange={(e) =>
                      handleFilterChange("status", e.target.value)
                    }
                    size="sm"
                    className="w-auto rounded-pill"
                  >
                    <option value="">Status</option>
                    <option value="NEW">New</option>
                    <option value="INPROGRESS">In Progress</option>
                    <option value="CLOSED">Closed</option>
                  </Form.Select>

                  <Form.Select
                    value={priorityFilter}
                    onChange={(e) =>
                      handleFilterChange("priority", e.target.value)
                    }
                    size="sm"
                    className="w-auto rounded-pill"
                  >
                    <option value="">Priority</option>
                    <option value="LOW">Low</option>
                    <option value="MEDIUM">Medium</option>
                    <option value="HIGH">High</option>
                  </Form.Select>

                  {(categoryFilter || statusFilter || priorityFilter) && (
                    <Button
                      variant="outline-danger"
                      size="sm"
                      className="rounded-pill"
                      onClick={clearFilters}
                    >
                      Clear
                    </Button>
                  )}
                </div>
              </Col>

              {/* Search Section */}
              <Col lg={5} md={12}>
                <div className="position-relative shadow-sm rounded-pill overflow-hidden">
                  <Form.Control
                    type="text"
                    placeholder="Search tickets..."
                    value={searchKey}
                    onChange={handleSearch}
                    className="form-control border-0 py-2 ps-5"
                    style={{
                      height: "3rem",
                      borderRadius: "50px",
                      backgroundColor: "#f8f9fa",
                    }}
                  />
                  <span
                    className="position-absolute top-50 start-0 translate-middle-y ps-3"
                    style={{ color: "#6c757d", fontSize: "1.1rem" }}
                  >
                    🔍
                  </span>
                  <Button
                    variant="primary"
                    onClick={() => getAllTickets(1, pageSize, searchKey)}
                    className="position-absolute top-0 end-0 rounded-pill px-4 py-2"
                    style={{
                      borderTopLeftRadius: 0,
                      borderBottomLeftRadius: 0,
                      lineHeight: 2,
                    }}
                  >
                    Search
                  </Button>
                </div>
              </Col>
            </Row>
          </Card.Body>
        </Card>

        {/* Data Table */}
        <Card>
          <Card.Body>
            <CustomDataTable
              loading={LoadData}
              data={tickets}
              columns={columns}
              total={total}
              countPerPage={pageSize}
              isServerSide={true}
              pagination={true}
              sortServer={true}
              dataFetchApi={getAllTickets}
            />
          </Card.Body>
        </Card>
      </div>

      {/* Delete Modal */}
      {show && (
        <DeleteModel
          show={show}
          content={
            <>
              <h5 className="mt-4 fw-bold">Ticket Detail</h5>
              <div className="p-3 border rounded-3 bg-light shadow-sm">
                {selectedTicket && (
                  <>
                    <div className="mb-2">
                      <span className="fw-bold">Name:</span>{" "}
                      {selectedTicket.firstName} {selectedTicket.lastName}
                      {selectedTicket.email && (
                        <span className="text-muted">
                          {" "}
                          – {selectedTicket.email}
                        </span>
                      )}
                    </div>
                    <div className="mb-2">
                      <span className="fw-bold">Subject:</span>{" "}
                      {selectedTicket.subject}
                    </div>
                    <div className="mb-2">
                      <span className="fw-bold">Message:</span>{" "}
                      {selectedTicket.message}
                    </div>
                  </>
                )}
              </div>
            </>
          }
          handleClose={handleClose}
          DeleteData={deleteTicket}
        />
      )}

      {/* Assign roles popup */}
      {showRolesPopUp && (
        <TicketAccess
          show={showRolesPopUp}
          handleClose={rolesPopUpClose}
          selectedTicket={selectedTicket}
          newAssign={newAssign}
          setNewAssign={setNewAssign}
        />
      )}

      {/* Messages modal */}
      {messagesOpen && (
        <MessagesModal
          show={messagesOpen}
          handleClose={mesagesPopUpClose}
          selectedTicket={selectedTicket}
          setSelectedTicket={setSelectedTicket}
        />
      )}
    </div>
  );
};

export default Tickets;
