import { useState, useEffect } from "react";
import { toast } from "react-toastify";
import { Modal, Button, Form } from "react-bootstrap";
import HelpApi from "../../api/helpApi";
import AdminApi from "../../api/admin";
import RetreatApi from "../../api/retreatApi";
import CommitteeApi from "../../api/committeeApi";

function TicketAccess({
  show,
  handleClose,
  selectedTicket,
  newAssign,
  setNewAssign,
}) {
  let retreatApi = new RetreatApi();
  let adminApi = new AdminApi();
  let committeeApi = new CommitteeApi();
  let helpApi = new HelpApi();

  const [allCommittee, setAllCommittee] = useState([]);
  const [allAdminList, setAllAdminList] = useState([]);
  const [allSubAdminList, setAllSubAdminList] = useState([]);
  const [allRetreatSubAdminList, setAllRetreatSubAdminList] = useState([]);
  const [loadData, setLoadData] = useState(false);

  const [selectedRecipient, setSelectedRecipient] = useState({
    type: "",
    value: "",
    message: "",
    priority: "",
  });

  const [previewAttachment, setPreviewAttachment] = useState(null); // ✅ full-page attachment preview

  // Parse attachments JSON string
  const ticketAttachments = selectedTicket?.attachments
    ? JSON.parse(selectedTicket.attachments)
    : [];

  useEffect(() => {
    adminsAndSubAdmins();
    getAllretreatSubAdmins();
    allCommitteMembers();
    if (selectedTicket) {
      setSelectedRecipient((prev) => ({
        ...prev,
        message: selectedTicket.message || "",
      }));
    }
  }, [selectedTicket]);

  const adminsAndSubAdmins = async () => {
    await adminApi
      .geAdminsAndSubAdmins()
      .then(async (res) => {
        if (res.success) {
          if (res.admins) setAllAdminList(res.admins);
          if (res.subAdmins) setAllSubAdminList(res.subAdmins);
        } else {
          toast.error(res.msg);
        }
      })
      .catch((error) => toast.error(error));
  };

  const getAllretreatSubAdmins = async () => {
    await retreatApi
      .allretreatSubAdmins()
      .then(async (res) => {
        if (res.success) setAllRetreatSubAdminList(res.data);
        else toast.error(res.msg);
      })
      .catch((error) => toast.error(error));
  };

  const allCommitteMembers = async () => {
    setLoadData(true);
    await committeeApi
      .getAllCommitteMembers()
      .then((res) => {
        setLoadData(false);
        if (res.success) setAllCommittee(res.data);
        else toast.error(res.msg);
      })
      .catch((error) => {
        setLoadData(false);
        toast.error(error);
      });
  };

  const handleSelectChange = (type, value) => {
    if (!value) {
      setSelectedRecipient({
        type: "",
        value: "",
        message: selectedRecipient.message,
        priority: selectedRecipient.priority,
      });
    } else {
      setSelectedRecipient({ ...selectedRecipient, type, value });
    }
  };

  const isFieldDisabled = (fieldType) => {
    return selectedRecipient.type && selectedRecipient.type !== fieldType;
  };

  const handleSend = async () => {
    if (!selectedRecipient.value) {
      toast.error("Please select a recipient.");
      return;
    } // ✅ Validate message
    if (!selectedRecipient.message || selectedRecipient.message.trim() === "") {
      toast.error("Message is required.");
      return;
    }

    // ✅ Validate priority
    if (!selectedRecipient.priority) {
      toast.error("Priority is required.");
      return;
    }
    const isInputField = selectedRecipient.type === "techMember";
    await helpApi
      .sendTicketMsg({
        selectedTicket,
        selectedRecipient: selectedRecipient.value,
        isInputField,
        message: selectedRecipient.message,
        priority: selectedRecipient.priority,
      })
      .then((res) => {
        setLoadData(false);
        if (res.success) {
          setNewAssign(false);
          toast.success(`Ticket sent to ${selectedRecipient.value}`);
          handleClose();
        } else toast.error(res.msg);
      })
      .catch((error) => {
        setLoadData(false);
        toast.error(error);
      });
  };

  const resetData = () => {
    setSelectedRecipient({
      type: "",
      value: "",
      message: selectedTicket?.message || "",
      priority: "",
    });
  };

  return (
    <>
      <Modal
        show={show}
        onHide={() => {
          handleClose();
          setNewAssign(false);
        }}
        backdrop="static"
        size={newAssign ? "md" : "lg"}
        centered
        className="help-modal"
      >
        <Modal.Header closeButton>
          <Modal.Title className="fw-bold text-primary">
            {newAssign ? "Re-assign the Ticket" : "Assign the Ticket"}
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <div className="ticket-form-container">
            <Form>
              <div className="row">
                <div className="col-12">
                  <h4 className="ticketValues">
                    Category : {selectedTicket.category}
                  </h4>
                </div>

                {/* LEFT COLUMN: Recipients */}
                <div className={newAssign ? "col-12" : "col-6"}>
                  <Form.Group className="mb-3">
                    <Form.Label>Admins</Form.Label>
                    <Form.Select
                      value={
                        selectedRecipient.type === "admin"
                          ? selectedRecipient.value
                          : ""
                      }
                      disabled={isFieldDisabled("admin")}
                      onChange={(e) =>
                        handleSelectChange("admin", e.target.value)
                      }
                    >
                      <option value="">Select Admin</option>
                      {allAdminList.map((admin) => (
                        <option key={admin.id} value={admin.id}>
                          {`${admin.lastName}, ${admin.firstName} - ${admin.email}`}
                        </option>
                      ))}
                    </Form.Select>
                  </Form.Group>

                  <Form.Group className="mb-3">
                    <Form.Label>SubAdmins</Form.Label>
                    <Form.Select
                      value={
                        selectedRecipient.type === "subAdmin"
                          ? selectedRecipient.value
                          : ""
                      }
                      disabled={isFieldDisabled("subAdmin")}
                      onChange={(e) =>
                        handleSelectChange("subAdmin", e.target.value)
                      }
                    >
                      <option value="">Select SubAdmin</option>
                      {allSubAdminList.map((sub) => (
                        <option key={sub.id} value={sub.id}>
                          {`${sub.lastName}, ${sub.firstName} - ${sub.email}`}
                        </option>
                      ))}
                    </Form.Select>
                  </Form.Group>

                  <Form.Group className="mb-3">
                    <Form.Label>Retreat SubAdmins</Form.Label>
                    <Form.Select
                      value={
                        selectedRecipient.type === "retreatSubAdmin"
                          ? selectedRecipient.value
                          : ""
                      }
                      disabled={isFieldDisabled("retreatSubAdmin")}
                      onChange={(e) =>
                        handleSelectChange("retreatSubAdmin", e.target.value)
                      }
                    >
                      <option value="">Select Retreat SubAdmin</option>
                      {allRetreatSubAdminList.flatMap((retreat) =>
                        retreat.subAdmins.map((sub) => (
                          <option key={sub.id} value={sub.id}>
                            {`${sub.name}`}
                          </option>
                        ))
                      )}
                    </Form.Select>
                  </Form.Group>

                  <Form.Group>
                    <Form.Label>Assign To Tech Member</Form.Label>
                    <Form.Select
                      value={
                        selectedRecipient.type === "techMember"
                          ? selectedRecipient.value
                          : ""
                      }
                      onChange={(e) =>
                        handleSelectChange("techMember", e.target.value)
                      }
                      disabled={isFieldDisabled("techMember")}
                    >
                      <option value="">Select Tech Member</option>
                      <option value="manpreet02701@gmail.com">
                        Manpreet Singh (manpreet02701@gmail.com)
                      </option>
                    </Form.Select>
                  </Form.Group>
                </div>

                {!newAssign ? (
                  <div className="col-6">
                    <h5 className="ticketValues">
                      Subject : {selectedTicket.subject}
                    </h5>

                    <Form.Group className="mb-3">
                      <Form.Label className="fw-semibold">
                        Message <span className="text-danger">*</span>
                      </Form.Label>
                      <Form.Control
                        as="textarea"
                        rows={8}
                        required
                        name="message"
                        value={selectedRecipient.message || ""}
                        onChange={(e) =>
                          setSelectedRecipient((prev) => ({
                            ...prev,
                            message: e.target.value,
                          }))
                        }
                      />
                    </Form.Group>

                    <Form.Group>
                      <Form.Label>Priority</Form.Label>
                      <Form.Select
                        value={selectedRecipient.priority || ""}
                        onChange={(e) =>
                          setSelectedRecipient((prev) => ({
                            ...prev,
                            priority: e.target.value,
                          }))
                        }
                      >
                        <option value="">Select Priority</option>
                        <option value="LOW">Low</option>
                        <option value="MEDIUM">Medium</option>
                        <option value="HIGH">High</option>
                      </Form.Select>
                    </Form.Group>

                    {/* ✅ Attachments Section */}
                    {ticketAttachments.length > 0 && (
                      <div className="mt-3">
                        <Form.Label>Attachments</Form.Label>
                        <div className="d-flex flex-wrap gap-2 mt-1">
                          {ticketAttachments.map((url, idx) => {
                            const ext = url.split(".").pop().toLowerCase();
                            if (["jpg", "jpeg", "png", "gif"].includes(ext)) {
                              return (
                                <img
                                  key={idx}
                                  src={url}
                                  alt="attachment"
                                  style={{
                                    maxWidth: 100,
                                    maxHeight: 100,
                                    cursor: "pointer",
                                    borderRadius: 4,
                                  }}
                                  onClick={() => setPreviewAttachment(url)}
                                />
                              );
                            } else {
                              return (
                                <a
                                  key={idx}
                                  href={url}
                                  target="_blank"
                                  rel="noreferrer"
                                  className="d-block"
                                >
                                  {url.split("/").pop()}
                                </a>
                              );
                            }
                          })}
                        </div>
                      </div>
                    )}
                  </div>
                ) : null}
              </div>

              {/* SEND Button */}
              <div className="d-flex justify-content-end mt-3 gap-2">
                <Button variant="outline-secondary bold" onClick={resetData}>
                  RESET
                </Button>
                {newAssign ? (
                  <Button variant="success" onClick={handleSend}>
                    ASSIGN
                  </Button>
                ) : (
                  <Button variant="primary" onClick={handleSend}>
                    SEND
                  </Button>
                )}
              </div>
            </Form>
          </div>
        </Modal.Body>
      </Modal>

      {/* Full-page Attachment Preview */}
      {previewAttachment && (
        <div
          onClick={() => setPreviewAttachment(null)}
          style={{
            position: "fixed",
            top: 0,
            left: 0,
            width: "100vw",
            height: "100vh",
            backgroundColor: "rgba(0,0,0,0.8)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            zIndex: 2000, // ✅ higher than modal
            cursor: "zoom-out",
          }}
        >
          <img
            src={previewAttachment}
            alt="preview"
            style={{
              maxWidth: "90%",
              maxHeight: "90%",
              borderRadius: 8,
              boxShadow: "0 0 20px rgba(0,0,0,0.5)",
            }}
          />
        </div>
      )}
    </>
  );
}

export default TicketAccess;
