const cron = require("node-cron");
const db = require("../database/dbConnection");
const retreatSchema = db.retreat;
const retreatRoasterSchema = db.retreatRoaster;
const coupleSchema = db.couple;
const moment = require("moment-timezone");
const { MAILTOTREASUREPEOPLE } = require("../helper/emailTemplate");
const Email = require("../helper/sendEmail");

const autoConfirmPicturesAndCloseRetreats = () => {
  cron.schedule("0 0 * * *", async () => {
    try {
      const retreats = await retreatSchema.findAll({
        where: {
          [db.Sequelize.Op.and]: [
            db.Sequelize.literal(`NOW() >= DATE_ADD(dateTo, INTERVAL 14 DAY)`),
            db.Sequelize.literal(`dateTo >= DATE_SUB(NOW(), INTERVAL 60 DAY)`),
            { retreatCloseStatus: 0 },
          ],
        },
        raw: true,
      });

      for (const retreat of retreats) {
        const roasters = await retreatRoasterSchema.findAll({
          where: { retreatId: retreat.id },
          attributes: ["coupleId"],
          raw: true,
        });
        const coupleIds = roasters.map((r) => r.coupleId);
        if (coupleIds.length === 0) {
          console.log(
            `⚠️ No couples found for retreat ${retreat.id}, skipping picture auto-confirm`
          );
          continue;
        }
        const couples = await coupleSchema.findAll({
          where: { id: coupleIds },
          attributes: ["id", "imageUrl"],
          raw: true,
        });
        const allHaveImages = couples.every(
          (c) => c.imageUrl && c.imageUrl.trim() !== ""
        );
        if (!allHaveImages) {
          console.log(
            `⛔ Not all couples have uploaded images for retreat ${retreat.id}, skipping picture auto-confirm`
          );
          continue;
        }
        let confirmationArray;
        try {
          if (
            !retreat.confirmation ||
            retreat.confirmation === "[]" ||
            retreat.confirmation === null
          ) {
            confirmationArray = [
              {
                roasterData: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Directory: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Program: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                pictures: true,
                confirmedByName: "SYSTEM AUTO-CONFIRM",
                confirmedByRole: "SYSTEM",
                confirmedByEmail: "info@lovestrongmarriage.com",
                ConfirmedDate: moment()
                  .tz("America/New_York")
                  .format("YYYY-MM-DD HH:mm:ss"),
              },
              {
                retreatStatus: "ACTIVE",
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
            ];
          } else {
            confirmationArray = JSON.parse(retreat.confirmation);
            let picturesFound = false;
            confirmationArray = confirmationArray.map((entry) => {
              if (entry.hasOwnProperty("pictures")) {
                picturesFound = true;
                return {
                  ...entry,
                  pictures: true,
                  confirmedByName: "SYSTEM AUTO-CONFIRM",
                  confirmedByRole: "SYSTEM",
                  confirmedByEmail: "info@lovestrongmarriage.com",
                  ConfirmedDate: moment()
                    .tz("America/New_York")
                    .format("YYYY-MM-DD HH:mm:ss"),
                };
              }
              return entry;
            });

            if (!picturesFound) {
              confirmationArray.push({
                pictures: true,
                confirmedByName: "SYSTEM AUTO-CONFIRM",
                confirmedByRole: "SYSTEM",
                confirmedByEmail: "info@lovestrongmarriage.com",
                ConfirmedDate: moment()
                  .tz("America/New_York")
                  .format("YYYY-MM-DD HH:mm:ss"),
              });
            }
          }
        } catch (err) {
          console.error(
            `❌ Invalid confirmation JSON for retreat ID ${retreat.id}`,
            err.message
          );
          confirmationArray = [
            {
              roasterData: false,
              confirmedByName: "",
              confirmedByRole: "",
              confirmedByEmail: "",
              ConfirmedDate: "",
            },
            {
              Directory: false,
              confirmedByName: "",
              confirmedByRole: "",
              confirmedByEmail: "",
              ConfirmedDate: "",
            },
            {
              Program: false,
              confirmedByName: "",
              confirmedByRole: "",
              confirmedByEmail: "",
              ConfirmedDate: "",
            },
            {
              pictures: true,
              confirmedByName: "SYSTEM AUTO-CONFIRM",
              confirmedByRole: "SYSTEM",
              confirmedByEmail: "info@lovestrongmarriage.com",
              ConfirmedDate: moment()
                .tz("America/New_York")
                .format("YYYY-MM-DD HH:mm:ss"),
            },
            {
              retreatStatus: "ACTIVE",
              confirmedByName: "",
              confirmedByRole: "",
              confirmedByEmail: "",
              ConfirmedDate: "",
            },
          ];
        }
        const allRequiredConfirmed =
          confirmationArray.some((e) => e.roasterData === true) &&
          confirmationArray.some((e) => e.Directory === true) &&
          confirmationArray.some((e) => e.Program === true) &&
          confirmationArray.some((e) => e.pictures === true);

        if (allRequiredConfirmed) {
          await retreatSchema.update(
            {
              confirmation: JSON.stringify(confirmationArray),
              retreatCloseStatus: 1,
            },
            { where: { id: retreat.id } }
          );

          const updatedRetreat = await retreatSchema.findOne({
            where: { id: retreat.id },
            raw: true,
          });
          const treasurerEmails = [
            // { name: "Irene Mitschke", email: "manpreet02701@gmail.com" },
              { name: "Jimmie Flores", email: "jimmieflores@gmail.com" },
          ];

          for (const person of treasurerEmails) {
            const replData = MAILTOTREASUREPEOPLE.replace(
              /#name#/g,
              person.name || ""
            )
              .replace(/#retreat#/g, updatedRetreat?.title || "")
              .replace(
                /#date#/g,
                moment(updatedRetreat.dateFrom).isSame(
                  updatedRetreat.dateTo,
                  "day"
                )
                  ? `(${moment(updatedRetreat.dateFrom)
                      .tz("America/New_York")
                      .format("MMM-DD-YYYY")})`
                  : `(${moment(updatedRetreat.dateFrom)
                      .tz("America/New_York")
                      .format("MMM DD")} - ${moment(updatedRetreat.dateTo)
                      .tz("America/New_York")
                      .format("MMM DD")}, ${moment(updatedRetreat.dateFrom)
                      .tz("America/New_York")
                      .format("YYYY")})`
              );

            await Email.sendEmail(
              person.email,
              "Retreat Validated and Closed",
              replData
            );
          }
        } else {
          await retreatSchema.update(
            { confirmation: JSON.stringify(confirmationArray) },
            { where: { id: retreat.id } }
          );
        }
      }

      console.log("✅ Auto-confirm + auto-close cron completed.");
    } catch (error) {
      console.error(
        "❌ Error in auto-confirm/close retreat cron:",
        error.message
      );
    }
  });
};

module.exports = autoConfirmPicturesAndCloseRetreats;
