const db = require("../database/dbConnection");
const { Op } = require("sequelize");
const parishSchema = db.parishCategories;

class ParishCategoriesController {
  async addCategory(req, res) {
    try {
      const { name, forceSave } = req.body;
      const cleanedName = name.trim().toLowerCase();

      // Exact Match Check (case-insensitive + trim)
      const exactMatch = await parishSchema.findOne({
        where: db.Sequelize.where(
          db.Sequelize.fn(
            "lower",
            db.Sequelize.fn("trim", db.Sequelize.col("name"))
          ),
          cleanedName
        ),
      });

      if (exactMatch && !forceSave) {
        return res.status(200).send({
          success: false,
          msg: "This category already exists.",
        });
      }

      // Save after trimming input
      const newCategory = await parishSchema.create({ name: name.trim() });

      return res.status(200).send({
        success: true,
        msg: "New Parish Category created.",
        data: newCategory,
      });
    } catch (error) {
      return res.status(500).send({
        success: false,
        msg: error.message,
      });
    }
  }

  async getAllCategories(req, res) {
    try {
      const categories = await parishSchema.findAll({
        order: [["createdAt", "DESC"]],
      });
      return res.status(200).send({
        success: true,
        msg: "All Parish Categories",
        data: categories.map((x) => ({
          id: x.id,
          name: x.name,
          createdAt: x.createdAt,
        })),
      });
    } catch (error) {
      return res.status(500).send({ success: false, msg: error.message });
    }
  }

  async updateCategory(req, res) {
    try {
      const { name } = req.body;
      const categoryId = req.params.id;
      const cleanedName = name.trim().toLowerCase();

      // Check for duplicate (ignore same id)
      const duplicate = await parishSchema.findOne({
        where: {
          [Op.and]: [
            db.Sequelize.where(
              db.Sequelize.fn(
                "lower",
                db.Sequelize.fn("trim", db.Sequelize.col("name"))
              ),
              cleanedName
            ),
            { id: { [Op.ne]: categoryId } },
          ],
        },
      });

      if (duplicate) {
        return res.status(200).send({
          success: false,
          msg: "This category name already exists.",
        });
      }

      const updated = await parishSchema.update(
        { name: name.trim() },
        { where: { id: categoryId } }
      );

      return res.status(200).send({
        success: true,
        msg: "Parish Category updated",
        data: updated,
      });
    } catch (error) {
      return res.status(500).send({ success: false, msg: error.message });
    }
  }

  async deleteCategory(req, res) {
    try {
      const categoryId = req.params.id;

      // 1️⃣ Check if any parishes exist for this category
      const parishCount = await db.parish.count({
        where: { parishCategoryId: categoryId },
      });

      if (parishCount > 0) {
        return res.status(200).send({
          success: false,
          msg: "This category is assigned to parishes. Please delete the associated parishes first.",
        });
      }

      // 2️⃣ Proceed to delete if no parish exists
      const deleted = await parishSchema.destroy({
        where: { id: categoryId },
      });

      return res.status(200).send({
        success: true,
        msg: "Parish Category deleted",
        data: deleted,
      });
    } catch (error) {
      return res.status(500).send({ success: false, msg: error.message });
    }
  }

  async searchCategory(req, res) {
    try {
      const key = req.params.key;
      const result = await parishSchema.findAll({
        where: {
          name: { [Op.like]: `%${key}%` },
        },
        raw: true,
      });
      return res.status(200).send({
        success: true,
        msg: "",
        data: result,
      });
    } catch (error) {
      return res.status(500).send({ success: false, msg: error.message });
    }
  }
}

module.exports = new ParishCategoriesController();
