const moment = require("moment-timezone");
const db = require("../database/dbConnection");
const fs = require("fs");
const path = require("path");

const retreatSchema = db.retreat;
const coupleSchema = db.couple;
const parishSchema = db.parish;
const userSchema = db.user;
const tempRetreatSchema = db.tempRetreat;
const perishSchema = db.parish;
const retreatRoastSchema = db.retreatRoaster;
const RolesSchema = db.roles;
const committeeMemberSchema = db.committeeMember;

const { PROJECT_DIR } = require("../../config");
const {
  RETREATACCESS,
  APPROVEREJECTRETREAT,
  SENDLOGINDETAILBYADMIN,
  MAILTOTREASUREPEOPLE,
} = require("../helper/emailTemplate");
const { Op } = require("sequelize");
const Email = require("../helper/sendEmail");

class RetreatController {
  async createRetreat(req, res) {
    try {
      let {
        parish,
        title,
        language,
        location,
        id,
        schedule,
        heading,
        mission_statement,
        description,
        mail_msg,
      } = JSON.parse(`${req.body.fromData}`);
      if (schedule && schedule.length) {
        let dateFrom;
        let dateTo;
        if (schedule.length > 0) {
          dateFrom = new Date(schedule[0].date);
          dateTo = new Date(schedule[schedule.length - 1].date);
        } else {
          dateFrom = null;
          dateTo = null;
        }
        let saveObj = {
          title: title,
          parishId: parish,
          language: language,
          location: location,
          schedule: JSON.stringify(schedule),
          heading: heading,
          mission_statement: mission_statement,
          description: description,
          dateFrom: dateFrom,
          dateTo: dateTo,
          mail_msg: mail_msg,
        };
        let newRetreat = new retreatSchema(saveObj);
        let succ = await newRetreat.save();
        let imageSavedCouple;
        if (req.files) {
          imageSavedCouple = await this.saveImagesForRetreat(
            req.files,
            succ.id
          );
        }
        try {
          let update = await retreatSchema.update(
            { mission_statement: mission_statement },
            {
              where: { language: language },
            }
          );
        } catch (error) {
          console.error("Error updating records:", error);
        }
        if (succ) {
          return res.status(200).send({
            success: true,
            msg: "Retreat added successfully.",
            data: succ,
          });
        } else {
          return res.status(200).send({
            msg: `Something went wrong. Please try again.`,
            error: true,
            success: false,
          });
        }
      } else {
        return res.status(200).json({
          success: false,
          msg: "Atleast one Date is required.",
        });
      }
    } catch (error) {
      console.log(
        `createRetreat [${new Date().toLocaleString("en-US", {
          timeZoneName: "short",
        })}] error -->`,
        error
      );
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async saveImagesForRetreat(files, retreatId) {
    try {
      if (files && files.file1) {
        let file1 = files.file1;
        let fileName1 = file1.name;
        let fileUrl1 = `${PROJECT_DIR}/retreatImage/${fileName1}`;
        let imgUrl1 = `${process.env.SERVERADDRESS}/public/retreatImage/${fileName1}`;
        // Update the retreat schema with image URL
        await retreatSchema.update(
          { image1: imgUrl1 },
          { where: { id: retreatId } }
        );

        // Move file1 asynchronously
        const moveFile1 = new Promise((resolve, reject) => {
          file1.mv(fileUrl1, (err) => {
            if (err) {
              reject(err);
            } else {
              resolve();
            }
          });
        });

        // Wait for file movement to complete
        await moveFile1;
      }

      // Check if file2 exists
      if (files && files.file2) {
        let file2 = files.file2;
        let fileName2 = file2.name;
        let fileUrl2 = `${PROJECT_DIR}/retreatImage/${fileName2}`;
        let imgUrl2 = `${process.env.SERVERADDRESS}/public/retreatImage/${fileName2}`;
        // Update the retreat schema with image URL
        await retreatSchema.update(
          { image2: imgUrl2 },
          { where: { id: retreatId } }
        );

        // Move file2 asynchronously
        const moveFile2 = new Promise((resolve, reject) => {
          file2.mv(fileUrl2, (err) => {
            if (err) {
              reject(err);
            } else {
              resolve();
            }
          });
        });
        await moveFile2;
      }

      // Return true if everything is successful
      return true;
    } catch (error) {
      console.error("Error saving images:", error);
      throw error;
    }
  }

  async getAllRetreats(req, res) {
    try {
      let {
        searchKeyword,
        sortBy,
        pageNo,
        size,
        allRetreat,
        coupleId,
        committeeId,
        parishValue,
      } = req.body;

      let query = { skip: 0, limit: 0 };
      let sort1 = "DESC";
      let sortColumn = "retreats.dateFrom";
      let searchColumn = "";
      let retreatIdList = [];

      // Pagination - only apply if not requesting all records
      if ((pageNo || size) && !allRetreat) {
        pageNo = parseInt(pageNo);
        size = parseInt(size);
        if (pageNo < 1) {
          return res.status(200).send({
            error: true,
            msg: "Invalid page number, should start with 1",
          });
        }
        query.skip = size * (pageNo - 1);
        query.limit = size;
      }

      // Sorting
      if (sortBy && sortBy.sortDirection == 1) {
        sort1 = "ASC";
      }

      if (sortBy && sortBy.sortColumn && sortBy.sortDirection) {
        const columnMap = {
          language: "retreats.language",
          location: "retreats.location",
          title: "retreats.title",
          heading: "retreats.heading",
          mission_statement: "retreats.mission_statement",
          description: "retreats.description",
        };
        sortColumn = columnMap[sortBy.sortColumn] || "retreats.dateFrom";
      }

      // Search filter
      if (searchKeyword && searchKeyword !== "") {
        searchColumn = `retreats.title LIKE "%${searchKeyword}%" OR
    retreats.language LIKE "%${searchKeyword}%" OR
    retreats.location LIKE "%${searchKeyword}%" OR
    retreats.heading LIKE "%${searchKeyword}%" OR
    retreats.mission_statement LIKE "%${searchKeyword}%" OR
    retreats.description LIKE "%${searchKeyword}%"`;
      }

      if (committeeId) {
        const committees = await committeeMemberSchema.findAll({
          where: { userId: committeeId },
        });

        if (committees && committees.length > 0) {
          try {
            let retreatObjects = [];

            committees.forEach((committee) => {
              let retreatsData = Array.isArray(committee.retreats)
                ? committee.retreats
                : JSON.parse(committee.retreats);

              retreatObjects.push(...retreatsData);
            });

            retreatIdList = retreatObjects
              .filter((r) => !r.disabled)
              .map((r) => r.retreatId);

            // remove duplicates
            retreatIdList = [...new Set(retreatIdList)];

            // if user has committee but no active retreats
            if (retreatIdList.length === 0) {
              return res.status(200).send({
                success: true,
                data: [],
                totalRecords: 0,
              });
            }
          } catch (err) {
            console.error("Error parsing committee.retreats:", err);
            return res.status(200).send({
              success: false,
              msg: "Invalid format for committee retreats.",
            });
          }
        }
      }

      // WHERE conditions
      let whereConditions = [
        `(retreats.deleteStatus='' OR retreats.deleteStatus IS NULL)`,
      ];

      if (searchKeyword && searchKeyword !== "") {
        whereConditions.push(`(${searchColumn})`);
      }

      if (coupleId && coupleId !== "") {
        whereConditions.push(
          `retreatroasters.coupleId = ${parseInt(coupleId)}`
        );
      }

      if (retreatIdList.length > 0) {
        whereConditions.push(`retreats.id IN (${retreatIdList.join(",")})`);
      }

      if (parishValue && parishValue !== "") {
        whereConditions.push(`retreats.parishId = ${parseInt(parishValue)}`);
      }

      const whereClause = " WHERE " + whereConditions.join(" AND ");

      // SELECT clause
      const select = `
    SELECT 
      retreats.id, retreats.title, retreats.status, retreats.language, retreats.freeze,retreats.parishId,
      retreats.location, retreats.heading,retreats.inactiveReason, retreats.mission_statement, retreats.dateFrom, 
      retreats.dateTo, retreats.description, retreats.schedule, retreats.image1, 
      retreats.image2, retreats.createdAt, retreats.contact_name, retreats.contact_email, 
      retreats.contact_mobile, retreats.directoryFile, retreats.confirmation, 
      retreats.programFile,
      parishes.parish AS parishName, parishes.city AS parishCity, 
      parishes.address AS parishAddress, parishes.state AS parishState, 
      parishes.zip AS parishzip, parishes.link AS parishLink, 
      parishes.imageUrl AS parishImageUrl,`;

      const count = `
    COUNT(IF(retreatroasters.attendeeType = 'Retreatant' AND retreatroasters.stage = 'VERIFIED', retreatroasters.id, NULL)) AS countRetreatant,
    COUNT(retreatroasters.id) AS roasterTotal,
    COUNT(IF(retreatroasters.attendeeType = 'Team' AND retreatroasters.stage = 'VERIFIED', retreatroasters.id, NULL)) AS countTeam
  `;

      const from = `
    FROM retreats
    LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
    LEFT JOIN parishes ON retreats.parishId = parishes.id
  `;

      const whereFrom = from + whereClause;

      let groupBy = ` GROUP BY retreats.id`;
      let orderAndLimit = ` ORDER BY ${sortColumn} ${sort1}`;

      // Only add LIMIT if not requesting all records
      if (!allRetreat) {
        orderAndLimit += ` LIMIT ${query.limit} OFFSET ${query.skip}`;
      }

      const finalQuery = `${select} ${count} ${whereFrom} ${groupBy} ${orderAndLimit}`;
      const totalCountQuery = `SELECT COUNT(DISTINCT retreats.id) as totalRetreat ${whereFrom}`;

      const responseData = await db.sequelize
        .query(finalQuery)
        .then(([results]) => results);

      const totalCount = await db.sequelize
        .query(totalCountQuery)
        .then(([results]) => results[0]);

      return res.status(200).send({
        success: true,
        data: responseData,
        totalRecords: totalCount.totalRetreat,
      });
    } catch (error) {
      console.error("Error in getAllRetreats:", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getAllRetreatSearch(req, res) {
    try {
      let select = `SELECT retreats.id,retreats.title,retreats.language,retreats.location,retreats.dateFrom, retreats.dateTo, retreats.createdAt,parishes.parish AS parishName,parishes.id AS parishId`;
      let from = ` FROM retreats 
      LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
      LEFT JOIN parishes ON retreats.parishId = parishes.id
      WHERE (retreats.deleteStatus='' OR retreats.deleteStatus IS NULL) 
      GROUP BY retreats.id 
      ORDER BY parishName ASC, retreats.dateFrom DESC`;

      let queryRecords = select + from;

      let responseData1 = await db.sequelize
        .query(queryRecords)
        .then(([results, metadata]) => results);

      return res.status(200).send({
        success: true,
        data: responseData1,
      });
    } catch (error) {
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getRetreatDetail(req, res) {
    try {
      let foundRetreat = await retreatSchema.findAll({
        where: { id: req.body.id },
      });

      if (!foundRetreat || foundRetreat.length === 0) {
        return res.status(404).send({
          success: false,
          msg: "Retreat not found.",
        });
      }

      // Parse `confirmation` if JSON
      if (foundRetreat && foundRetreat[0].confirmation) {
        try {
          foundRetreat[0].confirmation = JSON.parse(
            foundRetreat[0].confirmation
          );
        } catch (error) {
          console.error("Error parsing confirmation:", error);
        }
      }

      let accessIds = foundRetreat[0].dataValues.accessIds;

      let userEmails = [];
      if (accessIds !== null) {
        if (typeof accessIds === "string") {
          accessIds = JSON.parse(accessIds);
        }

        // ✅ Convert old format [id1, id2] → [{ userId, isDisabled:false }]
        if (Array.isArray(accessIds) && typeof accessIds[0] !== "object") {
          accessIds = accessIds.map((id) => ({
            userId: id,
            isDisabled: false,
          }));
        }

        // Collect all userIds
        let userIds = accessIds.map((u) => u.userId);

        // Fetch user details
        let foundUsers = await userSchema.findAll({
          where: { id: { [Op.in]: userIds } },
        });

        // Build email list with disable flag
        userEmails = foundUsers.map((user) => {
          let fullName = "";

          if (user.lastName) fullName += user.lastName;
          if (user.firstName) {
            fullName += (user.lastName ? ", " : "") + user.firstName;
          }

          let label = fullName
            ? `${fullName} - ${user.email || ""}`
            : user.email || "";

          // Match disable flag from accessIds
          let access = accessIds.find((a) => a.userId === user.id);

          return {
            label,
            isDisabled: access ? access.isDisabled : false,
          };
        });

        foundRetreat = foundRetreat.map((retreat) => ({
          ...retreat.dataValues,
          userEmails,
        }));
      } else {
        foundRetreat = foundRetreat.map((retreat) => retreat.dataValues);
      }

      // Add parish details if present
      if (foundRetreat[0].parishId) {
        let parishId = foundRetreat[0].parishId;
        let foundParish = await parishSchema.findOne({
          where: { id: parishId },
        });

        if (foundParish) {
          foundRetreat = foundRetreat.map((retreat) => ({
            ...retreat,
            myparish: foundParish.dataValues,
          }));
        }
      }

      return res.status(200).send({
        success: true,
        data: foundRetreat,
        msg: "Retreat found.",
      });
    } catch (error) {
      console.error(error);
      return res.status(500).send({
        error: true,
        msg: "Something went wrong, please try again.",
      });
    }
  }

  async getRetreatMetaData(retreatId) {
    try {
      let foundRetreat = await retreatSchema.findOne({
        where: { id: retreatId },
        attributes: [
          "title",
          "image1",
          "image2",
          "language",
          "dateFrom",
          "dateTo",
        ],
      });

      if (!foundRetreat) {
        return null;
      }

      const retreatData = foundRetreat.dataValues;

      // Determine the image to use
      let imageUrl =
        retreatData.image1 ||
        retreatData.image2 ||
        "https://lovestrong.koolderbyacademy.com/logo.png";

      // Hardcoded descriptions based on language
      const descriptions = {
        english:
          "The mission of the LOVESTRONG Marriage Ministry is to support, strengthen, and nurture the Christ-centered marital bond between a man and a woman utilizing an experiential-based weekend retreat program.",
        spanish:
          "La mission del del Ministerio de Matrimonios LOVESTRONG es el de apoyar, fortalecer y alimentar la unión centrada en Cristo, entre parejas casadas de nuestra Iglesia y Comunidad",
      };

      let description =
        retreatData.language === "spanish"
          ? descriptions.spanish
          : descriptions.english;

      // Format the date range
      let formattedDate = "";
      if (retreatData.dateFrom && retreatData.dateTo) {
        formattedDate = moment(retreatData.dateFrom).isSame(
          retreatData.dateTo,
          "day"
        )
          ? `(${moment(retreatData.dateFrom)
              .tz("America/New_York")
              .format("MMM-DD-YYYY")})`
          : `(${moment(retreatData.dateFrom)
              .tz("America/New_York")
              .format("MMM DD")} - ${moment(retreatData.dateTo)
              .tz("America/New_York")
              .format("MMM DD")}, ${moment(retreatData.dateFrom)
              .tz("America/New_York")
              .format("YYYY")})`;
      }

      return {
        title: `${retreatData.title || "Lovestrong Marriage"} ${formattedDate}`,
        description: description,
        imageUrl: imageUrl,
      };
    } catch (error) {
      console.error(error);
      return null;
    }
  }

  async editRetreat(req, res) {
    try {
      let {
        parish,
        title,
        language,
        location,
        id,
        schedule,
        heading,
        mission_statement,
        description,
        image1,
        image2,
        mail_msg,
      } = JSON.parse(`${req.body.fromData}`);
      if (id) {
        if (schedule && schedule.length) {
          let dateFrom;
          let dateTo;
          if (schedule.length > 0) {
            dateFrom = new Date(schedule[0].date);
            dateTo = new Date(schedule[schedule.length - 1].date);
          } else {
            dateFrom = null;
            dateTo = null;
          }
          let retreatToUpdate = {
            title: title,
            parishId: parish,
            language: language,
            location: location,
            schedule: JSON.stringify(schedule),
            heading: heading,
            mission_statement: mission_statement,
            description: description,
            dateFrom: dateFrom,
            dateTo: dateTo,
            mail_msg: mail_msg,
          };
          let update = await retreatSchema.update(retreatToUpdate, {
            where: { id: id },
          });
          let retreatDetail = await retreatSchema.findOne({
            where: { id: id },
            raw: true,
          });
          if (update) {
            let imageSavedRetreat;
            if (req.files) {
              if (retreatDetail) {
                if (retreatDetail.image1) {
                  let retreatImagePath = path.join(
                    PROJECT_DIR,
                    "retreatImage",
                    path.basename(retreatDetail.image1)
                  );
                  if (fs.existsSync(retreatImagePath)) {
                    fs.unlinkSync(retreatImagePath);
                  }
                }
                if (retreatDetail.image2) {
                  let retreatImagePath = path.join(
                    PROJECT_DIR,
                    "retreatImage",
                    path.basename(retreatDetail.image2)
                  );
                  if (fs.existsSync(retreatImagePath)) {
                    fs.unlinkSync(retreatImagePath);
                  }
                }
              }
              imageSavedRetreat = await this.saveImagesForRetreat(
                req.files,
                id
              );
            } else {
              if (image1 === "") {
                if (retreatDetail) {
                  if (retreatDetail.image1) {
                    let retreatImagePath = path.join(
                      PROJECT_DIR,
                      "retreatImage",
                      path.basename(retreatDetail.image1)
                    );
                    if (fs.existsSync(retreatImagePath)) {
                      fs.unlinkSync(retreatImagePath);
                    }
                  }
                }
                let removeImage1 = await retreatSchema.update(
                  { image1: "" },
                  { where: { id: id } }
                );
              }
              if (image2 === "") {
                if (retreatDetail) {
                  if (retreatDetail.image2) {
                    let retreatImagePath = path.join(
                      PROJECT_DIR,
                      "retreatImage",
                      path.basename(retreatDetail.image2)
                    );
                    if (fs.existsSync(retreatImagePath)) {
                      fs.unlinkSync(retreatImagePath);
                    }
                  }
                }
                let removeImage2 = await retreatSchema.update(
                  { image2: "" },
                  { where: { id: id } }
                );
              }
            }
            let foundRetreat = await retreatSchema.findOne({
              where: { id: id },
            });

            if (!foundRetreat) {
              return res.status(404).send({
                success: false,
                msg: "Retreat not found.",
              });
            }

            let accessIds = foundRetreat.accessIds;

            if (accessIds !== null) {
              if (!Array.isArray(accessIds)) {
                accessIds = JSON.parse(accessIds);
              }

              let foundUsers = await userSchema.findAll({
                where: { id: { [Op.in]: accessIds } },
              });

              let userEmails = foundUsers.map((user) => user.email);

              // Adding userEmails to the foundRetreat object
              foundRetreat = {
                ...foundRetreat.dataValues,
                userEmails: userEmails,
              };
            }

            try {
              let update = await retreatSchema.update(
                { mission_statement: mission_statement },
                {
                  where: { language: language },
                }
              );
            } catch (error) {
              console.error("Error updating records:", error);
            }

            return res.send({
              success: true,
              msg: `Retreat updated successfully`,
              data: foundRetreat,
            });
          } else {
            return res.send({
              error: true,
              msg: "Something went wrong please try again.",
            });
          }
        } else {
          return res.status(200).json({
            success: false,
            msg: "Atleast one Date is required.",
          });
        }
      } else {
        return res.status(200).json({
          success: false,
          msg: "Error in ID.",
        });
      }
    } catch (error) {
      console.log(
        `editRetreat [${new Date().toLocaleString("en-US", {
          timeZoneName: "short",
        })}] error -->`,
        error
      );
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async getRetreatLocation(req, res) {
    try {
      let select = `SELECT id, location`;
      let from = ` FROM retreats `;
      let where = `WHERE location IS NOT NULL AND location != ''`;
      let groupBy = ` GROUP by location `;
      let orderBy = ` ORDER BY location DESC `;
      let queryRecords = select + from + where + groupBy + orderBy;
      let responseData1 = await db.sequelize
        .query(queryRecords)
        .then(([results, metadata]) => results);
      return res.status(200).send({
        success: true,
        data: responseData1,
      });
    } catch (error) {
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async searchRetreatsAndCouples(req, res) {
    try {
      let {
        retreatId,
        parisheId,
        coupleId,
        dateFrom,
        dateTo,
        retreatAttended,
        cityState,
        location,
        primaryKey,
        language,
        serviceIds,
        roleIds,
        talkIds,
        accountStatus,
        neverServedTeam,
        neverServiceRole,
        neverTalk,
        missingMails,
        retreatants,
        searchFor,
        zip,
        is_Not_Head_Role,
      } = req.body;
      let SELECT = `SELECT 
                      retreats.id AS retreatId, 
                      retreats.parishId,
                      retreats.language,
                      retreats.dateFrom,
                      retreats.dateTo,
                      retreats.heading,
                      couples.id AS coupleId,
                      couples.herFirstName,
                      couples.hisFirstName,
                      couples.herLastName,
                      couples.hisLastName,
                      couples.hisMobile,
                      couples.herMobile,
                      couples.herEmail,
                      couples.hisEmail,
                      couples.city,
                      couples.state,
                      couples.accountStatus,
                      couples.primaryKey,
                      couples.zip,
couples.address,couples.allergies,couples.anniversary_date,couples.under_age_35,couples.hear_us,couples.referral_from_a_friend_or_family_member,couples.emergency_name1,couples.emergency_name2,couples.emergency_relationship1,couples.emergency_relationship2,couples.emergency_phone1,couples.emergency_phone2,
                      couples.createdAt,       
                      retreatroasters.retreatId,
                      retreatroasters.coupleId,`;

      let JOIN = `
                  JOIN retreats ON retreatroasters.retreatId = retreats.id
                  JOIN couples ON retreatroasters.coupleId = couples.id 
                  `;
      let WHERE =
        " WHERE (retreats.deleteStatus='' OR retreats.deleteStatus IS NULL)";
      let GROUPBY = "GROUP BY retreatroasters.id";

      if (Array.isArray(retreatId) && retreatId.length > 0) {
        WHERE += ` AND retreats.id IN (${retreatId}) `;
      } else {
        WHERE = WHERE;
      }
      if (Array.isArray(parisheId) && parisheId.length > 0) {
        SELECT += `parishes.id,parishes.parish,`;
        JOIN += ` JOIN parishes ON retreats.parishId = parishes.id`;
        if (searchFor == "retreat") {
          if (WHERE) {
            WHERE += ` AND retreats.parishId IN (${parisheId})`;
          } else {
            WHERE += ` WHERE retreats.parishId IN (${parisheId})`;
          }
        } else {
          if (WHERE) {
            WHERE += ` AND couples.parishId IN (${parisheId})`;
          } else {
            WHERE += ` WHERE couples.parishId IN (${parisheId})`;
          }
        }
      } else if (
        !retreatAttended &&
        !retreatants &&
        retreatId &&
        retreatId != "0"
      ) {
        WHERE += ` `;
      }
      // OR retreatroasters.parishId IS NOT NULL
      if (Array.isArray(coupleId) && coupleId.length > 0) {
        if (WHERE && coupleId !== "0") {
          WHERE += ` AND retreatroasters.coupleId IN (${coupleId})`;
        } else if (coupleId !== "0") {
          WHERE += ` WHERE retreatroasters.coupleId IN (${coupleId})`;
        }
      } else {
        WHERE = WHERE;
      }
      if (Array.isArray(language) && language.length > 0) {
        const languageValues = language.map((lang) => `'${lang}'`).join(", ");
        if (WHERE) {
          WHERE += ` AND retreats.language IN (${languageValues})`;
        } else {
          WHERE += ` WHERE retreats.language IN (${languageValues})`;
        }
      }

      if (cityState.length) {
        let stateArr = ``;
        let cityArr = ``;
        cityState.forEach((element) => {
          stateArr += `'` + element.value.split(" ")[0] + `'` + ",";
          cityArr = `'` + element.value.split(" ")[1] + `'` + ",";
        });
        stateArr = stateArr.substring(0, stateArr.length - 1);
        cityArr = cityArr.substring(0, cityArr.length - 1);
        if (WHERE) {
          WHERE += ` AND couples.city IN (${cityArr}) OR couples.state IN (${stateArr})`;
        } else {
          WHERE += ` WHERE couples.city IN (${cityArr}) OR couples.state IN (${stateArr})`;
        }
      }

      if (location != "") {
        let locationArr = ``;
        location.forEach((element) => {
          locationArr += `'` + element.value + `'` + ",";
        });
        locationArr = locationArr.substring(0, locationArr.length - 1);
        if (WHERE) {
          WHERE += ` AND retreats.location IN (${locationArr})`;
        } else {
          WHERE += ` WHERE retreats.location IN (${locationArr})`;
        }
      }

      if (zip != "") {
        let zipArr = ``;
        zip.forEach((element) => {
          zipArr += `'` + element.value + `'` + ",";
        });
        zipArr = zipArr.substring(0, zipArr.length - 1);
        if (WHERE) {
          WHERE += ` AND couples.zip IN (${zipArr})`;
        } else {
          WHERE += ` WHERE couples.zip IN (${zipArr})`;
        }
      }
      if (missingMails) {
        const condition = `(couples.hisEmail IS NULL OR couples.hisEmail = '' OR couples.herEmail IS NULL OR couples.herEmail = '')`;

        if (WHERE) {
          WHERE += ` AND ${condition}`;
        } else {
          WHERE = `WHERE ${condition}`;
        }
      }

      if (dateFrom && dateTo) {
        if (WHERE) {
          WHERE += ` AND retreats.dateFrom >= '${dateFrom}'`;
          WHERE += ` AND retreats.dateTo <= '${dateTo}'`;
        } else {
          WHERE += ` WHERE retreats.dateFrom >= '${dateFrom}'`;
          WHERE += ` AND retreats.dateTo <= '${dateTo}'`;
        }
      }

      if (accountStatus.length) {
        let status = ``;
        accountStatus.map((s) => (status += `'` + s + `'` + ","));
        status = status.substring(0, status.length - 1);
        if (WHERE) {
          WHERE += ` AND couples.accountStatus IN (${status})`;
        } else {
          WHERE += ` WHERE couples.accountStatus IN (${status})`;
        }
      }

      if (retreatants.length) {
        let retreatantValue = ``;
        retreatants.map((s) => (retreatantValue += `'` + s + `'` + ","));
        retreatantValue = retreatantValue.substring(
          0,
          retreatantValue.length - 1
        );
        if (WHERE) {
          WHERE += ` AND retreatroasters.attendeeType  IN (${retreatantValue})`;
        } else {
          WHERE += ` WHERE retreatroasters.attendeeType IN (${retreatantValue})`;
        }
      }
      if (primaryKey) {
        if (WHERE) {
          WHERE += ` AND couples.primaryKey = '${primaryKey}'`;
        } else {
          WHERE += ` WHERE couples.primaryKey = '${primaryKey}'`;
        }
      }
      if (serviceIds.length || roleIds.length || talkIds.length) {
        if (serviceIds.length) {
          SELECT += ` retreatroasters.serviceId,services.id,services.service,`;
          JOIN += ` JOIN services ON retreatroasters.serviceId = services.id`;
          if (WHERE) {
            WHERE += ` AND retreatroasters.serviceId IN (${serviceIds.join(
              ","
            )})`;
          } else {
            WHERE += ` WHERE retreatroasters.serviceId IN (${serviceIds.join(
              ","
            )})`;
          }
        }
        if (roleIds.length) {
          SELECT += ` retreatroasters.roleId,roles.id,roles.role,`;
          JOIN += ` JOIN roles ON retreatroasters.roleId = roles.id`;
          if (is_Not_Head_Role !== "" && is_Not_Head_Role !== undefined) {
            if (WHERE) {
              WHERE += ` AND couples.id NOT IN (SELECT rr1.coupleId FROM retreatroasters AS rr1 WHERE rr1.roleId NOT IN (${roleIds.join(
                ","
              )}))`;
            } else {
              WHERE += ` WHERE couples.id NOT IN (SELECT rr1.coupleId FROM retreatroasters AS rr1 WHERE rr1.roleId NOT IN (${roleIds.join(
                ","
              )}))`;
            }
          } else {
            if (WHERE) {
              WHERE += ` AND retreatroasters.roleId IN (${roleIds.join(",")})`;
            } else {
              WHERE += ` WHERE retreatroasters.roleId IN (${roleIds.join(
                ","
              )})`;
            }
          }
        }
        if (talkIds.length) {
          SELECT += ` retreatroasters.talkId,talks.id,talks.talk,`;
          JOIN += ` JOIN talks ON retreatroasters.talkId = talks.id`;
          if (WHERE) {
            WHERE += ` AND retreatroasters.talkId IN (${talkIds.join(",")})`;
          } else {
            WHERE += ` WHERE retreatroasters.talkId IN (${talkIds.join(",")})`;
          }
        }
      }

      let SELECTRETREAT = "";
      if (
        (coupleId !== 0 || primaryKey !== 0 || cityState !== "") &&
        !retreatAttended &&
        !retreatants
      ) {
        SELECTRETREAT += ` SELECT couples.hisLastName, couples.hisFirstName, 
        couples.herFirstName, couples.herLastName,
        couples.primaryKey, couples.hisEmail, couples.herEmail, couples.city, couples.state, couples.zip, couples.notes,couples.address,couples.allergies,couples.anniversary_date,couples.under_age_35,couples.hear_us,couples.referral_from_a_friend_or_family_member,couples.emergency_name1,couples.emergency_name2,couples.emergency_relationship1,couples.emergency_relationship2,couples.emergency_phone1,couples.emergency_phone2,
        couples.hisMobile, couples.herMobile, couples.imageUrl,retreats.heading,
        retreats.location,retreats.title,retreats.language,retreats.id AS reatreatId, retreatroasters.attendeeType,
        retreatroasters.coupleId, services.service, roles.role, talks.talk, couples.accountStatus `;
      } else if (retreatAttended && (coupleId !== 0 || primaryKey !== 0)) {
        SELECTRETREAT += `SELECT couples.hisLastName, couples.hisFirstName,
        couples.herFirstName, couples.herLastName,
        couples.primaryKey, couples.hisEmail, couples.herEmail,couples.city, couples.state, couples.state, couples.zip, couples.notes,couples.address,couples.allergies,couples.anniversary_date,couples.under_age_35,couples.hear_us,couples.referral_from_a_friend_or_family_member,couples.emergency_name1,couples.emergency_name2,couples.emergency_relationship1,couples.emergency_relationship2,couples.emergency_phone1,couples.emergency_phone2, couples.imageUrl,
        couples.hisMobile, couples.herMobile, couples.id as coupleId ,  
        retreats.location,retreats.title,retreats.language, retreats.id AS retreatId, retreats.heading,retreatroasters.attendeeType,
        retreatroasters.coupleId, services.service, roles.role, talks.talk, couples.accountStatus, 
        parish1.parish as coupleParish, parRetreat.parish as parish  `;
      } else if (retreatants && (coupleId !== 0 || primaryKey !== 0)) {
        SELECTRETREAT += `SELECT couples.hisLastName, couples.hisFirstName,couples.primaryKey,
        couples.herFirstName, couples.herLastName,
        couples.primaryKey, couples.hisEmail, couples.herEmail,
        couples.hisMobile, couples.herMobile, couples.id as coupleId, couples.accountStatus, couples.city, couples.state, couples.state, couples.zip, couples.notes,couples.address,couples.allergies,couples.anniversary_date,couples.under_age_35,couples.hear_us,couples.referral_from_a_friend_or_family_member,couples.emergency_name1,couples.emergency_name2,couples.emergency_relationship1,couples.emergency_relationship2,couples.emergency_phone1,couples.emergency_phone2,couples.imageUrl,
        retreats.id AS retreatId,retreats.location, retreatroasters.attendeeType,
        retreats.language, retreats.title,retreats.location,retreats.heading,
        retreats.dateFrom, retreats.dateTo, retreats.createdAt,
        parish1.parish as coupleParish, parRetreat.parish as parish, services.service, roles.role, talks.talk `;
      } else {
        SELECTRETREAT += `SELECT couples.hisLastName, couples.hisFirstName,couples.primaryKey,
        couples.herFirstName, couples.herLastName,
        couples.primaryKey, couples.hisEmail, couples.herEmail,
        couples.hisMobile, couples.herMobile, couples.id as coupleId, couples.accountStatus, couples.city, couples.state, couples.state, couples.zip, couples.notes, couples.address,couples.allergies,couples.anniversary_date,couples.under_age_35,couples.hear_us,couples.referral_from_a_friend_or_family_member,couples.emergency_name1,couples.emergency_name2,couples.emergency_relationship1,couples.emergency_relationship2,couples.emergency_phone1,couples.emergency_phone2,couples.imageUrl,
        retreats.id AS retreatId,retreats.location, retreatroasters.attendeeType,
        retreats.language,retreats.title, retreats.location,retreats.heading,
        retreats.dateFrom, retreats.dateTo, retreats.createdAt,
        parish1.parish as coupleParish, parRetreat.parish as parish,services.service, roles.role, talks.talk `;
      }
      let COUNTRETREAT = "";
      if (retreatAttended) {
        COUNTRETREAT += `, COUNT(retreats.id) AS retreatsCount `;
      } else {
        COUNTRETREAT += `, COUNT(IF(retreatroasters.attendeeType = 'Retreatant',retreatroasters.id,NULL)) AS countRetreatant,
        COUNT(retreatroasters.id) AS roasterTotal, 
        COUNT(IF(retreatroasters.attendeeType = 'Team',retreatroasters.id,NULL)) AS countTeam `;
      }

      let FROMRETREAT;
      if (
        neverServedTeam ||
        neverServiceRole ||
        neverTalk ||
        (is_Not_Head_Role && is_Not_Head_Role !== "")
      ) {
        FROMRETREAT = ` FROM couples 
        LEFT JOIN retreatroasters ON retreatroasters.coupleId = couples.id      
        LEFT JOIN retreats ON retreats.id = retreatroasters.retreatId
        LEFT JOIN services ON retreatroasters.serviceId = services.id
        LEFT JOIN parishes AS parRetreat ON retreats.parishId = parRetreat.id
        LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id
        LEFT JOIN roles ON retreatroasters.roleId = roles.id
        LEFT JOIN talks ON retreatroasters.talkId = talks.id`;
      } else if (Array.isArray(coupleId) && coupleId.length > 0) {
        FROMRETREAT = ` FROM retreatroasters 
        LEFT JOIN retreats ON retreatroasters.retreatId= retreats.id 
    LEFT JOIN parishes AS parRetreat ON retreats.parishId = parRetreat.id
    LEFT JOIN services ON retreatroasters.serviceId = services.id
    LEFT JOIN roles ON retreatroasters.roleId = roles.id
    LEFT JOIN talks ON retreatroasters.talkId = talks.id`;
        if (coupleId !== 0 || primaryKey !== 0 || cityState !== "") {
          FROMRETREAT += ` LEFT JOIN couples ON retreatroasters.coupleId = couples.id LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id`;
        } else if (retreatants) {
          FROMRETREAT += ` LEFT JOIN couples ON retreatroasters.coupleId = couples.id LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id`;
        }
      } else {
        FROMRETREAT = ` FROM retreats 
  LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
  LEFT JOIN parishes AS parRetreat ON retreats.parishId = parRetreat.id
  LEFT JOIN services ON retreatroasters.serviceId = services.id
  LEFT JOIN roles ON retreatroasters.roleId = roles.id
  LEFT JOIN talks ON retreatroasters.talkId = talks.id`;
        if (coupleId !== 0 || primaryKey !== 0 || cityState !== "") {
          FROMRETREAT += ` LEFT JOIN couples ON retreatroasters.coupleId = couples.id LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id`;
        } else if (retreatants) {
          FROMRETREAT += ` LEFT JOIN couples ON retreatroasters.coupleId = couples.id LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id`;
        }
      }
      let GROUPBYRETERAT = "";
      if (
        Array.isArray(coupleId) &&
        coupleId.length > 0 &&
        searchFor == "retreat" &&
        serviceIds.length == 0 &&
        roleIds.length == 0 &&
        talkIds.length == 0 &&
        retreatants.length == 0 &&
        !neverServedTeam &&
        !neverServiceRole &&
        !neverTalk &&
        !missingMails
      ) {
        // console.log("retreat 2");
        GROUPBYRETERAT += ` GROUP BY retreatroasters.coupleId`;
      } else if (
        !coupleId &&
        searchFor == "retreat" &&
        serviceIds.length == 0 &&
        roleIds.length == 0 &&
        talkIds.length == 0 &&
        retreatants.length == 0 &&
        !neverServedTeam &&
        !neverServiceRole &&
        !neverTalk &&
        !missingMails
      ) {
        // console.log("retreat 1");
        GROUPBYRETERAT += ` GROUP BY retreats.id`;
      } else if (
        searchFor == "retreat" &&
        (retreatants.length > 0 ||
          serviceIds.length > 0 ||
          roleIds.length > 0 ||
          talkIds.length > 0 ||
          neverServedTeam ||
          neverServiceRole ||
          neverTalk ||
          missingMails)
      ) {
        // console.log("retreat 2");
        GROUPBYRETERAT += ` GROUP BY retreatroasters.coupleId`;
      }
      // couples
      else if (searchFor == "couple" && retreatAttended) {
        // console.log("couple 3");
        GROUPBYRETERAT += ` GROUP BY couples.id`;
      } else if (
        searchFor == "couple" &&
        Array.isArray(coupleId) &&
        coupleId.length > 0
      ) {
        // console.log("couple 4");
        GROUPBYRETERAT += `  GROUP BY retreatroasters.coupleId`;
      } else if (
        searchFor == "couple" &&
        (retreatId == 0 ||
          retreatId !== "0" ||
          serviceIds.length > 0 ||
          roleIds.length > 0 ||
          talkIds.length > 0)
      ) {
        // console.log("couple 5");
        GROUPBYRETERAT += ` GROUP BY retreatroasters.coupleId`;
      }
      let HAVING = ``;

      //   if (neverTalk) {
      //     HAVING += ` HAVING COUNT(talks.id) = 0`
      //   }

      //    if (neverServedTeam) {
      //     HAVING += ` HAVING COUNT(services.id) = 0`
      //   }

      //  if (neverServiceRole) {
      //     HAVING += ` HAVING COUNT(roles.id) = 0`
      //   }

      if (neverServedTeam) {
        if (HAVING === ``) {
          // console.log("neverServedTeam 1", neverServedTeam);
          HAVING += ` HAVING COUNT(services.id) = 0`;
        } else {
          HAVING += ` AND COUNT(services.id) = 0`;
        }
      }

      if (neverServiceRole) {
        if (HAVING === ``) {
          HAVING += ` HAVING COUNT(roles.id) = 0`;
        } else {
          HAVING += ` AND COUNT(roles.id) = 0`;
        }
      }

      if (neverTalk) {
        if (HAVING === ``) {
          // console.log("efregt", neverTalk);
          HAVING += ` HAVING COUNT(talks.id) = 0`;
        } else {
          HAVING += ` AND COUNT(talks.id) = 0`;
        }
      }

      let queryRetreat =
        SELECTRETREAT +
        COUNTRETREAT +
        FROMRETREAT +
        WHERE +
        GROUPBYRETERAT +
        HAVING;

      // console.log(`${"\n"}queryRetreat retereat: \n--->`, queryRetreat);

      SELECT = SELECT.substring(0, SELECT.length - 1);
      let searchQuery = `${SELECT} from retreatroasters ${JOIN} ${WHERE} ${GROUPBY}`;

      // console.log("searchQuery couple: \n --->", searchQuery);

      let responseData = await db.sequelize
        .query(searchQuery)
        .then(([results]) => results);
      // console.log("searchQuery---------->>", searchQuery);
      // console.log("responseData--->", responseData);

      let responseDataRetreat = await db.sequelize
        .query(queryRetreat)
        .then(([results]) => results);
      // console.log("queryRetreat----->", queryRetreat);
      // console.log("responseDataRetreat---->", responseDataRetreat);

      res.send({
        success: true,
        couples: responseData,
        retreats: responseDataRetreat,
      });
    } catch (error) {
      // console.log("error: ", error);
      return res.status(200).json({
        success: false,
        msg: error.toString(),
      });
    }
  }

  async tempDeleteRetreat(req, res) {
    try {
      const { id } = req.body;
      if (!id) {
        return res.status(400).json({
          success: false,
          msg: "ID is required.",
        });
      }
      const committeeMembers = await committeeMemberSchema.findAll();

      for (const member of committeeMembers) {
        let retreats = [];
        if (typeof member.retreats === "string") {
          try {
            retreats = JSON.parse(member.retreats);
          } catch (err) {
            console.error(
              `Failed to parse retreats for member ID ${member.id}:`,
              err
            );
            continue;
          }
        } else if (Array.isArray(member.retreats)) {
          retreats = member.retreats;
        }
        const updatedRetreats = retreats.filter(
          (r) => r.retreatId !== parseInt(id)
        );
        if (updatedRetreats.length !== retreats.length) {
          await committeeMemberSchema.update(
            { retreats: JSON.stringify(updatedRetreats) },
            { where: { id: member.id } }
          );
        }
      }
      const removeTemporary = await retreatSchema.update(
        { deleteStatus: "temporary" },
        { where: { id: id } }
      );

      if (removeTemporary[0] > 0) {
        return res.status(200).send({
          success: true,
          msg: "Retreat deleted temporarily and removed from committee members.",
        });
      } else {
        return res.status(404).json({
          success: false,
          msg: "Retreat not found.",
        });
      }
    } catch (error) {
      console.error("Error deleting retreat:", error);
      return res.status(500).json({
        success: false,
        msg: "An error occurred while deleting the retreat.",
      });
    }
  }

  async autoSaveRetreat(req, res) {
    try {
      let { selectEditor, id, value, title } = req.body;
      if (!id) {
        return res.status(200).json({
          success: false,
          msg: "Please enter the required fields.",
        });
      }
      let updateData = {};
      if (selectEditor === "mission_statement") {
        let getRetreat = await retreatSchema.findOne({
          where: { id: id },
        });
        try {
          let update = await retreatSchema.update(
            { mission_statement: value },
            {
              where: { language: getRetreat.language },
            }
          );
        } catch (error) {
          console.error("Error updating records:", error);
        }
        updateData.mission_statement = value;
      } else if (selectEditor === "mail_msg") {
        updateData.mail_msg = value;
      } else if (selectEditor === "heading") {
        updateData.heading = value;
      } else if (selectEditor === "location") {
        if (title) {
          let updateTitle = await retreatSchema.update(
            { title: title },
            {
              where: { id: id },
            }
          );
        }
        updateData.location = value;
      } else if (selectEditor === "title") {
        updateData.title = value;
      } else {
        updateData.description = value;
      }
      let update = await retreatSchema.update(updateData, {
        where: { id: id },
      });
      if (update) {
        if (selectEditor === "heading") {
          return res.send({
            success: true,
            msg: `Updated the Retreat Theme.`,
            data: update,
          });
        } else {
          return res.send({
            success: true,
            msg: `Updated the ${selectEditor.replace("_", " ")}.`,
            data: update,
          });
        }
      } else {
        return res.send({
          error: true,
          msg: "Something went wrong, please try again.",
        });
      }
    } catch (error) {
      console.error("Error updating retreat:", error);
      return res.status(500).send({
        error: true,
        msg: "Something went wrong, please try again.",
      });
    }
  }

  async giveAccesssOfApprove(req, res) {
    try {
      const { coupleIds, retreatId } = req.body;
      // Find retreat
      let myRetreat = await retreatSchema.findOne({ where: { id: retreatId } });
      if (!myRetreat) {
        return res.status(404).send({ error: true, msg: "Retreat not found." });
      }

      // Parse accessIds (string or null → array)
      let accessIds = JSON.parse(myRetreat.accessIds || "[]");
      if (Array.isArray(accessIds) && typeof accessIds[0] !== "object") {
        accessIds = accessIds.map((id) => ({
          userId: id,
          isDisabled: false,
        }));
      }

      for (let item of coupleIds) {
        let email = item.label.split(" - ").pop().trim();
        let myUser = await userSchema.findOne({ where: { email } });
        if (myUser) {
          // Check if already exists
          let exists = accessIds.find((u) => u.userId === myUser.id);
          if (!exists) {
            accessIds.push({ userId: myUser.id, isDisabled: false });

            // Save retreat
            await retreatSchema.update(
              { accessIds: JSON.stringify(accessIds) },
              { where: { id: retreatId } }
            );

            // Send email
            let replData = RETREATACCESS.replace(
              /#userName#/g,
              myUser.userName || ""
            )
              .replace(/#retreatTitle#/g, myRetreat.title || "")
              .replace(
                /#link#/g,
                `${process.env.SERVERADDRESS}/registration-retreat/${retreatId}`
              )
              .replace(/#loginLink#/g, `${process.env.SERVERADDRESS}/login`)
              .replace(/#email#/g, myUser.email)
              .replace(/#password#/g, myUser.password);
            Email.sendEmail(
              myUser.email,
              "Access to Manage Retreat Registrations",
              replData
            );
          }
        } else {
          let randompass1 = Math.random().toString(36).slice(-8);
          let paswordForCouple1 = `p${randompass1.slice(1)}`;

          let findCouple = await coupleSchema.findOne({
            where: { id: item.value },
          });
          let adminObj = {
            userName: email,
            email,
            password: paswordForCouple1,
            accountStatus: "ACTIVE",
            userRole: "COUPLE",
          };
          if (findCouple) {
            adminObj.coupleId = item.value;
            adminObj.userName = findCouple.hisFirstName;
            adminObj.firstName = findCouple.herFirstName || "";
            adminObj.lastName = findCouple.herLastName || "";
            adminObj.mobile = findCouple.herMobile || "";
            adminObj.imageUpload = findCouple.imageUrl;
          }
          let newCouple = await userSchema.create(adminObj);

          accessIds.push({ userId: newCouple.id, isDisabled: false });

          await retreatSchema.update(
            { accessIds: JSON.stringify(accessIds) },
            { where: { id: retreatId } }
          );
          let replaceString = item.isNew
            ? ","
            : newCouple.userName
            ? " " + newCouple.userName + ","
            : ","; // if new user then empty username for mail

          let replData = RETREATACCESS.replace(/#userName#/g, replaceString)
            .replace(/#retreatTitle#/g, myRetreat.title || "")
            .replace(
              /#link#/g,
              `${process.env.SERVERADDRESS}/registration-retreat/${retreatId}`
            )
            .replace(/#loginLink#/g, `${process.env.SERVERADDRESS}/login`)
            .replace(/#email#/g, newCouple.email)
            .replace(/#password#/g, newCouple.password);

          Email.sendEmail(
            newCouple.email,
            "Access to Manage Retreat Registrations",
            replData
          );
        }
      }
      return res.send({
        success: true,
        msg: "New Retreat Sub-Admin(s) added for access.",
        data: accessIds,
      });
    } catch (error) {
      console.error("Error updating retreat:", error);
      return res
        .status(500)
        .send({ error: true, msg: "Something went wrong, please try again." });
    }
  }

  async removeAccessFromRetreat(req, res) {
    try {
      let { remove_access_mail, retreatId } = req.body;
      let email = remove_access_mail.split(" - ").pop().trim();
      let user = await userSchema.findOne({
        where: { email: email },
      });
      if (!user) {
        return res.status(200).send({
          error: true,
          msg: "User not found.",
        });
      }
      let myRetreat = await retreatSchema.findOne({ where: { id: retreatId } });
      if (!myRetreat) {
        return res.status(200).send({
          error: true,
          msg: "Retreat not found.",
        });
      }
      let accessIds;
      try {
        accessIds = JSON.parse(myRetreat.accessIds || "[]");
      } catch (e) {
        return res.status(200).send({
          error: true,
          msg: "Invalid access IDs format.",
        });
      }
      accessIds = accessIds.filter((obj) => obj.userId !== user.id);
      let update = await retreatSchema.update(
        { accessIds: JSON.stringify(accessIds) },
        { where: { id: retreatId } }
      );
      if (update[0] > 0) {
        await userSchema.update(
          { deleteAccess: true },
          { where: { id: user.id } }
        );
        return res.send({
          success: true,
          msg: `This mail has been removed from access.`,
        });
      } else {
        return res.send({
          error: true,
          msg: "Something went wrong, please try again.",
        });
      }
    } catch (error) {
      console.error("Error updating retreat:", error);
      return res.status(200).send({
        error: true,
        msg: "Something went wrong, please try again.",
      });
    }
  }

  async getAllAccessedRetreats(req, res) {
    try {
      let { userId } = req.body;

      let query = `
      SELECT 
        retreats.*, 
        programs.status AS programStatus, 
        directories.status AS directoryStatus 
      FROM 
        retreats
      LEFT JOIN 
        programs ON programs.retreatId = retreats.id
      LEFT JOIN 
        directories ON directories.retreatId = retreats.id
      WHERE 
        (retreats.deleteStatus='' OR retreats.deleteStatus IS NULL) 
        -- ✅ userId must exist in accessIds
        AND JSON_CONTAINS(retreats.accessIds, '{"userId": ${userId}}', '$')
        -- ✅ but user must NOT be disabled
        AND NOT JSON_CONTAINS(retreats.accessIds, '{"userId": ${userId}, "isDisabled": true}', '$')
      ORDER BY 
        retreats.dateFrom DESC;
    `;

      let allRetreats = await db.sequelize
        .query(query)
        .then(([results]) => results);

      const parsedData = allRetreats.map((item) => {
        const { confirmation, schedule, ...rest } = item;
        return {
          ...rest,
          confirmation: confirmation ? JSON.parse(confirmation) : [],
          schedule: schedule
            ? JSON.parse(schedule).map((entry) => ({
                date: new Date(entry.date),
                from: entry.from,
                to: entry.to,
              }))
            : [],
        };
      });

      return res.status(200).send({
        success: true,
        msg: "All Incoming Retreats",
        data: parsedData,
      });
    } catch (error) {
      console.error("Error fetching retreats:", error);
      return res.status(500).json({
        success: false,
        msg: "Error fetching new retreats",
        error: error.message,
      });
    }
  }

  async allretreatSubAdmins(req, res) {
    try {
      const getretreats = await retreatSchema.findAll({ raw: true });
      const result = [];

      for (const retreat of getretreats) {
        if (retreat.accessIds && retreat.accessIds.length > 0) {
          let accessIdsArray;

          try {
            accessIdsArray = JSON.parse(retreat.accessIds);
          } catch (e) {
            console.error("Invalid accessIds JSON:", retreat.accessIds);
            continue; // skip this retreat if data is invalid
          }

          if (Array.isArray(accessIdsArray) && accessIdsArray.length > 0) {
            const retreatData = {
              title: retreat.title,
              id: retreat.id,
              subAdmins: [],
            };

            // Collect userIds
            const userIds = accessIdsArray.map((obj) => obj.userId);

            // Fetch users by IDs
            const subAdmins = await userSchema.findAll({
              where: { id: userIds },
              attributes: [
                "id",
                "userName",
                "firstName",
                "lastName",
                "email",
                "password",
              ],
            });

            // Merge user info with accessIds (to get isDisabled)
            subAdmins.forEach((admin) => {
              const accessInfo = accessIdsArray.find(
                (obj) => obj.userId === admin.id
              );
              retreatData.subAdmins.push({
                name: `${admin.firstName || ""} ${admin.lastName || ""} (${
                  admin.email
                })`,
                email: admin.email,
                id: admin.id,
                isDisabled: accessInfo ? accessInfo.isDisabled : false,
                password: admin.password,
              });
            });

            if (retreatData.subAdmins.length > 0) {
              result.push(retreatData);
            }
          }
        }
      }

      return res.status(200).send({
        success: true,
        msg: "Sub-Admins Retrieved Successfully",
        data: result,
      });
    } catch (err) {
      console.error("Error retrieving sub-admins:", err);
      return res.status(500).json({
        success: false,
        msg: "An error occurred while retrieving the sub-admins.",
      });
    }
  }

  async retreatAdminAccess(req, res) {
    try {
      const { userId, retreatId, checkAccess } = req.body;
      let retreatData = await retreatSchema.findOne({
        where: { id: retreatId },
      });
      if (!retreatData) {
        return res.status(200).send({
          success: false,
          msg: "Retreat not found",
        });
      }
      let accessIds = [];
      try {
        accessIds = retreatData.accessIds
          ? JSON.parse(retreatData.accessIds)
          : [];
      } catch (e) {
        return res.status(200).send({
          success: false,
          msg: "Invalid accessIds format",
        });
      }
      let userAccess = accessIds.find((obj) => obj.userId === userId);
      if (userAccess) {
        userAccess.isDisabled = !checkAccess;
      } else {
        accessIds.push({ userId, isDisabled: !checkAccess });
      }
      await retreatSchema.update(
        { accessIds: JSON.stringify(accessIds) },
        { where: { id: retreatId } }
      );
      return res.status(200).send({
        success: true,
        msg: "Access updated successfully",
      });
    } catch (err) {
      console.error("Error updating retreat access:", err);
      return res.status(200).json({
        success: false,
        msg: "An error occurred while updating access.",
      });
    }
  }

  async getAllTempRetreats(req, res) {
    try {
      let { searchKeyword, sortBy, pageNo, size } = req.body;

      let query = { skip: 0, limit: 0 };
      let sort1 = "DESC";
      let sortColumn = "tempRetreats.createdAt";
      let searchColumn = "";

      // Pagination setup
      if (pageNo || size) {
        pageNo = parseInt(pageNo);
        size = parseInt(size);
        if (pageNo < 1) {
          return res.status(200).send({
            error: true,
            msg: "Invalid page number, should start with 1",
          });
        }
        query.skip = size * (pageNo - 1);
        query.limit = size;
      }

      // Sorting setup
      if (sortBy && sortBy.sortDirection == 1) {
        sort1 = "ASC";
      }
      if (sortBy && sortBy.sortColumn) {
        if (sortBy.sortColumn === "language") {
          sortColumn = "tempRetreats.language";
        } else if (sortBy.sortColumn === "location") {
          sortColumn = "tempRetreats.location";
        } else if (sortBy.sortColumn === "title") {
          sortColumn = "tempRetreats.title";
        } else if (sortBy.sortColumn === "parish") {
          sortColumn = "parishes.parish";
        } else if (sortBy.sortColumn === "contact_name") {
          sortColumn = "tempRetreats.contact_name";
        } else if (sortBy.sortColumn === "contact_email") {
          sortColumn = "tempRetreats.contact_email";
        } else if (sortBy.sortColumn === "contact_mobile") {
          sortColumn = "tempRetreats.contact_mobile";
        }
      }

      // Search filters
      if (searchKeyword && searchKeyword.trim() !== "") {
        const keyword = `%${searchKeyword.trim()}%`;
        searchColumn = `
        tempRetreats.title LIKE "${keyword}" OR
        tempRetreats.language LIKE "${keyword}" OR
        tempRetreats.location LIKE "${keyword}" OR
        parishes.parish LIKE "${keyword}" OR
        tempRetreats.contact_name LIKE "${keyword}" OR
        tempRetreats.contact_email LIKE "${keyword}" OR
        tempRetreats.contact_mobile LIKE "${keyword}"
      `;
      }

      // Main query
      let select = `
      SELECT tempRetreats.*, parishes.parish 
      FROM tempRetreats 
      LEFT JOIN parishes ON parishes.id = tempRetreats.parishId
    `;
      let whereClause = searchColumn ? `WHERE ${searchColumn}` : "";
      let groupBy = `
      GROUP BY tempRetreats.id
      ORDER BY ${sortColumn} ${sort1}
      LIMIT ${query.limit}
      OFFSET ${query.skip}
    `;
      let queryRecords = select + " " + whereClause + " " + groupBy;

      // Count query (with JOIN added!)
      let retretTotSelect = `
      SELECT COUNT(DISTINCT tempRetreats.id) as totalRetreat 
      FROM tempRetreats 
      LEFT JOIN parishes ON parishes.id = tempRetreats.parishId
    `;
      let retreatTotalQuery = retretTotSelect + " " + whereClause;

      // Execute queries
      let data = await db.sequelize
        .query(queryRecords)
        .then(([results]) => results);
      let totalResult = await db.sequelize
        .query(retreatTotalQuery)
        .then(([results]) => results);

      return res.status(200).send({
        success: true,
        data,
        totalRecords: totalResult[0]?.totalRetreat || 0,
      });
    } catch (error) {
      console.error("error----->", error);
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async deleteTempRetreat(req, res) {
    try {
      if (req.body.id) {
        let replData = APPROVEREJECTRETREAT.replace(
          /#contactName#/g,
          req.body.contact_name || ""
        )
          .replace(/#retreatTitle#/g, req.body.title || "")
          .replace(/#result#/g, "rejected");
        Email.sendEmail(req.body.contact_email, "Retreat Rejected", replData);

        let deleteRoaster = await tempRetreatSchema.destroy({
          where: { id: req.body.id },
        });
        return res.status(200).send({
          success: true,
          msg: "Retreat deleted successfully.",
        });
      } else {
        return res.status(200).json({
          success: false,
          msg: "Error in id.",
        });
      }
    } catch (error) {
      console.log("error:", error);
      return res.status(200).json({
        success: false,
        msg: error,
      });
    }
  }

  async approveRetreat(req, res) {
    try {
      // Destructure and parse data from request body
      const {
        id,
        parishId,
        language,
        location,
        schedule,
        heading,
        mission_statement,
        description,
        mail_msg,
        contact_name,
        contact_email,
        contact_mobile,
        image1,
        retreat_team,
        subAdmins,
      } = JSON.parse(req.body.fromData);
      // Check if schedule has valid dates
      let retreat_team1 = retreat_team.filter(
        (item) => item && (item.hisEmail || item.herEmail)
      );
      console.log("retreat_team1---->", retreat_team1);

      let schedule1 = schedule.filter((item) => item && item.date);
      if (schedule1 && schedule1.length > 0) {
        const dateFrom = new Date(schedule1[0].date);
        const dateTo = new Date(schedule1[schedule1.length - 1].date);
        let parishCheck = await perishSchema.findOne({
          where: { id: parishId },
        });
        const saveObj = {
          title: `${parishCheck.parish} (${
            language.charAt(0).toUpperCase() + language.slice(1).toLowerCase()
          }) ${location}`,
          language,
          location,
          schedule: JSON.stringify(schedule1),
          heading,
          mission_statement,
          description,
          dateFrom,
          dateTo,
          mail_msg,
          contact_name,
          contact_email,
          contact_mobile,
          parishId,
        };
        const newRetreat = new retreatSchema(saveObj);
        const succ = await newRetreat.save();
        let accessIds = JSON.parse(succ.accessIds || "[]");
        for (let item of subAdmins) {
          let email = item.label.split(" - ").pop().trim();
          let myUser = await userSchema.findOne({ where: { email } });
          if (myUser) {
            if (myUser.email && !accessIds.includes(myUser.id)) {
              accessIds.push(myUser.id);
              const newIds = JSON.stringify(accessIds);
              await retreatSchema.update(
                { accessIds: newIds },
                { where: { id: succ.id } }
              );
              let replData = RETREATACCESS.replace(
                /#userName#/g,
                myUser.userName || ""
              )
                .replace(/#retreatTitle#/g, succ.title || "")
                .replace(
                  /#link#/g,
                  `${process.env.SERVERADDRESS}/registration-retreat/${succ.id}`
                )
                .replace(/#loginLink#/g, `${process.env.SERVERADDRESS}/login`)
                .replace(/#email#/g, myUser.email)
                .replace(/#password#/g, myUser.password);

              Email.sendEmail(
                myUser.email,
                "Access to Manage Retreat Registrations",
                replData
              );
            }
          } else {
            let randompass1 = Math.random().toString(36).slice(-8);
            let paswordForCouple1 = `p${randompass1.slice(1)}`;
            let findCouple = await coupleSchema.findOne({
              where: { id: item.value },
            });
            let namePart = item.label.split(" - ")[0].trim();
            let [lastName, firstName] = namePart
              .split(",")
              .map((s) => s.trim());

            let adminObj = {
              userName: `${firstName} ${lastName}`.trim(),
              firstName: firstName || "",
              lastName: lastName || "",
              email: email,
              password: paswordForCouple1,
              accountStatus: "ACTIVE",
              userRole: "COUPLE",
            };
            if (findCouple) {
              adminObj.coupleId = item.value;
              adminObj.userName = findCouple.hisFirstName || adminObj.userName;
              adminObj.firstName = findCouple.herFirstName || "";
              adminObj.lastName = findCouple.herLastName || "";
              adminObj.mobile = findCouple.herMobile || "";
              adminObj.imageUpload = findCouple.imageUrl;
            }

            let newCouple = await userSchema.create(adminObj);

            accessIds.push(newCouple.id);
            const newIds = JSON.stringify(accessIds);
            await retreatSchema.update(
              { accessIds: newIds },
              { where: { id: succ.id } }
            );

            let replData = RETREATACCESS.replace(
              /#userName#/g,
              newCouple.userName || ""
            )
              .replace(/#retreatTitle#/g, succ.title || "")
              .replace(
                /#link#/g,
                `${process.env.SERVERADDRESS}/registration-retreat/${succ.id}`
              )
              .replace(/#loginLink#/g, `${process.env.SERVERADDRESS}/login`)
              .replace(/#email#/g, newCouple.email)
              .replace(/#password#/g, newCouple.password);

            Email.sendEmail(
              newCouple.email,
              "Access to Manage Retreat Registrations",
              replData
            );
          }
        }

        await this.handleRetreatTeam(retreat_team1, succ.id);
        // Handle file uploads (if any)
        let imageSavedRetreat;
        if (req.files) {
          imageSavedRetreat = await this.saveImagesForRetreat(
            req.files,
            succ.id
          );
        } else {
          if (image1 === "" || image1 === null) {
            await retreatSchema.update(
              { image1: "" },
              { where: { id: succ.id } }
            );
          } else {
            await retreatSchema.update(
              { image1: image1 },
              { where: { id: succ.id } }
            );
          }
        }
        // Update retreat with mission statement
        try {
          const update = await retreatSchema.update(
            { mission_statement },
            { where: { language } }
          );
        } catch (updateError) {
          console.error("Error updating retreat:", updateError);
          return res.status(500).json({
            success: false,
            msg: "Error updating retreat mission statement.",
          });
        }
        if (contact_email) {
          let replData = APPROVEREJECTRETREAT.replace(
            /#contactName#/g,
            contact_name || ""
          )
            .replace(/#retreatTitle#/g, succ.title || "")
            .replace(/#result#/g, "approved");
          Email.sendEmail(contact_email, "Retreat Approved", replData);
        }
        // Remove the temporary retreat entry
        try {
          await tempRetreatSchema.destroy({ where: { id: id } });
        } catch (deleteError) {
          console.error("Error deleting temporary retreat:", deleteError);
          return res.status(500).json({
            success: false,
            msg: "Error deleting temporary retreat entry.",
          });
        }
        // Respond with success if everything was successful
        return res.status(200).send({
          success: true,
          msg: "Retreat saved successfully:",
          data: succ,
        });
      } else {
        // Handle case where schedule is empty or not provided
        return res.status(400).json({
          success: false,
          msg: "At least one date is required in the schedule.",
        });
      }
    } catch (error) {
      // Handle unexpected errors
      console.log(
        `approveRetreat [${new Date().toLocaleString("en-US", {
          timeZoneName: "short",
        })}] error -->`,
        error
      );
      return res.status(500).json({
        success: false,
        msg: "An unexpected error occurred.",
        error: error.message,
      });
    }
  }

  async handleRetreatTeam(retreatTeam, retreatId) {
    for (const item of retreatTeam) {
      switch (item.role) {
        case "lead_couple":
          await this.processCouple(item, "lead_couple", "Lead", retreatId);
          break;
        case "co_lead_couple":
          await this.processCouple(
            item,
            "co_lead_couple",
            "Co-Lead",
            retreatId
          );
          break;
        case "mentor_couple":
          await this.processCouple(item, "mentor_couple", "Mentor", retreatId);
          break;
        // Add other roles here as needed
      }
    }
  }

  async processCouple(item, role, roleName, retreatId) {
    const roleRecord = await RolesSchema.findOne({ where: { role: roleName } });
    let coupleCheck = await coupleSchema.findOne({
      where: {
        [Op.or]: [
          {
            [Op.and]: [
              { hisEmail: { [Op.ne]: null } },
              { hisEmail: { [Op.ne]: "" } },
              {
                [Op.or]: [
                  { hisEmail: item.hisEmail },
                  { hisEmail: item.herEmail },
                ],
              },
            ],
          },
          {
            [Op.and]: [
              { herEmail: { [Op.ne]: null } },
              { herEmail: { [Op.ne]: "" } },
              {
                [Op.or]: [
                  { herEmail: item.hisEmail },
                  { herEmail: item.herEmail },
                ],
              },
            ],
          },
        ],
      },
    });
    console.log("coupleCheck---->", coupleCheck);

    if (coupleCheck) {
      // console.log("111111111111111");
      await retreatRoastSchema.create({
        coupleId: coupleCheck.id,
        roleId: roleRecord.id,
        attendeeType: "Team",
        retreatId,
      });
    } else {
      // console.log("2222222222222222");
      let maxPrimaryKeyDB = await coupleSchema.max("primaryKey");
      if (maxPrimaryKeyDB == null) {
        maxPrimaryKeyDB = 1000;
      }
      let coupleCreateCheck = await coupleSchema.create({
        primaryKey: maxPrimaryKeyDB + 1,
        hisFirstName: item.hisFirstName,
        hisLastName: item.hisLastName,
        hisEmail:
          item.hisEmail && item.hisEmail.trim() !== "" ? item.hisEmail : null,
        herFirstName: item.herFirstName,
        herLastName: item.herLastName,
        herEmail:
          item.herEmail && item.herEmail.trim() !== "" ? item.herEmail : null,
      });
      let paswordForCouple1;
      let paswordForCouple2;
      if (
        coupleCreateCheck.hisEmail &&
        coupleCreateCheck.hisEmail !== "" &&
        coupleCreateCheck.hisEmail !== undefined
      ) {
        let randompass1 = Math.random().toString(36).slice(-8);
        paswordForCouple1 =
          randompass1.slice(0, 0) + "p" + randompass1.slice(1);
      }
      if (
        item.herEmail &&
        item.herEmail !== "" &&
        item.herEmail !== undefined
      ) {
        let randompass2 = Math.random().toString(36).slice(-8);
        paswordForCouple2 =
          randompass2.slice(0, 0) + "p" + randompass2.slice(1);
      }
      let adminObj = [
        {
          userName: item.hisFirstName,
          email: item.hisEmail,
          password: paswordForCouple1 ? paswordForCouple1 : null,
          accountStatus: "ACTIVE",
          userRole: "COUPLE",
          coupleId: coupleCreateCheck.id,
          firstName: item.hisFirstName ? item.hisFirstName : "",
          lastName: item.hisLastName ? item.hisLastName : "",
        },
        {
          userName: item.herFirstName,
          email: item.herEmail,
          password: paswordForCouple2 ? paswordForCouple2 : null,
          accountStatus: "ACTIVE",
          userRole: "COUPLE",
          coupleId: coupleCreateCheck.id,
          firstName: item.herFirstName ? item.herFirstName : "",
          lastName: item.herLastName ? item.herLastName : "",
        },
      ];
      if (
        item &&
        item.hisEmail &&
        item.hisEmail !== "" &&
        item.hisEmail !== undefined
      ) {
        let newUser = new userSchema(adminObj[0]);
        let succUser1 = await newUser.save();
        let replData = SENDLOGINDETAILBYADMIN.replace(
          /#firstName#/g,
          item.hisFirstName ? item.hisFirstName : ""
        )
          .replace(/#lastName#/g, item.hisLastName ? item.hisLastName : "")
          .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
          .replace(/#email#/g, item.hisEmail)
          .replace(/#password#/g, paswordForCouple1);
        Email.sendEmail(
          item.hisEmail,
          "Couple Detail for Login- LOVESTRONG Marriage",
          replData
        );
      }
      if (
        item &&
        item.herEmail &&
        item.herEmail !== "" &&
        item.herEmail !== undefined
      ) {
        let newUser = new userSchema(adminObj[1]);
        let succUser2 = await newUser.save();
        let replData = SENDLOGINDETAILBYADMIN.replace(
          /#firstName#/g,
          item.herFirstName ? item.herFirstName : ""
        )
          .replace(/#lastName#/g, item.herLastName ? item.herLastName : "")
          .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)

          .replace(/#email#/g, item.herEmail)
          .replace(/#password#/g, paswordForCouple2);
        Email.sendEmail(
          item.herEmail,
          "Couple Detail for Login- LOVESTRONG Marriage",
          replData
        );
      }
      if (roleRecord) {
        await retreatRoastSchema.create({
          coupleId: coupleCreateCheck.id,
          roleId: roleRecord.id,
          attendeeType: "Team",
          retreatId,
        });
      }
    }
  }

  async uploadFiles(req, res) {
    const retreatId = req.body.retreatId;
    const userData = JSON.parse(req.body.userData);
    if (!retreatId) {
      return res
        .status(200)
        .json({ success: false, message: "Retreat ID is required" });
    }

    if (!req.files || (!req.files.programFile && !req.files.directoryFile)) {
      return res.status(200).json({
        success: false,
        message: "At least one file (programFile or directoryFile) is required",
      });
    }

    let file,
      fileName,
      fileUrl,
      imgUrl,
      item = "";

    if (req.files.programFile) {
      file = req.files.programFile;
      fileName = file.name;
      fileUrl = `${PROJECT_DIR}/programImages/${fileName}`;
      imgUrl = `${process.env.SERVERADDRESS}/public/programImages/${fileName}`;
      item = "Program";
    } else if (req.files.directoryFile) {
      file = req.files.directoryFile;
      fileName = file.name;
      fileUrl = `${PROJECT_DIR}/directoryImages/${fileName}`;
      imgUrl = `${process.env.SERVERADDRESS}/public/directoryImages/${fileName}`;
      item = "Directory";
    }

    try {
      await this.moveFile(file, fileUrl);
      const updateData = {};
      if (req.files.programFile) updateData.programFile = imgUrl;
      if (req.files.directoryFile) updateData.directoryFile = imgUrl;

      await retreatSchema.update(updateData, { where: { id: retreatId } });
      const getRetreat = await retreatSchema.findOne({
        where: { id: retreatId },
        raw: true,
      });

      let confirmation =
        Array.isArray(getRetreat.confirmation) && getRetreat.confirmation.length
          ? getRetreat.confirmation
          : typeof getRetreat.confirmation === "string"
          ? JSON.parse(getRetreat?.confirmation)
          : [
              {
                roasterData: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Directory: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Program: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                pictures: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                retreatStatus: "ACTIVE",
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
            ];

      const now = new Date();
      const padZero = (num) => (num < 10 ? `0${num}` : num);
      const myDate = `${now.getFullYear()}-${padZero(
        now.getMonth() + 1
      )}-${padZero(now.getDate())} ${padZero(now.getHours())}:${padZero(
        now.getMinutes()
      )}:${padZero(now.getSeconds())}`;

      const updatedConfirmation = confirmation.map((entry, index) => {
        if (
          (item === "Directory" && index === 1) ||
          (item === "Program" && index === 2)
        ) {
          return {
            ...entry,
            [item]: true,
            confirmedByName: `${userData.firstName} ${userData.lastName}`,
            confirmedByRole:
              userData.access === true
                ? userData.userRole === "ADMIN"
                  ? userData.userRole
                  : "RETREAT SUB-ADMIN"
                : userData.userRole,
            confirmedByEmail: userData.email,
            ConfirmedDate: myDate,
          };
        }
        return entry;
      });

      const allRequiredConfirmed =
        updatedConfirmation.some((e) => e.roasterData === true) &&
        updatedConfirmation.some((e) => e.Directory === true) &&
        updatedConfirmation.some((e) => e.Program === true) &&
        updatedConfirmation.some((e) => e.pictures === true);

      const updatePayload = {
        confirmation: JSON.stringify(updatedConfirmation),
      };

      if (allRequiredConfirmed) {
        updatePayload.retreatCloseStatus = 1;
      }
      await retreatSchema.update(updatePayload, { where: { id: retreatId } });
      const updatedRetreat = await retreatSchema.findOne({
        where: { id: retreatId },
        raw: true,
      });
      if (updatedRetreat.schedule) {
        try {
          updatedRetreat.schedule = JSON.parse(updatedRetreat.schedule);
        } catch (err) {
          console.error("Error parsing schedule:", err);
        }
      }
      if (updatedRetreat.confirmation) {
        try {
          updatedRetreat.confirmation = JSON.parse(updatedRetreat.confirmation);
        } catch (err) {
          console.error("Error parsing confirmation:", err);
        }
      }
      if (allRequiredConfirmed) {
        const treasurerEmails = [
          // { name: "Irene Mitschke", email: "manpreet02701@gmail.com" },
          // { name: "Didi Kolkebeck", email: "kolkebeckd@gmail.com" },
          // { name: "Tom Kolkebeck", email: "tkolkebeck@gmail.com" },
          { name: "Jimmie Flores", email: "jimmieflores@gmail.com" },
        ];
        for (const person of treasurerEmails) {
          const replData = MAILTOTREASUREPEOPLE.replace(
            /#name#/g,
            person.name || ""
          )
            .replace(/#retreat#/g, updatedRetreat?.title || "")
            .replace(
              /#date#/g,
              moment(updatedRetreat.dateFrom).isSame(
                updatedRetreat.dateTo,
                "day"
              )
                ? `(${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("MMM-DD-YYYY")})`
                : `(${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("MMM DD")} - ${moment(updatedRetreat.dateTo)
                    .tz("America/New_York")
                    .format("MMM DD")}, ${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("YYYY")})` || ""
            );

          await Email.sendEmail(
            person.email,
            "Retreat Validated and Closed",
            replData
          );
        }
      }

      return res.json({
        success: true,
        data: updatedRetreat,
        message: allRequiredConfirmed
          ? "✅ File uploaded → All confirmations completed → Retreat auto-closed & email sent"
          : "✅ File uploaded & status updated successfully",
      });
    } catch (error) {
      console.error("❌ File upload error:", error);
      return res.status(200).json({
        success: false,
        message: "File upload failed",
        error: error.message,
      });
    }
  }

  // Helper function to move file using promise
  async moveFile(file, filePath) {
    return new Promise((resolve, reject) => {
      file.mv(filePath, (err) => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

  async deleteFile(req, res) {
    try {
      const { retreatId, fileType, userData } = req.body;
      if (!retreatId || !fileType) {
        return res
          .status(400)
          .json({ message: "Missing required parameters." });
      }

      // Fetch current retreat to get file URL before updating
      const retreat = await retreatSchema.findOne({
        where: { id: retreatId },
        raw: true,
      });

      if (!retreat) {
        return res.status(404).json({ message: "Retreat not found." });
      }

      // Build update payload
      const updateData = {};
      updateData[fileType] = null;

      // Delete actual file from disk if it exists
      const fileUrl = retreat[fileType];
      if (fileUrl) {
        const folder =
          fileType === "programFile" ? "programImages" : "directoryImages";
        const filePath = path.join(PROJECT_DIR, folder, path.basename(fileUrl));
        fs.unlink(filePath, (err) => {
          if (err) {
            console.warn("File not found or already deleted:", filePath);
          } else {
            console.log("Deleted file:", filePath);
          }
        });
      }

      // Update DB to remove file reference
      await retreatSchema.update(updateData, { where: { id: retreatId } });

      const item =
        fileType === "programFile"
          ? "Program"
          : fileType === "directoryFile"
          ? "Directory"
          : "";

      if (item) {
        // Reset confirmation for this file
        let confirmation =
          Array.isArray(retreat.confirmation) && retreat.confirmation.length
            ? retreat.confirmation
            : typeof retreat.confirmation === "string"
            ? JSON.parse(retreat.confirmation)
            : [
                {
                  roasterData: false,
                  confirmedByName: "",
                  confirmedByRole: "",
                  confirmedByEmail: "",
                  ConfirmedDate: "",
                },
                {
                  Directory: false,
                  confirmedByName: "",
                  confirmedByRole: "",
                  confirmedByEmail: "",
                  ConfirmedDate: "",
                },
                {
                  Program: false,
                  confirmedByName: "",
                  confirmedByRole: "",
                  confirmedByEmail: "",
                  ConfirmedDate: "",
                },
                {
                  pictures: false,
                  confirmedByName: "",
                  confirmedByRole: "",
                  confirmedByEmail: "",
                  ConfirmedDate: "",
                },
                {
                  retreatStatus: "ACTIVE",
                  confirmedByName: "",
                  confirmedByRole: "",
                  confirmedByEmail: "",
                  ConfirmedDate: "",
                },
              ];

        const now = new Date();
        const padZero = (num) => (num < 10 ? `0${num}` : num);
        const myDate = `${now.getFullYear()}-${padZero(
          now.getMonth() + 1
        )}-${padZero(now.getDate())} ${padZero(now.getHours())}:${padZero(
          now.getMinutes()
        )}:${padZero(now.getSeconds())}`;

        const updatedConfirmation = confirmation.map((entry, index) => {
          if (
            (item === "Directory" && index === 1) ||
            (item === "Program" && index === 2)
          ) {
            return {
              ...entry,
              [item]: false,
              confirmedByName: `${userData.firstName} ${userData.lastName}`,
              confirmedByRole:
                userData.access === true
                  ? userData.userRole === "ADMIN"
                    ? userData.userRole
                    : "RETREAT SUB-ADMIN"
                  : userData.userRole,
              confirmedByEmail: userData.email,
              ConfirmedDate: myDate,
            };
          }
          return entry;
        });

        const shouldReopen =
          retreat.retreatCloseStatus === 1 ? 0 : retreat.retreatCloseStatus;

        await retreatSchema.update(
          {
            confirmation: JSON.stringify(updatedConfirmation),
            retreatCloseStatus: shouldReopen,
          },
          { where: { id: retreatId } }
        );
      }

      res.json({
        success: true,
        message:
          "✅ File deleted from disk & DB, confirmation reset, retreat reopened if necessary.",
      });
    } catch (error) {
      console.error("Error in deleteFile:", error);
      res.status(500).json({
        message: "Failed to delete the file.",
        error: error.message,
      });
    }
  }

  async retreatFreeze(req, res) {
    try {
      const { retreatId, value } = req.body;
      if (!retreatId || !["FREEZE", "UNFREEZE"].includes(value)) {
        return res.status(400).json({
          success: false,
          msg: "Invalid retreatId or value.",
        });
      }
      await retreatSchema.update(
        { freeze: value === "FREEZE" ? "UNFREEZE" : "FREEZE" },
        {
          where: { id: retreatId },
        }
      );
      return res.status(200).send({
        success: true,
        msg: `Retreat ${
          value === "FREEZE" ? "unfreeze" : "FREEZE"
        } successfully.`,
      });
    } catch (error) {
      console.error("Error toggling retreat freeze status:", error);
      return res.status(500).json({
        success: false,
        msg: "An error occurred while toggling the retreat freeze status.",
      });
    }
  }

  async handleConfirmation(req, res) {
    try {
      if (!Array.isArray(req.body) || req.body.length < 1) {
        return res
          .status(200)
          .json({ success: false, message: "Invalid request data format." });
      }

      const { retreatData, item, userData } = req.body[0];

      if (!retreatData || !retreatData.id) {
        return res
          .status(200)
          .json({ success: false, message: "Retreat data or ID is missing." });
      }

      if (
        !userData ||
        !userData.firstName ||
        !userData.lastName ||
        !userData.email
      ) {
        return res.status(200).json({
          success: false,
          message: "Invalid or incomplete user data.",
        });
      }

      if (!["Directory", "Program", "roasterData", "pictures"].includes(item)) {
        return res
          .status(200)
          .json({ success: false, message: "Invalid item type provided." });
      }
      // Parse and handle `confirmation` safely
      const confirmation =
        Array.isArray(retreatData.confirmation) &&
        retreatData.confirmation.length
          ? retreatData.confirmation
          : typeof retreatData.confirmation === "string"
          ? JSON.parse(retreatData.confirmation)
          : [
              {
                roasterData: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Directory: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                Program: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                pictures: false,
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
              {
                retreatStatus: "ACTIVE",
                confirmedByName: "",
                confirmedByRole: "",
                confirmedByEmail: "",
                ConfirmedDate: "",
              },
            ];
      const now = new Date();
      const padZero = (num) => (num < 10 ? `0${num}` : num);
      let myDate = `${now.getFullYear()}-${padZero(
        now.getMonth() + 1
      )}-${padZero(now.getDate())} ${padZero(now.getHours())}:${padZero(
        now.getMinutes()
      )}:${padZero(now.getSeconds())}`;

      // Helper function to update confirmation data
      const getUpdatedConfirmation = (confirmation, item, userData) => {
        return confirmation.map((entry, index) => {
          if (
            (item === "roasterData" && index === 0) ||
            (item === "Directory" && index === 1) ||
            (item === "Program" && index === 2) ||
            (item === "pictures" && index === 3)
          ) {
            return {
              ...entry,
              [item]: !entry[item],
              confirmedByName: `${userData.firstName} ${userData.lastName}`,
              confirmedByRole:
                userData.access === true
                  ? userData.userRole === "ADMIN"
                    ? userData.userRole
                    : "RETREAT SUB-ADMIN"
                  : userData.userRole,
              confirmedByEmail: userData.email,
              ConfirmedDate: myDate,
            };
          }
          return entry;
        });
      };

      const confirmData = getUpdatedConfirmation(confirmation, item, userData);
      const confirmDataString = JSON.stringify(confirmData);

      // Perform the update using Sequelize
      const updateResult = await retreatSchema.update(
        { confirmation: confirmDataString },
        { where: { id: retreatData.id } }
      );
      if (updateResult[0] === 0) {
        return res.status(200).json({
          success: false,
          message: "Retreat not found or no updates applied.",
        });
      }

      // Fetch updated retreat data
      const getRetreat = await retreatSchema.findOne({
        where: { id: retreatData.id },
        raw: true,
      });

      if (getRetreat.schedule) {
        try {
          getRetreat.schedule = JSON.parse(getRetreat.schedule);
        } catch (error) {
          console.error("Error parsing schedule:", error);
        }
      }

      if (getRetreat.confirmation) {
        try {
          getRetreat.confirmation = JSON.parse(getRetreat.confirmation);
        } catch (error) {
          console.error("Error parsing confirmation:", error);
        }
      }
      res.status(200).json({
        success: true,
        data: getRetreat,
        message: "Confirmation updated successfully.",
      });
    } catch (error) {
      console.error("Error in handleConfirmation:", error);
      res.status(500).json({
        success: false,
        message: "Failed to update confirmation.",
        error: error.message,
      });
    }
  }

  async retreatChangeStatus(req, res) {
    try {
      const { alertData, userData, editBox } = req.body;
      if (!alertData || !userData) {
        return res.status(200).json({
          success: false,
          message: "Invalid input: alertData and userData are required.",
        });
      }

      const { id, status, confirmation, inactiveReason } = alertData;
      if (!id) {
        return res.status(200).json({
          success: false,
          message: "Retreat ID is required.",
        });
      }

      const updateData = {};

      if (editBox) {
        // Only update inactiveReason when editBox is true
        updateData.inactiveReason = inactiveReason || null;
      } else {
        // Toggle status
        const newStatus = status === "ACTIVE" ? "INACTIVE" : "ACTIVE";

        let conf;
        try {
          conf =
            Array.isArray(confirmation) && confirmation.length
              ? confirmation
              : typeof confirmation === "string"
              ? JSON.parse(confirmation)
              : [
                  {
                    roasterData: false,
                    confirmedByName: "",
                    confirmedByRole: "",
                    confirmedByEmail: "",
                    ConfirmedDate: "",
                  },
                  {
                    Directory: false,
                    confirmedByName: "",
                    confirmedByRole: "",
                    confirmedByEmail: "",
                    ConfirmedDate: "",
                  },
                  {
                    Program: false,
                    confirmedByName: "",
                    confirmedByRole: "",
                    confirmedByEmail: "",
                    ConfirmedDate: "",
                  },
                  {
                    pictures: false,
                    confirmedByName: "",
                    confirmedByRole: "",
                    confirmedByEmail: "",
                    ConfirmedDate: "",
                  },
                  {
                    retreatStatus: "ACTIVE",
                    confirmedByName: "",
                    confirmedByRole: "",
                    confirmedByEmail: "",
                    ConfirmedDate: "",
                  },
                ];
        } catch (error) {
          return res.status(200).json({
            success: false,
            message: "Invalid confirmation format.",
            error: error.message,
          });
        }

        const padZero = (num) => (num < 10 ? `0${num}` : num);
        const formatDate = (date) =>
          `${date.getFullYear()}-${padZero(date.getMonth() + 1)}-${padZero(
            date.getDate()
          )} ${padZero(date.getHours())}:${padZero(
            date.getMinutes()
          )}:${padZero(date.getSeconds())}`;
        const myDate = formatDate(new Date());

        // Update confirmation info
        conf[3].retreatStatus = newStatus;
        conf[3].confirmedByName = `${userData.firstName} ${userData.lastName}`;
        conf[3].confirmedByRole =
          userData.access === true
            ? userData.userRole === "ADMIN"
              ? userData.userRole
              : "RETREAT SUB-ADMIN"
            : userData.userRole;
        conf[3].confirmedByEmail = userData.email;
        conf[3].ConfirmedDate = myDate;

        updateData.status = newStatus;
        updateData.confirmation = JSON.stringify(conf);

        if (newStatus === "INACTIVE") {
          updateData.inactiveReason = inactiveReason || null;
        }
      }

      const updateResult = await retreatSchema.update(updateData, {
        where: { id },
      });

      if (updateResult[0] === 0) {
        return res.status(200).json({
          success: false,
          message:
            "No changes made: Retreat not found or already in the desired status.",
        });
      }

      const getRetreat = await retreatSchema.findOne({ where: { id } });
      if (getRetreat) {
        try {
          getRetreat.schedule = getRetreat.schedule
            ? JSON.parse(getRetreat.schedule)
            : null;
          getRetreat.confirmation = getRetreat.confirmation
            ? JSON.parse(getRetreat.confirmation)
            : null;
        } catch (error) {
          console.error("Error parsing retreat fields:", error);
        }
      }

      res.status(200).json({
        success: true,
        data: getRetreat,
        message: editBox
          ? "Inactive reason updated successfully."
          : `Retreat is now ${
              updateData.status === "ACTIVE" ? "Activated" : "Deactivated"
            }.`,
      });
    } catch (error) {
      console.error("Error in retreatChangeStatus:", error);
      res.status(500).json({
        success: false,
        message: "Failed to update retreat status.",
        error: error.message,
      });
    }
  }

  async retreatCloseStatus(req, res) {
    try {
      const { retreatId } = req.body;
      if (!retreatId) {
        return res.status(200).json({
          success: false,
          message: "Retreat ID is required.",
        });
      }

      const currentRetreat = await retreatSchema.findOne({
        where: { id: retreatId },
      });

      if (!currentRetreat) {
        return res.status(200).json({
          success: false,
          message: "Retreat not found.",
        });
      }

      const newStatus = !currentRetreat.retreatCloseStatus;

      await retreatSchema.update(
        { retreatCloseStatus: newStatus },
        { where: { id: retreatId } }
      );

      const updatedRetreat = await retreatSchema.findOne({
        where: { id: retreatId },
        raw: true,
      });

      if (newStatus === true) {
        const treasurerEmails = [
          // { name: "Irene Mitschke", email: "manpreet02701@gmail.com" },
          // { name: "Didi Kolkebeck", email: "kolkebeckd@gmail.com" },
          // { name: "Tom Kolkebeck", email: "tkolkebeck@gmail.com" },
          { name: "Jimmie Flores", email: "jimmieflores@gmail.com" },
        ];

        for (const person of treasurerEmails) {
          const replData = MAILTOTREASUREPEOPLE.replace(
            /#name#/g,
            person.name || ""
          )
            .replace(/#retreat#/g, updatedRetreat?.title || "")
            .replace(
              /#date#/g,
              moment(updatedRetreat.dateFrom).isSame(
                updatedRetreat.dateTo,
                "day"
              )
                ? `(${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("MMM-DD-YYYY")})`
                : `(${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("MMM DD")} - ${moment(updatedRetreat.dateTo)
                    .tz("America/New_York")
                    .format("MMM DD")}, ${moment(updatedRetreat.dateFrom)
                    .tz("America/New_York")
                    .format("YYYY")})` || ""
            );
          await Email.sendEmail(
            person.email,
            "Retreat Validated and Closed",
            replData
          );
        }
      }
      if (updatedRetreat.schedule) {
        try {
          updatedRetreat.schedule = JSON.parse(updatedRetreat.schedule);
        } catch (error) {
          console.error("Error parsing schedule:", error);
        }
      }

      if (updatedRetreat.confirmation) {
        try {
          updatedRetreat.confirmation = JSON.parse(updatedRetreat.confirmation);
        } catch (error) {
          console.error("Error parsing confirmation:", error);
        }
      }
      return res.status(200).json({
        success: true,
        data: updatedRetreat,
        message: `Retreat is now ${newStatus ? "Closed" : "Reopened"}.`,
      });
    } catch (error) {
      console.error("Error in retreatCloseStatus:", error);
      return res.status(200).json({
        success: false,
        message: "Failed to update retreat close status.",
        error: error.message,
      });
    }
  }

  async getSelectedRetreatData(req, res) {
    try {
      const { retreats = [] } = req.body;
      if (!Array.isArray(retreats) || retreats.length === 0) {
        return res.status(200).json({
          success: false,
          message: "No retreat IDs provided.",
        });
      }

      const query = `
      SELECT 
        retreats.id AS retreatId, retreats.location, retreats.language, retreats.title,
        retreats.heading, retreats.dateFrom, retreats.dateTo, retreats.createdAt,
        parRetreat.parish AS parish,

        retreatroasters.attendeeType,
        couples.hisLastName, couples.hisFirstName, couples.primaryKey,
        couples.herFirstName, couples.herLastName, couples.hisEmail, couples.herEmail,
        couples.hisMobile, couples.herMobile, couples.id AS coupleId,
        couples.accountStatus, couples.city, couples.state, couples.zip,
        couples.notes, couples.address, couples.allergies, couples.anniversary_date,
        couples.under_age_35, couples.hear_us, couples.referral_from_a_friend_or_family_member,
        couples.emergency_name1, couples.emergency_name2, couples.emergency_relationship1,
        couples.emergency_relationship2, couples.emergency_phone1, couples.emergency_phone2,
        couples.imageUrl,
        parish1.parish AS coupleParish,
        services.service, roles.role, talks.talk,

        COUNT(IF(retreatroasters.attendeeType = 'Retreatant', 1, NULL)) AS countRetreatant,
        COUNT(IF(retreatroasters.attendeeType = 'Team', 1, NULL)) AS countTeam,
        COUNT(retreatroasters.id) AS roasterTotal

      FROM retreats
      LEFT JOIN parishes AS parRetreat ON retreats.parishId = parRetreat.id
      LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id
      LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id

      WHERE (retreats.deleteStatus IS NULL OR retreats.deleteStatus = '')
        AND retreats.id IN (${retreats.map(() => "?").join(",")})

      GROUP BY retreats.id, couples.id, retreatroasters.id
      ORDER BY couples.primaryKey DESC
    `;

      const responseData = await db.sequelize.query(query, {
        replacements: retreats,
        type: db.Sequelize.QueryTypes.SELECT,
      });

      return res.status(200).json({
        success: true,
        data: responseData,
      });
    } catch (error) {
      console.error("Error fetching retreat export data:", error);
      return res.status(200).json({
        success: false,
        message: "Failed to fetch retreat data.",
        error: error.message,
      });
    }
  }

  async getAllRetreatData(req, res) {
    try {
      const query = `
      SELECT 
        retreats.id AS retreatId, retreats.location, retreats.language, retreats.title,
        retreats.heading, retreats.dateFrom, retreats.dateTo, retreats.createdAt,
        parRetreat.parish AS parish,

        retreatroasters.attendeeType,
        couples.hisLastName, couples.hisFirstName, couples.primaryKey,
        couples.herFirstName, couples.herLastName, couples.hisEmail, couples.herEmail,
        couples.hisMobile, couples.herMobile, couples.id AS coupleId,
        couples.accountStatus, couples.city, couples.state, couples.zip,
        couples.notes, couples.address, couples.allergies, couples.anniversary_date,
        couples.under_age_35, couples.hear_us, couples.referral_from_a_friend_or_family_member,
        couples.emergency_name1, couples.emergency_name2, couples.emergency_relationship1,
        couples.emergency_relationship2, couples.emergency_phone1, couples.emergency_phone2,
        couples.imageUrl,
        parish1.parish AS coupleParish,
        services.service, roles.role, talks.talk,

        COUNT(IF(retreatroasters.attendeeType = 'Retreatant', 1, NULL)) AS countRetreatant,
        COUNT(IF(retreatroasters.attendeeType = 'Team', 1, NULL)) AS countTeam,
        COUNT(retreatroasters.id) AS roasterTotal

      FROM retreats
      LEFT JOIN parishes AS parRetreat ON retreats.parishId = parRetreat.id
      LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
      LEFT JOIN couples ON retreatroasters.coupleId = couples.id
      LEFT JOIN parishes AS parish1 ON couples.parishId = parish1.id
      LEFT JOIN services ON retreatroasters.serviceId = services.id
      LEFT JOIN roles ON retreatroasters.roleId = roles.id
      LEFT JOIN talks ON retreatroasters.talkId = talks.id

      WHERE (retreats.deleteStatus IS NULL OR retreats.deleteStatus = '')
      GROUP BY retreats.id, couples.id, retreatroasters.id
      ORDER BY couples.primaryKey DESC
    `;

      const responseData = await db.sequelize.query(query, {
        type: db.Sequelize.QueryTypes.SELECT,
      });

      return res.status(200).json({
        success: true,
        data: responseData,
      });
    } catch (error) {
      console.error("Error fetching retreat export data:", error);
      return res.status(200).json({
        success: false,
        message: "Failed to fetch retreat data.",
        error: error.message,
      });
    }
  }

  async retreatDisabledSubAdmin(req, res) {
    try {
      const { retreatData, mail } = req.body;

      if (!retreatData.id || !mail) {
        return res.status(400).json({
          success: false,
          message: "retreatId and mail are required",
        });
      }

      let extractedEmail = mail.split(" - ").pop().trim();
      // Find userId from email
      const user = await userSchema.findOne({
        where: { email: extractedEmail },
      });
      if (!user) {
        return res.status(404).json({
          success: false,
          message: "User not found for email: " + extractedEmail,
        });
      }
      const userId = user.id;
      // Fetch retreat
      let retreat = await retreatSchema.findOne({
        where: { id: retreatData.id },
      });

      if (!retreat) {
        return res.status(404).json({
          success: false,
          message: "Retreat not found",
        });
      }
      // Parse accessIds
      let accessIds = retreat.accessIds;
      if (typeof accessIds === "string") {
        accessIds = JSON.parse(accessIds);
      }
      if (Array.isArray(accessIds) && typeof accessIds[0] !== "object") {
        accessIds = accessIds.map((id) => ({
          userId: id,
          isDisabled: false,
        }));
      }

      // Find this user in accessIds
      let foundUser = accessIds.find((u) => u.userId === userId);

      if (foundUser) {
        // Toggle disable flag
        foundUser.isDisabled = !foundUser.isDisabled;
      } else {
        // If not found, add with default enabled
        accessIds.push({ userId, isDisabled: false });
      }

      // Save back
      await retreatSchema.update(
        { accessIds: JSON.stringify(accessIds) },
        { where: { id: retreatData.id } }
      );

      return res.status(200).json({
        success: true,
        message: "Sub-admin status updated successfully",
        data: accessIds,
      });
    } catch (error) {
      console.error("Error updating sub-admin status:", error);
      return res.status(500).json({
        success: false,
        message: "Failed to update sub-admin status",
        error: error.message,
      });
    }
  }
}

module.exports = new RetreatController();
