const db = require("../database/dbConnection");
const retreatSchema = db.retreat;
const programSchema = db.program;
const retreatRoastSchema = db.retreatRoaster;
const directorySchema = db.directory;
const { PROJECT_DIR } = require("../../config");
const fs = require("fs");
const path = require("path");

class RetreatBackUpController {
  
  async getBackUpRetreats(req, res) {
    try {
      let { searchKeyword, sortBy, pageNo, size, allRetreat, coupleId } =
        req.body;
      let query = { skip: 0, limit: 0 };
      let sort1 = "DESC";
      let sortColumn = "retreats.createdAt";
      let searchColumn = "";
      if (pageNo || size) {
        pageNo = parseInt(pageNo);
        size = parseInt(size);
        if (pageNo < 0 || pageNo === 0) {
          response = {
            error: true,
            msg: "Invalid page number, should start with 1",
          };
          return res.status(200).send(response);
        }
        query.skip = size * (pageNo - 1);
        query.limit = size;
      }
      if (sortBy && sortBy.sortDirection == 1) {
        sort1 = "ASC";
      } else {
        sort1 = "DESC";
      }
      if (sortBy && sortBy.sortColumn && sortBy.sortDirection) {
        if (sortBy.sortColumn === "language") {
          sortColumn = "retreats.language";
        } else if (sortBy.sortColumn === "location") {
          sortColumn = "retreats.location";
        } else if (sortBy.sortColumn === "title") {
          sortColumn = "retreats.title";
        } else if (sortBy.sortColumn === "heading") {
          sortColumn = "retreats.heading";
        } else if (sortBy.sortColumn === "mission_statement") {
          sortColumn = "retreats.mission_statement";
        } else if (sortBy.sortColumn === "description") {
          sortColumn = "retreats.description";
        }
      }
      if (searchKeyword && searchKeyword != "") {
        searchColumn = `retreats.title LIKE "%${searchKeyword}%" OR
        retreats.language LIKE "%${searchKeyword}%" OR
        retreats.location LIKE "%${searchKeyword}%" OR
        retreats.title LIKE "%${searchKeyword}%" OR
        retreats.heading LIKE "%${searchKeyword}%" OR
        retreats.mission_statement LIKE "%${searchKeyword}%" OR
        retreats.description LIKE "%${searchKeyword}%"`;
      }
      if (coupleId && coupleId !== "") {
        searchColumn += ` retreatroasters.coupleId = ${coupleId}`;
      }
      let select = `SELECT retreats.id,retreats.title,retreats.language,retreats.location,retreats.title,retreats.heading,retreats.mission_statement,retreats.dateFrom,retreats.dateTo,retreats.description,retreats.schedule,retreats.deleteStatus,retreats.image1,retreats.image2,retreats.createdAt,retreats.updatedAt,parishes.parish AS parishName,parishes.city AS parishCity,parishes.address AS parishAddress,parishes.state AS parishState,parishes.zip AS parishzip,parishes.link AS parishLink,parishes.imageUrl AS parishImageUrl,`;
      let count = ` COUNT(IF(retreatroasters.attendeeType = 'Retreatant',retreatroasters.id,NULL)) AS countRetreatant,
      COUNT(retreatroasters.id) AS roasterTotal, 
      COUNT(IF(retreatroasters.attendeeType = 'Team',retreatroasters.id,NULL)) AS countTeam `;
      let from = ` FROM retreats 
      LEFT JOIN retreatroasters ON retreats.id = retreatroasters.retreatId
      LEFT JOIN parishes ON retreats.parishId = parishes.id`;
      let where =
        (searchKeyword && searchKeyword != "") || (coupleId && coupleId !== "")
          ? ` WHERE (${searchColumn}) AND retreats.deleteStatus='temporary'`
          : ` WHERE retreats.deleteStatus='temporary'`;
      let whereFrom = from + where;
      let groupBy = ``;
      if (allRetreat) {
        groupBy += ` GROUP BY retreats.id`;
      } else {
        groupBy += ` GROUP BY retreats.id
        ORDER BY ${sortColumn} ${sort1}
        LIMIT ${query.limit}
        OFFSET ${query.skip}`;
      }
      let retretTotSelect = `SELECT COUNT(DISTINCT retreats.id) as totalRetreat`;
      let queryRecords = select + count + whereFrom + groupBy;
      let retreatTotal = retretTotSelect + whereFrom;
      let responseData1 = await db.sequelize
        .query(queryRecords)
        .then(([results, metadata]) => results);
      let totalRetreats = await db.sequelize
        .query(retreatTotal)
        .then(([results, metadata]) => results);
      return res.status(200).send({
        success: true,
        data: responseData1,
        totalRecords: totalRetreats[0].totalRetreat,
      });
    } catch (error) {
      return res.send({
        error: true,
        msg: "Something went wrong please try again.",
      });
    }
  }

  async permanentDeleteRetreat(req, res) {
    try {
      const { id } = req.body;
      if (!id) {
        return res.status(200).json({
          success: false,
          msg: "ID is required.",
        });
      }

      // Find retreat first (to get image paths before deleting DB row)
      const retreat = await retreatSchema.findOne({ where: { id } });

      if (!retreat) {
        return res.status(200).json({
          success: false,
          msg: "Retreat not found.",
        });
      }

      // Delete related records
      await directorySchema.destroy({ where: { retreatId: id } });
      await programSchema.destroy({ where: { retreatId: id } });
      await retreatRoastSchema.destroy({ where: { retreatId: id } });

      // Delete the retreat record
      const deleted = await retreatSchema.destroy({ where: { id } });

      if (deleted) {
        // Remove images if they exist
        const images = [retreat.image1, retreat.image2];
        images.forEach((imgUrl) => {
          if (imgUrl) {
            const filePath = path.join(
              PROJECT_DIR,
              "retreatImage",
              path.basename(imgUrl)
            );
            fs.unlink(filePath, (err) => {
              if (err) console.warn("Could not delete file:", filePath, err);
            });
          }
        });

        return res.status(200).send({
          success: true,
          msg: "Retreat and related data deleted permanently.",
        });
      } else {
        return res.status(200).json({
          success: false,
          msg: "Retreat not found.",
        });
      }
    } catch (error) {
      console.error("Error deleting retreat:", error);
      return res.status(200).json({
        success: false,
        msg: error.message || error,
      });
    }
  }

  async restoreRetreat(req, res) {
    try {
      const { id } = req.body;
      if (!id) {
        return res.status(400).json({
          success: false,
          msg: "ID is required.",
        });
      }
      let restoreTemporary = await retreatSchema.update(
        { deleteStatus: "" },
        { where: { id: id } }
      );
      if (restoreTemporary) {
        return res.status(200).send({
          success: true,
          msg: "Retreat has been activated again.",
        });
      } else {
        return res.status(404).json({
          success: false,
          msg: "Retreat not found.",
        });
      }
    } catch (error) {
      console.error("Error deleting retreat:", error);
      return res.status(500).json({
        success: false,
        msg: "An error occurred while deleting the retreat.",
      });
    }
  }
}
module.exports = new RetreatBackUpController();
