const db = require("../database/dbConnection");
const userSchema = db.user;
const helpSchema = db.help;
const ticketMessageSchema = db.ticketMessage;
const { ASSIGNTICKETTOUSER } = require("../helper/emailTemplate");
const Email = require("../helper/sendEmail");
const { PROJECT_DIR } = require("../../config");
const fs = require("fs");
const path = require("path");

class HelpController {
  // ✅ Create Ticket
  async createTicket(req, res) {
    try {
      const {
        firstName,
        lastName,
        email,
        mobile,
        subject,
        message,
        captcha,
        captchaAnswer,
        pageName,
        category,
      } = req.body;

      if (captcha !== captchaAnswer) {
        return res.status(200).send({ success: false, msg: "Invalid captcha" });
      }

      const ticketNumber = Date.now().toString();
      let attachmentUrls = [];

      // ✅ Handle file uploads
      if (req.files && req.files.attachments) {
        let files = req.files.attachments;
        // Normalize single file vs multiple files
        if (!Array.isArray(files)) {
          files = [files];
        }
        for (const file of files) {
          const now = new Date();
          const currentDate = now.toISOString().split("T")[0];
          const fileExtension = file.name.substring(file.name.lastIndexOf("."));
          const fileName = `${ticketNumber}_${currentDate}_${file.name}`;
          const filePath = PROJECT_DIR + "/ticketAttachments/" + fileName;
          const fileUrl =
            process.env.SERVERADDRESS + "/public/ticketAttachments/" + fileName;

          // Move file
          await file.mv(filePath, (err) => {
            if (err) throw err;
          });

          attachmentUrls.push(fileUrl);
        }
      }

      const newTicket = await helpSchema.create({
        firstName,
        lastName,
        email,
        mobile,
        subject,
        message,
        captcha,
        pageName,
        category,
        ticketNumber,
        attachments: attachmentUrls.length > 0 ? attachmentUrls : null,
      });

      return res.status(200).send({
        success: true,
        msg: "Ticket created successfully",
        data: newTicket,
      });
    } catch (error) {
      return res.status(200).send({
        success: false,
        msg: error.message,
      });
    }
  }

  // ✅ Get All Tickets
  async getAllTickets(req, res) {
    try {
      let { searchKeyword, sortBy, pageNo, size, filters } = req.body;
      let query = { skip: 0, limit: 0 };
      let sort1 = "ASC";
      let sortColumn = "help.createdAt";
      // 📌 Pagination
      if (pageNo || size) {
        pageNo = parseInt(pageNo);
        size = parseInt(size);
        if (pageNo < 1) {
          return res.status(200).send({
            success: false,
            msg: "Invalid page number, should start with 1",
          });
        }
        query.skip = size * (pageNo - 1);
        query.limit = size;
      }
      // 📌 Sorting
      if (sortBy?.sortDirection === 1) {
        sort1 = "ASC";
      } else {
        sort1 = "DESC";
      }

      if (sortBy?.sortColumn) {
        const columnMap = {
          firstName: "help.firstName",
          lastName: "help.lastName",
          email: "help.email",
          mobile: "help.mobile",
          subject: "help.subject",
          created: "help.createdAt",
          assignedTo: "help.assignedTo",
          category: "help.category",
          ticketNumber: "help.ticketNumber",
        };
        sortColumn = columnMap[sortBy.sortColumn] || sortColumn;
      }

      // 📌 WHERE clause
      const whereParts = [];

      // 🔍 Search
      if (searchKeyword && searchKeyword !== "") {
        const searchColumn = `
        help.firstName LIKE "%${searchKeyword}%" OR
        help.lastName LIKE "%${searchKeyword}%" OR
        help.email LIKE "%${searchKeyword}%" OR
        help.mobile LIKE "%${searchKeyword}%" OR
        help.subject LIKE "%${searchKeyword}%" OR
        help.message LIKE "%${searchKeyword}%" OR
        help.category LIKE "%${searchKeyword}%" OR
        u.firstName LIKE "%${searchKeyword}%" OR
        u.lastName LIKE "%${searchKeyword}%" OR
        u.email LIKE "%${searchKeyword}%"OR
        help.ticketNumber LIKE "%${searchKeyword}%"
      `;
        whereParts.push(`(${searchColumn})`);
      }

      // 🎯 Filters
      if (filters) {
        const { category, status, priority } = filters;
        if (category) whereParts.push(`help.category = "${category}"`);
        if (status) whereParts.push(`help.status = "${status}"`);
        if (priority) whereParts.push(`help.priority = "${priority}"`);
      }

      const where = whereParts.length
        ? `WHERE ${whereParts.join(" AND ")}`
        : "";

      // 📌 Main Query
      const select = `
      SELECT 
        help.*, 
        u.firstName AS assignedUserFirstName,
        u.lastName AS assignedUserLastName,
        u.email AS assignedUserEmail
      FROM help
      LEFT JOIN users u ON u.id = help.assignedTo
    `;

      const groupBy = ` GROUP BY help.id ORDER BY ${sortColumn} ${sort1}`;
      const limitOffset = req.body.export
        ? ""
        : ` LIMIT ${query.limit} OFFSET ${query.skip}`;

      const queryRecords = select + where + groupBy + limitOffset;

      const responseData = await db.sequelize
        .query(queryRecords)
        .then(([results]) => results);

      // 📌 Count query
      const countQuery = `
      SELECT COUNT(DISTINCT help.id) AS totalRecords
      FROM help
      LEFT JOIN users u ON u.id = help.assignedTo
      ${where}
    `;

      const totalRecords = await db.sequelize
        .query(countQuery)
        .then(([results]) => results[0]);

      return res.status(200).send({
        success: true,
        data: responseData,
        totalRecords: totalRecords?.totalRecords || 0,
      });
    } catch (error) {
      console.log("getAllTickets error:", error);
      return res.status(200).send({
        success: false,
        msg: "Something went wrong, please try again.",
      });
    }
  }

  // ✅ Delete Ticket
async deleteTicket(req, res) {
  try {
    const { ticket_id } = req.body;
    if (!ticket_id) {
      return res.status(200).send({
        success: false,
        msg: "Ticket ID is required.",
      });
    }

    // 1. Get ticket details (to find attachments)
    const ticketMessages = await ticketMessageSchema.findAll({
      where: { helpId: ticket_id },
      raw: true,
    });

    // 2. Delete attached files if they exist
    for (const message of ticketMessages) {
      if (message.attachmentUrl) {
        const filePath = path.join(
          PROJECT_DIR,
          "ticketAttachments",
          path.basename(message.attachmentUrl)
        );

        if (fs.existsSync(filePath)) {
          fs.unlinkSync(filePath);
        }
      }
    }

    // 3. Delete child messages
    await ticketMessageSchema.destroy({
      where: { helpId: ticket_id },
    });

    // 4. Delete the ticket itself
    await helpSchema.destroy({
      where: { id: ticket_id },
    });

    return res.status(200).send({
      success: true,
      msg: "Ticket and attachments deleted successfully.",
    });
  } catch (error) {
    console.log("error:", error);
    return res.status(200).send({
      success: false,
      msg: error.message,
    });
  }
}


  // ✅ Update Ticket (status, priority, or both)
  async updateStatus(req, res) {
    try {
      const { id, status, priority } = req.body;

      if (!id) {
        return res.status(200).json({
          success: false,
          msg: "Ticket ID is required.",
        });
      }

      // Find ticket
      let ticket = await helpSchema.findByPk(id);
      if (!ticket) {
        return res.status(200).json({
          success: false,
          msg: "Ticket not found.",
        });
      }

      // Validate & update status
      if (status) {
        const validStatuses = ["NEW", "INPROGRESS", "CLOSED"];
        if (!validStatuses.includes(status)) {
          return res.status(200).json({
            success: false,
            msg: `Invalid status. Allowed: ${validStatuses.join(", ")}`,
          });
        }
        ticket.status = status;
      }

      // Validate & update priority
      if (priority) {
        const validPriorities = ["LOW", "MEDIUM", "HIGH"];
        if (!validPriorities.includes(priority)) {
          return res.status(200).json({
            success: false,
            msg: `Invalid priority. Allowed: ${validPriorities.join(", ")}`,
          });
        }
        ticket.priority = priority;
      }

      await ticket.save();

      return res.status(200).json({
        success: true,
        msg: "Ticket updated successfully.",
        data: ticket,
      });
    } catch (error) {
      console.error("updateStatus error:", error.message || error);
      return res.status(200).json({
        success: false,
        msg: "Something went wrong while updating ticket.",
      });
    }
  }

  // Controller function
  async sendTicketMsg(req, res) {
    try {
      const {
        selectedTicket,
        selectedRecipient,
        isInputField,
        message,
        priority,
      } = req.body;

      let updatedMessage = message || selectedTicket.message;
      let updatedPriority = priority || "MEDIUM";

      // 🔹 Get existing attachments from the help ticket
      const parentTicket = await helpSchema.findByPk(selectedTicket.id, {
        attributes: ["attachments"],
      });
      let attachments = null;
      if (parentTicket?.attachments) {
        try {
          // Parse to confirm it’s valid JSON
          attachments = JSON.parse(parentTicket.attachments);
        } catch (err) {
          // If already plain string
          attachments = parentTicket.attachments;
        }
      }

      if (isInputField) {
        const replData = ASSIGNTICKETTOUSER.replace(/#RecipientName#/g, `Dear`)
          .replace(
            /#SenderName#/g,
            `${selectedTicket.firstName} ${selectedTicket.lastName}`
          )
          .replace(/#ticketNumber#/g, selectedTicket.ticketNumber)
          .replace(/#Subject#/g, selectedTicket.subject)
          .replace(/#Message#/g, updatedMessage)
          .replace(/#Priority#/g, updatedPriority)
          .replace(
            /#TicketLink#/g,
            `${process.env.SERVERADDRESS}/ticketReply/${selectedTicket.id}`
          );

        await Email.sendEmail(
          "manpreet02701@gmail.com",
          "New Ticket Assigned",
          replData
        );

        await helpSchema.update(
          {
            assignedTo: "manpreet02701@gmail.com",
            status: "INPROGRESS",
            message: updatedMessage,
            priority: updatedPriority,
          },
          { where: { id: selectedTicket.id } }
        );

        await ticketMessageSchema.create({
          helpId: selectedTicket.id,
          message: updatedMessage, // ✅ use updatedMessage
          senderName: `ADMIN`,
          senderEmail: "admin@gmail.com",
          attachments, // ✅ reuse same file URLs
        });
      } else {
        const findUser = await userSchema.findByPk(selectedRecipient);
        if (!findUser) {
          return res
            .status(200)
            .json({ success: false, msg: "Recipient not found." });
        }

        const replData = ASSIGNTICKETTOUSER.replace(
          /#RecipientName#/g,
          `Dear ${findUser.firstName} ${findUser.lastName}`
        )
          .replace(
            /#SenderName#/g,
            `${selectedTicket.firstName} ${selectedTicket.lastName}`
          )
          .replace(/#ticketNumber#/g, selectedTicket.ticketNumber)
          .replace(/#Subject#/g, selectedTicket.subject)
          .replace(/#Message#/g, updatedMessage)
          .replace(/#Priority#/g, updatedPriority)
          .replace(
            /#TicketLink#/g,
            `${process.env.SERVERADDRESS}/ticketReply/${selectedTicket.id}`
          );

        await Email.sendEmail(findUser.email, "New Ticket Assigned", replData);

        await helpSchema.update(
          {
            assignedTo: findUser.id,
            status: "INPROGRESS",
            message: updatedMessage,
            priority: updatedPriority,
          },
          { where: { id: selectedTicket.id } }
        );

        await ticketMessageSchema.create({
          helpId: selectedTicket.id,
          message: updatedMessage,
          senderName: `ADMIN`,
          senderEmail: `admin@gmail.com`,
          attachments, // ✅ reuse same file URLs
        });
      }

      return res.status(200).json({
        success: true,
        msg: "Ticket assigned and email sent successfully.",
      });
    } catch (error) {
      console.error("sendTicketMsg error:", error.message || error);
      return res.status(200).json({
        success: false,
        msg: "Something went wrong while assigning the ticket.",
      });
    }
  }

  async getTicketById(req, res) {
    try {
      const { ticket_id } = req.body;

      if (!ticket_id) {
        return res.status(200).send({
          success: false,
          msg: "Ticket ID is required.",
        });
      }

      const ticket = await helpSchema.findOne({ where: { id: ticket_id } });

      if (!ticket) {
        return res.status(200).send({
          success: false,
          msg: "Ticket not found.",
        });
      }

      return res.status(200).send({
        success: true,
        data: ticket,
      });
    } catch (error) {
      console.log("error:", error);
      return res.status(200).send({
        success: false,
        msg: error.message,
      });
    }
  }

  // ✅ Get all messages for a ticket
  async getTicketMessages(req, res) {
    try {
      const { ticket_id } = req.body;
      if (!ticket_id) {
        return res.status(200).send({
          success: false,
          msg: "Ticket ID is required.",
        });
      }

      const messages = await ticketMessageSchema.findAll({
        where: { helpId: ticket_id },
        order: [["createdAt", "DESC"]],
      });

      return res.status(200).send({
        success: true,
        data: messages,
      });
    } catch (error) {
      console.log("error:", error);
      return res.status(200).send({
        success: false,
        msg: error.message,
      });
    }
  }

  // ✅ Add a new reply message to a ticket
  async addTicketMessage(req, res) {
    try {
      const { ticket_id, message, userId } = req.body;

      if (!ticket_id || !message || !userId) {
        return res.status(200).send({
          success: false,
          msg: "Ticket ID, message, and userId are required.",
        });
      }

      let attachmentUrls = [];

      // ✅ Handle file uploads
      if (req.files && req.files.attachments) {
        let files = req.files.attachments;

        // Normalize single vs multiple files
        if (!Array.isArray(files)) {
          files = [files];
        }

        for (const file of files) {
          const now = new Date();
          const currentDate = now.toISOString().split("T")[0];
          const fileExtension = file.name.substring(file.name.lastIndexOf("."));
          const fileName = `${ticket_id}_${currentDate}_${file.name}`;
          const filePath = PROJECT_DIR + "/ticketAttachments/" + fileName;
          const fileUrl =
            process.env.SERVERADDRESS + "/public/ticketAttachments/" + fileName;

          // Move file
          await file.mv(filePath, (err) => {
            if (err) throw err;
          });

          attachmentUrls.push(fileUrl);
        }
      }

      let newMessage = "";

      if (userId === "manpreet02701@gmail.com") {
        newMessage = await ticketMessageSchema.create({
          helpId: ticket_id,
          message,
          senderName: `Manpreet Singh`,
          senderEmail: "manpreet02701@gmail.com",
          attachments: attachmentUrls.length > 0 ? attachmentUrls : null,
        });
      } else if (userId === "ADMIN") {
        newMessage = await ticketMessageSchema.create({
          helpId: ticket_id,
          message,
          senderName: `ADMIN`,
          senderEmail: "admin@gmail.com",
          attachments: attachmentUrls.length > 0 ? attachmentUrls : null,
        });
      } else {
        let findUser = await userSchema.findByPk(userId);
        newMessage = await ticketMessageSchema.create({
          helpId: ticket_id,
          message,
          senderName: `${findUser.firstName ? findUser.firstName : ""} ${
            findUser.lastName ? findUser.lastName : ""
          }`,
          senderEmail: findUser.email,
          attachments: attachmentUrls.length > 0 ? attachmentUrls : null,
        });
      }

      return res.status(200).send({
        success: true,
        msg: "Message added successfully.",
        data: newMessage,
      });
    } catch (error) {
      console.log("error:", error);
      return res.status(200).send({
        success: false,
        msg: error.message,
      });
    }
  }
}
module.exports = new HelpController();
