const Email = require("../helper/sendEmail");
const db = require("../database/dbConnection");
const { SENDLOGINDETAILTOSUBADMIN } = require("../helper/emailTemplate");
const { Op, Sequelize } = require("sequelize");
const committeeMemberSchema = db.committeeMember;
const userSchema = db.user;
const retreatSchema = db.retreat;
const coupleSchema = db.couple;
const perishSchema = db.parish;

class CommitteeController {
  async createCommitteeMember(req, res) {
    try {
      let { id, userId, parishId, retreats } = JSON.parse(
        `${req.body.fromData}`
      );
      let formattedRetreats = [];
      if (Array.isArray(retreats)) {
        formattedRetreats = retreats.map((retreat) => ({
          retreatId: retreat.retreatId
            ? Number(retreat.retreatId)
            : Number(retreat),
          disabled: retreat.disabled || false,
        }));
      } else if (typeof retreats === "string") {
        const retreatIds = retreats.split(",").map(Number);
        formattedRetreats = retreatIds.map((id) => ({
          retreatId: id,
          disabled: false,
        }));
      }
      if (!id || (Array.isArray(id) && id.length === 0)) {
        if (!userId || !Array.isArray(userId) || userId.length === 0) {
          return res.status(200).json({
            success: false,
            error: true,
            msg: "Please add at least one user.",
          });
        }
        let createdMembers = [];

        for (let u of userId) {
          const email = u.value;
          let existingUser = await userSchema.findOne({ where: { email } });
          if (!existingUser) {
            const couple = await coupleSchema.findOne({
              where: {
                [Sequelize.Op.or]: [{ hisEmail: email }, { herEmail: email }],
              },
            });

            if (!couple) {
              console.log(`No couple found for ${email}, skipping`);
              continue;
            }
            const isHis = couple.hisEmail === email;
            const firstName = isHis ? couple.hisFirstName : couple.herFirstName;
            const lastName = isHis ? couple.hisLastName : couple.herLastName;
            const mobile = isHis ? couple.hisMobile : couple.herMobile;
            const coupleId = couple.id;
            const randompass = Math.random().toString(36).slice(-8);
            const passwordForMember = "p" + randompass.slice(1);
            existingUser = await userSchema.create({
              userName: firstName,
              firstName,
              lastName,
              mobile,
              coupleId,
              email,
              password: passwordForMember,
              accountStatus: "ACTIVE",
              userRole: "COUPLE",
            });
          }
          let committeeRecord = await committeeMemberSchema.findOne({
            where: { userId: existingUser.id, parishId },
          });
          if (committeeRecord) {
            let existingRetreats = [];
            try {
              existingRetreats = JSON.parse(committeeRecord.retreats) || [];
            } catch (e) {}
            const merged = [
              ...existingRetreats,
              ...formattedRetreats.filter(
                (r) =>
                  !existingRetreats.some((er) => er.retreatId === r.retreatId)
              ),
            ];

            await committeeMemberSchema.update(
              { retreats: JSON.stringify(merged) },
              { where: { id: committeeRecord.id } }
            );
            createdMembers.push({
              ...committeeRecord.toJSON(),
              retreats: merged,
            });
          } else {
            const newMember = await committeeMemberSchema.create({
              userId: existingUser.id,
              parishId,
              retreats: JSON.stringify(formattedRetreats),
            });
            createdMembers.push(newMember);
            // Send login email (only on first creation)
            const replData = SENDLOGINDETAILTOSUBADMIN.replace(
              /#firstName#/g,
              existingUser.firstName || ""
            )
              .replace(/#lastName#/g, existingUser.lastName || "")
              .replace(/#role#/g, "committee member")
              .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
              .replace(/#email#/g, existingUser.email)
              .replace(/#password#/g, existingUser.password);

            await Email.sendEmail(
              existingUser.email,
              "Login Details - LOVESTRONG Marriage",
              replData
            );
          }
        }
        return res.status(200).json({
          success: true,
          msg: "Members processed successfully.",
          data: createdMembers,
        });
      }
    } catch (error) {
      console.error(`[createCommitteeMember] error:`, error);
      return res.status(200).json({
        success: false,
        msg: "Something went wrong.",
        error: error.message,
      });
    }
  }

async getAllCommitteMembers(req, res) {
  try {
    const members = await committeeMemberSchema.findAll();
    if (!members.length) {
      return res.status(200).send({ success: true, data: {} });
    }

    const allRetreatIds = new Set();
    const allUserIds = members.map((member) => member.userId);

    // fetch all users
    const users = await userSchema.findAll({
      where: { id: allUserIds },
    });

    const userMap = {};
    for (const user of users) {
      userMap[user.id] = user;
    }

    // prepare member data
    const memberData = members.map((member) => {
      let retreatData = [];
      try {
        retreatData = JSON.parse(member.retreats || "[]");
      } catch (e) {
        retreatData = [];
      }

      retreatData.forEach((r) => {
        if (r.retreatId) allRetreatIds.add(r.retreatId);
      });

      const user = userMap[member.userId] || {};

      return {
        committeeId: member.id,
        userId: member.userId,
        firstName: user.firstName || "",
        lastName: user.lastName || "",
        eMail: user.email || "",
        password: user.password || "",
        coupleId: user.coupleId || "",
        parishId: member.parishId,
        retreatIds: retreatData,
        createdAt: member.createdAt,
        updatedAt: member.updatedAt,
        retreats: [],
      };
    });

    // fetch all retreats linked by members (exclude deleted)
    const retreatList = await retreatSchema.findAll({
      where: { 
        id: Array.from(allRetreatIds),
        [Op.or]: [
          { deleteStatus: "" },
          { deleteStatus: null }
        ]
      },
      attributes: ["id", "title", "dateFrom", "dateTo", "parishId"],
      include: [
        {
          model: perishSchema,
          attributes: ["id", "parish"],
        },
      ],
      order: [["dateFrom", "DESC"]],
    });

    const retreatMap = {};
    for (const r of retreatList) {
      retreatMap[r.id] = {
        retreatId: r.id,
        retreatTitle: r.title,
        dateFrom: r.dateFrom,
        dateTo: r.dateTo,
        retreatParishId: r.parishId,
        parishName: r.parish ? r.parish.parish : null,
      };
    }

    // attach retreats to each member
    for (const member of memberData) {
      for (const r of member.retreatIds) {
        if (r && r.retreatId && retreatMap[r.retreatId]) {
          member.retreats.push({
            ...retreatMap[r.retreatId],
            disabled: r.disabled || false,
          });
        }
      }
      member.retreats.sort(
        (a, b) => new Date(b.dateFrom) - new Date(a.dateFrom)
      );
    }

    // group by parish
    const parishData = {};
    for (const member of memberData) {
      let parishName = "";
      if (member.retreats.length > 0) {
        parishName =
          member.retreats[0].parishName || `Parish-${member.parishId}`;
      } else {
        parishName = `Parish-${member.parishId}`;
      }

      if (!parishData[parishName]) {
        parishData[parishName] = {
          parishId: member.parishId,
          parishName: parishName,
          committeeIds: [],
          userIds: [],
          retreatIds: [],
          users: [],
          retreats: [],
        };
      }

      // add committee id
      if (!parishData[parishName].committeeIds.includes(member.committeeId)) {
        parishData[parishName].committeeIds.push(member.committeeId);
      }

      // add user id
      if (!parishData[parishName].userIds.includes(member.userId)) {
        parishData[parishName].userIds.push(member.userId);
      }

      // add user details
      parishData[parishName].users.push({
        committeeId: member.committeeId,
        userId: member.userId,
        firstName: member.firstName,
        lastName: member.lastName,
        email: member.eMail,
        coupleId: member.coupleId,
        password: member.password,
      });

      // add retreats already linked to members
      member.retreats.forEach((retreat) => {
        if (!parishData[parishName].retreatIds.includes(retreat.retreatId)) {
          parishData[parishName].retreatIds.push(retreat.retreatId);
          parishData[parishName].retreats.push({
            ...retreat,
            committeeId: member.committeeId,
          });
        }
      });
    }

    // Add retreats from retreatSchema for each parish (if missing) - exclude deleted
    const allParishIds = [
      ...new Set(members.map((member) => member.parishId)),
    ];

    const allParishRetreats = await retreatSchema.findAll({
      where: { 
        parishId: allParishIds,
        [Op.or]: [
          { deleteStatus: "" },
          { deleteStatus: null }
        ]
      },
      attributes: ["id", "title", "dateFrom", "dateTo", "parishId"],
      include: [
        {
          model: perishSchema,
          attributes: ["id", "parish"],
        },
      ],
      order: [["dateFrom", "DESC"]],
    });

    for (const retreat of allParishRetreats) {
      const parishName =
        retreat.parish?.parish || `Parish-${retreat.parishId}`;

      if (!parishData[parishName]) {
        parishData[parishName] = {
          parishId: retreat.parishId,
          parishName,
          committeeIds: [],
          userIds: [],
          users: [],
          retreatIds: [],
          retreats: [],
        };
      }
      const alreadyLinked = parishData[parishName].retreats.some(
        (r) => r.retreatId === retreat.id
      );
      if (!alreadyLinked) {
        parishData[parishName].retreatIds.push(retreat.id);
        parishData[parishName].retreats.push({
          retreatId: retreat.id,
          retreatTitle: retreat.title,
          dateFrom: retreat.dateFrom,
          dateTo: retreat.dateTo,
          retreatParishId: retreat.parishId,
          parishName,
          committeeId: null, 
        });
      }
    }

    return res.status(200).send({
      success: true,
      data: parishData,
    });
  } catch (error) {
    console.error("getAllCommitteMembers error:", error);
    return res.status(200).send({
      error: true,
      msg: "Something went wrong, please try again.",
    });
  }
}

  async deleteMember(req, res) {
    try {
      const { userId, parishId } = req.body;
      if (!userId || !parishId) {
        return res.status(200).json({
          success: false,
          msg: "Both userId and parishId are required",
        });
      }

      const delete_member = await committeeMemberSchema.destroy({
        where: {
          userId: userId,
          parishId: parishId,
        },
      });

      if (delete_member === 0) {
        return res.status(200).json({
          success: false,
          msg: "No record found with the given userId and parishId",
        });
      }

      return res.status(200).json({
        success: true,
        msg: "Record Deleted",
      });
    } catch (err) {
      return res.status(200).json({
        success: false,
        msg: "Server error",
        error: err.message,
      });
    }
  }

  async newSteeringMember(req, res) {
    try {
      let { userId, parishId, retreats } = JSON.parse(`${req.body.fromData}`);

      // ✅ Format retreats
      let formattedRetreats = [];
      if (Array.isArray(retreats)) {
        formattedRetreats = retreats.map((retreat) => ({
          retreatId: retreat.retreatId
            ? Number(retreat.retreatId)
            : Number(retreat),
          disabled: retreat.disabled || false,
        }));
      } else if (typeof retreats === "string") {
        const retreatIds = retreats.split(",").map(Number);
        formattedRetreats = retreatIds.map((id) => ({
          retreatId: id,
          disabled: false,
        }));
      }

      if (!userId || !Array.isArray(userId) || userId.length === 0) {
        return res.status(200).json({
          success: false,
          error: true,
          msg: "Please add at least one user.",
        });
      }

      let createdMembers = [];

      for (let u of userId) {
        const email = u.value;
        let existingUser = await userSchema.findOne({ where: { email } });

        // ✅ If user does not exist, create from coupleSchema
        if (!existingUser) {
          const couple = await coupleSchema.findOne({
            where: {
              [Sequelize.Op.or]: [{ hisEmail: email }, { herEmail: email }],
            },
          });

          if (!couple) {
            console.log(`No couple found for ${email}, skipping`);
            continue;
          }

          const isHis = couple.hisEmail === email;
          const firstName = isHis ? couple.hisFirstName : couple.herFirstName;
          const lastName = isHis ? couple.hisLastName : couple.herLastName;
          const mobile = isHis ? couple.hisMobile : couple.herMobile;
          const coupleId = couple.id;

          const randompass = Math.random().toString(36).slice(-8);
          const passwordForMember = "p" + randompass.slice(1);

          existingUser = await userSchema.create({
            userName: firstName,
            firstName,
            lastName,
            mobile,
            coupleId,
            email,
            password: passwordForMember,
            accountStatus: "ACTIVE",
            userRole: "COUPLE",
          });
        }

        // ✅ Check if committee entry already exists
        let committeeRecord = await committeeMemberSchema.findOne({
          where: { userId: existingUser.id, parishId },
        });

        if (committeeRecord) {
          // Merge retreats with existing
          let existingRetreats = [];
          try {
            existingRetreats = JSON.parse(committeeRecord.retreats) || [];
          } catch (e) {}

          const merged = [
            ...existingRetreats,
            ...formattedRetreats.filter(
              (r) =>
                !existingRetreats.some((er) => er.retreatId === r.retreatId)
            ),
          ];

          await committeeMemberSchema.update(
            { retreats: JSON.stringify(merged) },
            { where: { id: committeeRecord.id } }
          );

          createdMembers.push({
            ...committeeRecord.toJSON(),
            retreats: merged,
          });
        } else {
          // Create new committee member (steering role)
          const newMember = await committeeMemberSchema.create({
            userId: existingUser.id,
            parishId,
            retreats: JSON.stringify(formattedRetreats),
          });

          createdMembers.push(newMember);

          // Send login email (only on first creation)
          const replData = SENDLOGINDETAILTOSUBADMIN.replace(
            /#firstName#/g,
            existingUser.firstName || ""
          )
            .replace(/#lastName#/g, existingUser.lastName || "")
            .replace(/#role#/g, "steering committee member")
            .replace(/#url#/g, `${process.env.SERVERADDRESS}/login`)
            .replace(/#email#/g, existingUser.email)
            .replace(/#password#/g, existingUser.password);

          await Email.sendEmail(
            existingUser.email,
            "Login Details - LOVESTRONG Marriage",
            replData
          );
        }
      }

      return res.status(200).json({
        success: true,
        msg: "Steering members processed successfully.",
        data: createdMembers,
      });
    } catch (err) {
      console.error(`[newSteeringMember] error:`, err);
      return res.status(200).json({
        success: false,
        msg: "Server error",
        error: err.message,
      });
    }
  }

  async deleteParishMembers(req, res) {
    try {
      const { parishId } = req.body;
      if (!parishId) {
        return res.status(200).json({
          success: false,
          msg: "Missing parishId.",
        });
      }
      const deletedCount = await committeeMemberSchema.destroy({
        where: { parishId },
      });
      return res.status(200).json({
        success: true,
        msg: `Committee member(s) deleted successfully.`,
      });
    } catch (err) {
      return res.status(200).json({
        success: false,
        msg: "Server error",
        error: err.message,
      });
    }
  }

  async retreatDisabledForMem(req, res) {
    try {
      const { retreat, parishId } = req.body;

      if (!parishId || !retreat || !retreat.retreatId) {
        return res.status(200).json({
          success: false,
          msg: "Missing parishId or retreatId.",
        });
      }

      // Get all members in that parish
      const members = await committeeMemberSchema.findAll({
        where: { parishId },
      });

      if (!members || members.length === 0) {
        return res.status(200).json({
          success: false,
          msg: "No committee members found for this parish.",
        });
      }

      let statusChanged = null;

      // Loop through all members and update their retreat array
      for (const member of members) {
        let retreatData = [];
        try {
          retreatData = JSON.parse(member.retreats || "[]");
        } catch (e) {
          continue; // skip invalid format
        }

        const updatedRetreats = retreatData.map((r) => {
          if (r.retreatId === retreat.retreatId) {
            const newDisabled = !r.disabled;
            statusChanged = newDisabled ? "disabled" : "enabled";
            return { ...r, disabled: newDisabled };
          }
          return r;
        });

        await committeeMemberSchema.update(
          { retreats: JSON.stringify(updatedRetreats) },
          { where: { id: member.id } }
        );
      }

      return res.status(200).json({
        success: true,
        msg: `Retreat has been successfully ${statusChanged}.`,
      });
    } catch (err) {
      return res.status(200).json({
        success: false,
        msg: "Server error",
        error: err.message,
      });
    }
  }

  async committeeAccessedRetreats(req, res) {
    try {
      let { committeeId } = req.body;
      const committee = await committeeMemberSchema.findOne({
        where: { userId: committeeId },
      });
      if (!committee || !committee.retreats) {
        return res
          .status(200)
          .json({ success: false, msg: "Committee not found or no retreats." });
      }
      const accessedRetreats = JSON.parse(committee.retreats).filter(
        (r) => !r.disabled
      );
      const retreatIds = accessedRetreats.map((r) => r.retreatId);
      if (!retreatIds.length) {
        return res
          .status(200)
          .send({ success: true, msg: "No accessible retreats", data: [] });
      }
      const query = `
        SELECT 
          retreats.*, 
          programs.status AS programStatus, 
          directories.status AS directoryStatus 
        FROM retreats
        LEFT JOIN programs ON programs.retreatId = retreats.id
        LEFT JOIN directories ON directories.retreatId = retreats.id
        WHERE retreats.id IN (:retreatIds)
          AND (retreats.deleteStatus='' OR retreats.deleteStatus IS NULL)
        ORDER BY dateFrom DESC;
      `;
      const allRetreats = await db.sequelize.query(query, {
        replacements: { retreatIds },
        type: db.Sequelize.QueryTypes.SELECT,
      });
      const parsedData = allRetreats.map((item) => {
        const { confirmation, schedule, ...rest } = item;
        const parsedSchedule = schedule ? JSON.parse(schedule) : [];
        const parsedConfirmations = confirmation
          ? JSON.parse(confirmation)
          : [];
        return {
          ...rest,
          confirmation: parsedConfirmations,
          schedule: parsedSchedule.map((entry) => ({
            date: new Date(entry.date),
            from: entry.from,
            to: entry.to,
          })),
        };
      });
      return res.status(200).send({
        success: true,
        msg: "All Incoming Retreats",
        data: parsedData,
      });
    } catch (error) {
      return res.status(200).json({
        success: false,
        msg: "Error fetching new retreats",
        error: error.message,
      });
    }
  }
}

module.exports = new CommitteeController();
